from gurobipy import Model, GRB

def aircraft_landing(n, E, T, L, alpha, beta, S):
    """
    Args:
        n      : int, number of aircraft
        E      : list of length n, earliest landing times
        T      : list of length n, target landing times
        L      : list of length n, latest landing times
        alpha  : list of length n, penalty per hour of earliness
        beta   : list of length n, penalty per hour of lateness
        S      : 2D list [n x n], separation times S[i][j]
    Returns:
        float: minimum total penalty
    """
    # Create model
    m = Model("AircraftLanding")

    # Decision variables
    t = {}   # landing times
    e = {}   # earliness amounts
    ℓ = {}   # lateness amounts
    for i in range(n):
        t[i] = m.addVar(lb=E[i], ub=L[i], name=f"t_{i}")
        e[i] = m.addVar(lb=0.0, name=f"e_{i}")
        ℓ[i] = m.addVar(lb=0.0, name=f"l_{i}")

    # Objective: minimize total earliness & lateness penalties
    m.setObjective(
        sum(alpha[i] * e[i] + beta[i] * ℓ[i] for i in range(n)),
        GRB.MINIMIZE
    )

    # Time‐balance constraints: t_i - T_i = ℓ_i - e_i
    for i in range(n):
        m.addConstr(
            t[i] - T[i] == ℓ[i] - e[i],
            name=f"time_balance_{i}"
        )

    # Order & separation constraints
    for i in range(n - 1):
        # maintain landing order
        m.addConstr(
            t[i] <= t[i+1],
            name=f"order_{i}_{i+1}"
        )

        # ensure separation ≥ max( S[i][i+1], ℓ_i )
        # introduce auxiliary var for the max
        z = m.addVar(lb=0.0, name=f"sep_max_{i}_{i+1}")
        m.addGenConstrMax(
            z,
            [ S[i][i+1], ℓ[i] ],
            name=f"max_sep_{i}_{i+1}"
        )
        m.addConstr(
            t[i+1] - t[i] >= z,
            name=f"sep_constr_{i}_{i+1}"
        )

    # Optimize
    m.optimize()

    # Extract ALL solution variables
    t_sol = {i: t[i].X for i in range(n)}
    e_sol = {i: e[i].X for i in range(n)}
    l_sol = {i: ℓ[i].X for i in range(n)}
    
    # Extract separation variables
    z_sol = {}
    for i in range(n - 1):
        z_sol[f"sep_max_{i}_{i+1}"] = m.getVarByName(f"sep_max_{i}_{i+1}").X
    
    all_vars = {
        "landing_times": t_sol,
        "earliness": e_sol,
        "lateness": l_sol,
        "separation": z_sol
    }
    return m.objVal, all_vars