from gurobipy import Model, GRB, quicksum

def blend2(J, I,  p, c, d, lam):
    """
    Args:
        alloys:   list of alloy names (length J)
        elements: list of element names (length I)
        p:        list of length J, price per unit of each alloy
        c:        2D list [I x J], fraction of element i in alloy j
        d:        list of length I, target fraction of element i in blend
        lam:      list of length I, penalty weight for deviation of element i

    Returns:
        (objVal, x_dict): minimum total cost + weighted absolute deviations,
                         and dict of optimal x[j] for each alloy j
    """
    # Create a new model
    m = Model()

    # Decision variables: amount of each alloy to purchase
    x = {}
    for j in range(J):
        x[j] = m.addVar(lb=0.0, name=f"x_{j}")

    # Expressions and vars for deviations |sum_j c[i][j] x_j - d[i]|
    y = {}      # signed deviation
    aabs = {}   # absolute deviation
    for i in range(I):
        y[i] = m.addVar(lb=-GRB.INFINITY, ub=GRB.INFINITY, name=f"y_{i}")
        aabs[i] = m.addVar(lb=0.0, name=f"a_{i}")
        # y_e == sum_j c[i][j] * x_j - d[i]
        m.addConstr(
            y[i] == quicksum(c[i][j] * x[j] for j in range(J))
                    - d[i], name=f"ElemBal_{i}"
        )
        # a_e == | y_e |
        m.addGenConstrAbs(aabs[i], y[i], name=f"AbsDev_{i}")

    # Blend‐sum constraint: sum_j x_j == 1
    m.addConstr(
        quicksum(x[j] for j in range(J)) == 1,
        name="TotalBlend"
    )

    # Allow nonconvex features (absolute deviations in obj)
    m.params.NonConvex = 2

    # Objective: minimize purchase cost + weighted absolute deviations
    cost_expr = quicksum(p[j] * x[j] for j in range(J))
    penalty_expr = sum(lam[i] * aabs[i] for i in range(I))
    m.setObjective(cost_expr + penalty_expr, GRB.MINIMIZE)

    # Optimize
    m.optimize()

    # Extract ALL solution variables
    x_sol = {j: x[j].X for j in range(J)}
    y_sol = {i: y[i].X for i in range(I)}
    aabs_sol = {i: aabs[i].X for i in range(I)}
    
    all_vars = {
        "alloy_amounts": x_sol,
        "signed_deviations": y_sol,
        "absolute_deviations": aabs_sol
    }
    return m.ObjVal, all_vars