from gurobipy import Model, GRB, quicksum

def blend_min_cost_per_heat(unit_costs, heat_contents,
                            min_heat_kcal=7200, total_mass_kg=1000):
    """
    Minimise (Σ c_i x_i)/(Σ h_i x_i)
    s.t. Σ h_i x_i ≥ min_heat_kcal,
         Σ x_i      = total_mass_kg,
         x_i ≥ 0.
    """
    n = len(unit_costs)
    m = Model(); m.setParam("OutputFlag", 0)

    # CC variables
    y = m.addVars(n, lb=0, name="y")      # y_i = x_i · t
    t = m.addVar(lb=0, name="t")          # t  = 1/(Σ h_i x_i)

    # objective  min Σ c_i y_i
    m.setObjective(quicksum(unit_costs[i]*y[i] for i in range(n)), GRB.MINIMIZE)

    # 1. heat normalisation   Σ h_i y_i = 1
    m.addConstr(quicksum(heat_contents[i]*y[i] for i in range(n)) == 1)

    # 2. total mass           Σ y_i = total_mass · t
    m.addConstr(quicksum(y[i] for i in range(n)) == total_mass_kg * t)

    # 3. minimum heat (original scale)  t ≤ 1/min_heat
    m.addConstr(t <= 1.0 / min_heat_kcal)

    m.optimize()
    if m.Status != GRB.OPTIMAL:
        raise RuntimeError("Solver did not find an optimal solution.")

    # Extract ALL solution variables
    y_sol = {i: y[i].X for i in range(n)}
    t_sol = t.X
    
    # recover original mix in kg
    x_opt = [y[i].X / t.X for i in range(n)]                    # $ per kcal

    all_vars = {
        "cc_variables": y_sol,
        "ratio_variable": t_sol,
        "original_mix": x_opt
    }
    return x_opt, m.ObjVal, all_vars