from gurobipy import Model, GRB, quicksum


def diet_optimization(p, a, m_lower, M, x_lb, x_ub, B):       
    """
    Minimise | total cost – B | subject to nutrient bands
              and food-quantity bounds.

    Returns: (abs_cost_gap , optimal_food_quantities)
    """
    J = len(p)
    I = len(a)

    # ----------------------- 1) Build model -------------------------------
    m = Model()

    # Decision variables: quantities of each food
    x = m.addVars(J, lb=x_lb if isinstance(x_lb, list) else [x_lb]*J, ub=x_ub if isinstance(x_ub, list) else [x_ub]*J, name="x")

    # ----------------------- 2) Objective | Σ p_j x_j – B | --------------
    cost         = quicksum(p[j] * x[j] for j in range(J))
    diff_raw     = m.addVar(lb=-GRB.INFINITY, name="cost_minus_B")  # Σp_j x_j − B
    m.addConstr(diff_raw == cost - B, name="def_diff")

    abs_diff     = m.addVar(lb=0.0, name="abs_cost_gap")
    m.addGenConstrAbs(abs_diff, diff_raw, name="abs_link")           # |·|  :contentReference[oaicite:0]{index=0}

    m.setObjective(abs_diff, GRB.MINIMIZE)

    # ----------------------- 3) Nutrient bookkeeping ----------------------
    # Helper vars: y_i = Σ a_ij x_j         (actual intake of nutrient i)
    y = m.addVars(I, name="nutrient_intake")
    for i in range(I):
        m.addConstr(
            y[i] == quicksum(a[i][j] * x[j] for j in range(J)),
            name=f"intake_{i}"
        )

    # Intake of *reference* nutrient 1 (index 0)
    y1 = y[0]

    # ----------------------- 4) Nutrient bounds ---------------------------
    for i in range(I):
        # --- upper bound ---
        m.addConstr(y[i] <= M[i], name=f"upper_{i}")

        # --- lower bound with   y_i ≥ min(m_i , y_1) ---
        const_m  = m.addVar(lb=m_lower[i], ub=m_lower[i], name=f"m_const_{i}")  # fixed var
        min_var  = m.addVar(name=f"min_{i}")

        m.addGenConstrMin(min_var, [const_m, y1], name=f"min_link_{i}")         # min() :contentReference[oaicite:1]{index=1}
        m.addConstr(y[i] >= min_var, name=f"lower_{i}")

    # ----------------------- 5) Solve & report ----------------------------
    m.optimize()
    if m.status != GRB.OPTIMAL:
        raise RuntimeError("No optimal solution found.")

    # Extract ALL solution variables
    x_sol = {j: x[j].X for j in range(J)}
    diff_raw_sol = m.getVarByName("cost_minus_B").X
    abs_diff_sol = m.getVarByName("abs_cost_gap").X
    y_sol = {i: y[i].X for i in range(I)}
    min_sol = {i: m.getVarByName(f"min_{i}").X for i in range(I)}
    
    all_vars = {
        "food_quantities": x_sol,
        "cost_difference": diff_raw_sol,
        "absolute_cost_gap": abs_diff_sol,
        "nutrient_intake": y_sol,
        "min_values": min_sol
    }
    return m.objVal, all_vars
