from gurobipy import Model, GRB, quicksum
import math

def diet_log_cost(p, a, m, M, underline_x, overline_x):
    """
    Diet problem with logarithmic cost objective
    
    min log(Σ_j p_j * x_j)
    s.t. underline_x_j ≤ x_j ≤ overline_x_j     ∀j
         m_i ≤ Σ_j a_ij * x_j ≤ M_i            ∀i
         x_j ≥ 0                               ∀j
    
    Returns: (log_cost, food_quantities, all_vars)
    """
    J = len(p)
    I = len(a)
    
    # Create model
    m_model = Model("diet_log_cost")
    m_model.Params.OutputFlag = 0  # Silence solver output
    
    # Decision variables: quantities of each food
    x = m_model.addVars(J, lb=underline_x, ub=overline_x, name="x")
    
    # Total cost expression
    total_cost = quicksum(p[j] * x[j] for j in range(J))
    
    # Add a small constant to avoid log(0) issues
    epsilon = 1e-6
    cost_with_epsilon = total_cost + epsilon
    
    # Since Gurobi doesn't directly support log() in objectives,
    # we'll use a piecewise linear approximation or transform
    # For simplicity, we'll minimize the total cost directly
    # and then compute log(cost) in the return statement
    
    # Set objective to minimize total cost
    m_model.setObjective(total_cost, GRB.MINIMIZE)
    
    # Nutrient constraints
    for i in range(I):
        nutrient_intake = quicksum(a[i][j] * x[j] for j in range(J))
        m_model.addConstr(nutrient_intake >= m[i], name=f"nutrient_{i}_lower")
        m_model.addConstr(nutrient_intake <= M[i], name=f"nutrient_{i}_upper")
    
    # Solve the model
    m_model.optimize()
    
    if m_model.Status != GRB.OPTIMAL:
        raise RuntimeError("Model did not solve to optimality.")
    
    # Extract solution
    x_sol = {j: x[j].X for j in range(J)}
    total_cost_val = total_cost.getValue()
    log_cost_val = math.log(total_cost_val + epsilon)
    
    # Extract all variables for comprehensive output
    all_vars = {
        "food_quantities": x_sol,
        "total_cost": total_cost_val,
        "log_cost": log_cost_val,
        "nutrient_intake": {
            i: quicksum(a[i][j] * x[j] for j in range(J)).getValue() 
            for i in range(I)
        }
    }
    
    return log_cost_val, x_sol, all_vars 