from gurobipy import Model, GRB, quicksum

def diet(nutrients, foods, a, p, x_min, x_max, M_req):
    """
    Simplified nonlinear model

        min   Σ_j p_j x_j  /  Σ_j x_j
        s.t.  Σ_j a_ij x_j ≤ M_i                              (i = 1,...,I)
              x_min_j ≤ x_j ≤ x_max_j
              Σ_j x_j ≥ Q_min                                 (minimum total quantity)

    All variables are continuous; Gurobi is run with NonConvex=2.
    """
    I, J = len(nutrients), len(foods)

    m = Model("fractional_min_diet")
    m.Params.NonConvex = 2          # allow bilinear & min constraints
    m.Params.OutputFlag = 0         # silence solver output (optional)

    # ------------------------------------------------------------------
    # 1. decision variables x_j
    # ------------------------------------------------------------------
    x = m.addVars(J,
                  lb=x_min,
                  ub=x_max,
                  name="x")

    # ------------------------------------------------------------------
    # 2. numerator, denominator, and q = num/denom
    # ------------------------------------------------------------------
    num   = quicksum(p[j] * x[j] for j in range(J))
    denom = quicksum(x[j]       for j in range(J))

    min_total_quantity = 1  # Require at least 1 unit total
    m.addConstr(denom >= min_total_quantity, name="min_total_quantity")

    q = m.addVar(lb=0, name="q")                      # bound helps numerics
    m.addConstr(num == q * denom, name="fraction_def")

    # ------------------------------------------------------------------
    # 3. nutrient constraints (only upper bounds)
    # ------------------------------------------------------------------
    for i in range(I):
        total_i = quicksum(a[i][j] * x[j] for j in range(J))
        m.addConstr(total_i <= M_req[i], name=f"nutrient_{i}_upper")


    # ------------------------------------------------------------------
    # 4. objective
    # ------------------------------------------------------------------
    m.setObjective(q, GRB.MINIMIZE)

    m.optimize()
    if m.Status != GRB.OPTIMAL:
        raise RuntimeError("Model did not solve to optimality.")

    # Extract ALL solution variables
    x_vals = [x[j].X for j in range(J)]
    q_sol = q.X
    
    all_vars = {
        "food_quantities": {j: x[j].X for j in range(J)},
        "cost_ratio": q_sol
    }
    return q.X, x_vals, all_vars