from gurobipy import Model, GRB, quicksum

def knapsack_optimization(item_values, item_weights, lb_x, ub_x, max_weight_knapsack):
    """
    Args:
        item_values: a list of integers, indicating the value of each item
        item_weights: a list of integers, indicating the weight of each item
        lb_x: a list of integers (0 or 1), lower bounds for each x_j
        ub_x: a list of integers (0 or 1), upper bounds for each x_j
        max_weight_knapsack: an integer, the maximum weight capacity W

    Returns:
        max_total_value: a float, the maximum total value of the items in the knapsack
    """
    # Create a new Gurobi model
    m = Model()

    # Number of items
    n = len(item_values)

    # Decision variables: x[j] ∈ {0,1} with explicit bounds
    x = m.addVars(n,
                  vtype=GRB.BINARY,
                  lb=lb_x,
                  ub=ub_x,
                  name="x")

    # Auxiliary variable for min(x₁, x₃)
    min_x1_x3 = m.addVar(vtype=GRB.CONTINUOUS, name="min_x1_x3")

    # Objective: maximize total value
    m.setObjective(
        quicksum(item_values[j] * x[j] for j in range(n)),
        GRB.MAXIMIZE
    )

    # Generalized constraint: min(x₁, x₃)
    # (using Gurobi's built‐in min‐constraint, even though it's non‐linear)
    m.addGenConstrMin(min_x1_x3, [x[0], x[2]], name="min_x1_x3_constr")

    # Weight (volume) constraint with reduction term
    m.addConstr(
        quicksum(item_weights[j] * x[j] for j in range(n))
        - 10 * min_x1_x3
        <= max_weight_knapsack,
        name="weight_constraint"
    )

    # Optimize the model
    m.optimize()

    # Extract ALL solution variables
    x_sol = {j: x[j].X for j in range(n)}
    min_x1_x3_sol = m.getVarByName("min_x1_x3").X
    
    all_vars = {
        "item_selections": x_sol,
        "min_x1_x3": min_x1_x3_sol
    }
    return m.objVal, all_vars