# knapsack_with_synergy.py

from gurobipy import Model, GRB, quicksum

def knapsack_with_synergy(n, v, w, W, b):
    """
    Solves the Knapsack Problem with an extra synergy term b * min(x1, x2).

    Args:
        n: number of items
        v: list of item values (length n)
        w: list of item weights (length n)
        x_lb: list of lower bounds for x_j (length n)
        x_ub: list of upper bounds for x_j (length n)
        W: knapsack capacity
        b: synergy parameter
        
    Returns:
        float: optimal objective value
    """
    # Create a new Gurobi model
    m = Model()

    # Decision variables: x[j] in [x_lb[j], x_ub[j]], binary
    x = m.addVars(
        n,
        vtype=GRB.BINARY,
        name="x"
    )

    # Create auxiliary variable for min(x[0], x[1])
    min_x0_x1 = m.addVar(vtype=GRB.CONTINUOUS, name="min_x0_x1")
    m.addGenConstrMin(min_x0_x1, [x[0], x[1]], name="min_constraint")

    # Objective: maximize sum_j v[j]*x[j] + b * min(x[0], x[1])
    m.setObjective(
        quicksum(v[j] * x[j] for j in range(n))
        + b * min_x0_x1,
        GRB.MAXIMIZE
    )

    # Weight constraint
    m.addConstr(
        quicksum(w[j] * x[j] for j in range(n)) <= W,
        name="weight_constraint"
    )

    # Optimize
    m.optimize()
    
    if m.Status != GRB.OPTIMAL:
        raise RuntimeError("Model did not solve to optimality.")

    # Extract ALL solution variables
    x_sol = {j: x[j].X for j in range(n)}
    min_x0_x1_sol = m.getVarByName("min_x0_x1").X
    
    all_vars = {
        "item_selections": x_sol,
        "min_x0_x1": min_x0_x1_sol
    }
    return m.objVal, all_vars