import gurobipy as gp
from gurobipy import GRB

def media_selection(c, d, a, m):
    """
    Media Selection Problem with Synergy
    
    Args:
        c: List of media costs c_m
        d: Synergy discount parameter
        a: 2D list where a[t][m] = 1 if media m covers audience t
        m: List of coverage requirements m_t for each audience t
    Returns:
        float: Minimum total cost (including the negative synergy term)
    """
    # Initialize the model
    model = gp.Model("Media Selection with Synergy")
    model.setParam('OutputFlag', 0)  # Suppress Gurobi output

    num_media = len(c)
    num_audiences = len(m)

    # Decision variables: x[m] = 1 if media m is selected
    x = model.addVars(num_media, vtype=GRB.BINARY, name="x")

    # Objective: minimize sum(c_m * x_m) - d * x_1 * x_2
    model.setObjective(
        gp.quicksum(c[m] * x[m] for m in range(num_media))
        - d * x[0] * x[1],
        GRB.MINIMIZE
    )

    # Coverage constraints: each audience t covered at least m_t[t] times
    for t in range(num_audiences):
        model.addConstr(
            gp.quicksum(a[t][m] * x[m] for m in range(num_media))
            >= m[t],
            name=f"CoverAudience_{t}"
        )

    # Solve
    model.optimize()
    
    if model.Status != GRB.OPTIMAL:
        raise RuntimeError("Model did not solve to optimality.")

    # Extract ALL solution variables
    x_sol = {m: x[m].X for m in range(num_media)}
    
    all_vars = {
        "media_selections": x_sol
    }
    return model.objVal, all_vars