import gurobipy as gp
from gurobipy import GRB

def media_selection(c, r, a, m, budget):
    """
    Media Selection Problem with Binary-Continuous Variables
    
    Args:
        c: List of media costs c_m
        r: List of revenue parameters r_m
        a: 2D list where a[t][m] = 1 if media m covers audience t
        m: List of coverage requirements m_t for each audience t
        budget: Total budget for quality increases
    Returns:
        float: Optimal objective value
    """
    # Initialize model
    model = gp.Model("MediaSelection")
    model.setParam('OutputFlag', 0)  # Suppress Gurobi output
    model.Params.NonConvex = 2

    num_media = len(c)
    num_audiences = len(m)

    # Decision variables
    x = model.addVars(num_media, vtype=GRB.BINARY, name="x")   # select media?
    q = model.addVars(num_media, lb=0.0, name="q")             # quality dollars on each media

    # Constraints: coverage
    for t in range(num_audiences):
        model.addConstr(
            gp.quicksum(a[t][m] * x[m] for m in range(num_media))
            >= m[t],
            name=f"CoverAudience_{t}"
        )

    # Constraint: total quality budget
    model.addConstr(
        gp.quicksum(q[m] for m in range(num_media)) <= budget,
        name="QualityBudget"
    )

    # Objective: minimize cost minus revenue from quality
    model.setObjective(
        gp.quicksum(
            c[m] * x[m]
            - r[m] * x[m] * q[m]
            for m in range(num_media)
        ),
        GRB.MINIMIZE
    )

    # Solve
    model.optimize()
    
    if model.Status != GRB.OPTIMAL:
        raise RuntimeError("Model did not solve to optimality.")

    # Extract ALL solution variables
    x_sol = {m: x[m].X for m in range(num_media)}
    
    all_vars = {
        "media_selections": x_sol
    }
    return model.objVal, all_vars