from gurobipy import Model, GRB, quicksum

def multi(Origins, Destinations, Products, Supply, Demand, Limit, Cost, r):
    """
    Args:
        Origins: number of origins
        Destinations: number of destinations
        Products: number of products
        Supply: a 2D list, Supply[i][p] indicates the amount of product p available at origin i
        Demand: a 2D list, Demand[p][j] indicates the amount of product p required at destination j
        Limit: a 2D list, Limit[i][j] indicates the maximum total amount of all products
               that can be shipped from origin i to destination j
        Cost: a 3D list, Cost[i][j][p] indicates the shipment cost per unit of product p
              from origin i to destination j
        r: a scalar, penalty cost per unit deviation from demand
    Returns:
        total_cost: a float, the minimized total cost of shipping all products
                    including penalty for demand deviations
    """
    # Index sets
    origins = range(Origins)
    products = range(Products)
    destinations = range(Destinations)

    # Create model
    model = Model("MultiCommodityTransportationWithDeviation")

    # Decision variables: x[i,j,p] ≥ 0
    x = {}
    for i in origins:
        for j in destinations:
            for p in products:
                x[i, j, p] = model.addVar(lb=0, name=f"x_{i}_{j}_{p}")

    # Variables for absolute deviations
    dev_vars = {}
    for j in destinations:
        for p in products:
            dev_vars[j, p] = model.addVar(lb=0, name=f"dev_{j}_{p}")

    # Variables for differences
    diff_vars = {}
    for j in destinations:
        for p in products:
            diff_vars[j, p] = model.addVar(lb=-float('inf'), ub=float('inf'), name=f"diff_{j}_{p}")

    # Integrate variables
    model.update()

    # Define differences: actual - demand
    for j in destinations:
        for p in products:
            actual = quicksum(x[i, j, p] for i in origins)
            model.addConstr(diff_vars[j, p] == actual - Demand[p][j], name=f"diff_def_{j}_{p}")

    # Define absolute deviations
    for j in destinations:
        for p in products:
            model.addGenConstrAbs(dev_vars[j, p], diff_vars[j, p], name=f"abs_dev_{j}_{p}")

    # Objective: shipping cost + penalty for unmet/excess demand
    model.setObjective(
        quicksum(Cost[i][j][p] * x[i, j, p]
                 for i in origins for j in destinations for p in products)
        + r * quicksum(dev_vars[j, p] for j in destinations for p in products),
        GRB.MINIMIZE
    )

    # Supply constraints: ship exactly the available supply
    for i in origins:
        for p in products:
            model.addConstr(
                quicksum(x[i, j, p] for j in destinations) == Supply[i][p],
                name=f"Supply_{i}_{p}"
            )

    # Capacity limits: total across products ≤ limit
    for i in origins:
        for j in destinations:
            model.addConstr(
                quicksum(x[i, j, p] for p in products) <= Limit[i][j],
                name=f"Limit_{i}_{j}"
            )

    # Solve
    model.optimize()

    # Return objective value
    if model.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(i, j, p): x[i, j, p].X for i in origins for j in destinations for p in products}
        dev_sol = {(j, p): dev_vars[j, p].X for j in destinations for p in products}
        diff_sol = {(j, p): diff_vars[j, p].X for j in destinations for p in products}
        
        all_vars = {
            "shipments": x_sol,
            "deviations": dev_sol,
            "differences": diff_sol
        }
        return model.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")