def netasgn(I, J, S, D, c, ell, r):
    """
    Args:
        I: number of people
        J: number of projects
        S: list of length I, hours each person is available
        D: list of length J, hours each project requires
        c: 2D list of size I x J, cost per hour
        ell: 2D list of size I x J, max hours per person/project
        r: float, weight on fairness term (max pairwise deviation)
    Returns:
        total_cost: float, minimized total cost including fairness penalty
    """
    import gurobipy as gp
    from gurobipy import GRB

    # Indices
    num_people = I
    num_projects = J
    people = range(num_people)
    projects = range(num_projects)

    # Model
    model = gp.Model('net_assignment_fair')

    # Decision variables x_{i,j}
    x = {}
    for i in people:
        for j in projects:
            x[i, j] = model.addVar(
                lb=0,
                ub=ell[i][j],
                obj=c[i][j],
                vtype=GRB.CONTINUOUS,
                name=f"x_{i}_{j}"
            )

    # Fairness: absolute deviations for each unordered pair (i<k)
    dev_pair = {}
    diff_vars = {}  # Variables to store the differences
    for i in people:
        for k in people:
            if k > i:
                # Create variable for the difference
                diff_vars[i, k] = model.addVar(
                    lb=-float('inf'),
                    ub=float('inf'),
                    vtype=GRB.CONTINUOUS,
                    name=f"diff_{i}_{k}"
                )
                
                dev_pair[i, k] = model.addVar(
                    lb=0,
                    vtype=GRB.CONTINUOUS,
                    name=f"dev_{i}_{k}"
                )

    # Max of those deviations
    dev = model.addVar(
        lb=0,
        obj=r,
        vtype=GRB.CONTINUOUS,
        name="dev"
    )

    # Integrate variables
    model.update()

    # Constraints: define the differences
    for (i, k) in diff_vars.keys():
        diff_expr = gp.quicksum(x[i, j] for j in projects) - gp.quicksum(x[k, j] for j in projects)
        model.addConstr(diff_vars[i, k] == diff_expr, name=f"diff_def_{i}_{k}")

    # Gen‐constraint: absolute deviation for each pair
    for (i, k), dv in dev_pair.items():
        model.addGenConstrAbs(dv, diff_vars[i, k], name=f"abs_dev_{i}_{k}")

    # Gen‐constraint: maximum deviation across all pairs
    if len(dev_pair) > 0:
        model.addGenConstrMax(dev, list(dev_pair.values()), name="max_dev")
    else:
        # If no pairs, set dev to 0
        model.addConstr(dev == 0, name="no_deviation")

    # Supply constraints: each person uses exactly their available hours
    for i in people:
        model.addConstr(
            gp.quicksum(x[i, j] for j in projects) == S[i],
            name=f"Supply_{i}"
        )

    # Demand constraints: each project receives exactly its required hours
    for j in projects:
        model.addConstr(
            gp.quicksum(x[i, j] for i in people) == D[j],
            name=f"Demand_{j}"
        )

    # Solve
    model.optimize()

    # Return objective value
    if model.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(i, j): x[i, j].X for i in people for j in projects}
        dev_pair_sol = {(i, k): dev_pair[i, k].X for i in people for k in people if k > i}
        diff_sol = {(i, k): diff_vars[i, k].X for i in people for k in people if k > i}
        dev_sol = model.getVarByName("dev").X
        
        all_vars = {
            "assignments": x_sol,
            "deviation_pairs": dev_pair_sol,
            "differences": diff_sol,
            "max_deviation": dev_sol
        }
        return model.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")