def netasgn_fair(I, J, S, D, c, ell, r):
    """
    Args:
        I: number of people
        J: number of projects
        S: list of length I, hours each person is available
        D: list of length J, hours each project requires
        c: 2D list of size I x J, cost per hour
        ell: 2D list of size I x J, max hours per person/project
        r: float, fairness multiplier for maximum deviation of total work between any two people

    Returns:
        total_cost: a float, denotes the minimized total cost including fairness penalty
    """
    import gurobipy as gp
    from gurobipy import GRB

    # Create indices for people and projects
    num_people = I
    num_projects = J
    people = range(num_people)
    projects = range(num_projects)

    # Create the model
    model = gp.Model('net_assignment_fair')

    # Define variables x_{i,j} with bounds [0, ell[i][j]] and cost coefficients
    x = {}
    for i in people:
        for j in projects:
            x[i, j] = model.addVar(
                lb=0,
                ub=ell[i][j],
                obj=c[i][j],
                vtype=GRB.CONTINUOUS,
                name=f"x_{i}_{j}"
            )

    # Fairness: maximum deviation of total work between any two people
    total_hours_vars = {}  # Variables to store total hours per person
    for i in people:
        total_hours_vars[i] = model.addVar(
            lb=0,
            vtype=GRB.CONTINUOUS,
            name=f"total_hours_{i}"
        )

    # Maximum deviation variable
    max_deviation = model.addVar(
        lb=0,
        obj=r,
        vtype=GRB.CONTINUOUS,
        name="max_deviation"
    )

    # Integrate new variables
    model.update()

    # Constraints: define total hours for each person
    for i in people:
        model.addConstr(
            total_hours_vars[i] == gp.quicksum(x[i, j] for j in projects),
            name=f"total_hours_def_{i}"
        )

    # Fairness constraints: maximum deviation between any two people
    deviation_vars = []
    for i in people:
        for k in people:
            if k > i:
                # Create variable for absolute deviation between person i and k
                dev_var = model.addVar(
                    lb=0,
                    vtype=GRB.CONTINUOUS,
                    name=f"dev_{i}_{k}"
                )
                deviation_vars.append(dev_var)
                
                # Create variable for the difference
                diff_var = model.addVar(
                    lb=-float('inf'),
                    ub=float('inf'),
                    vtype=GRB.CONTINUOUS,
                    name=f"diff_{i}_{k}"
                )
                
                # Define the difference
                model.addConstr(
                    diff_var == total_hours_vars[i] - total_hours_vars[k],
                    name=f"diff_def_{i}_{k}"
                )
                
                # Define the absolute deviation
                model.addGenConstrAbs(dev_var, diff_var, name=f"abs_dev_{i}_{k}")

    # Maximum deviation across all pairs
    if deviation_vars:
        model.addGenConstrMax(max_deviation, deviation_vars, name="max_deviation_constraint")

    # Supply constraints: each person i works exactly S[i] hours
    for i in people:
        model.addConstr(
            gp.quicksum(x[i, j] for j in projects) == S[i],
            name=f"Supply_{i}"
        )

    # Demand constraints: each project j receives exactly D[j] hours
    for j in projects:
        model.addConstr(
            gp.quicksum(x[i, j] for i in people) == D[j],
            name=f"Demand_{j}"
        )

    # Set the objective: assignment cost + fairness penalty
    model.setObjective(
        gp.quicksum(c[i][j] * x[i, j] for i in people for j in projects)
        + max_deviation,
        GRB.MINIMIZE
    )

    # Optimize the model
    model.optimize()

    # Retrieve the minimized total cost
    if model.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(i, j): x[i, j].X for i in people for j in projects}
        total_hours_sol = {i: total_hours_vars[i].X for i in people}
        max_deviation_sol = model.getVarByName("max_deviation").X
        
        # Extract deviation variables
        dev_sol = {}
        for i in people:
            for k in people:
                if k > i:
                    dev_sol[f"dev_{i}_{k}"] = model.getVarByName(f"dev_{i}_{k}").X
                    dev_sol[f"diff_{i}_{k}"] = model.getVarByName(f"diff_{i}_{k}").X
        
        all_vars = {
            "assignments": x_sol,
            "total_hours": total_hours_sol,
            "deviations": dev_sol,
            "max_deviation": max_deviation_sol
        }
        return model.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")