# netmcol_with_contract.py

from gurobipy import Model, GRB, quicksum

def netmcol(n, P, Supply, Demand, ShipmentCost, Capacity, JointCapacity, ContractCosts):
    """
    Args:
        n: number of cities
        P: number of products
        Supply: 2D list [city][product] - supply at each city for each product
        Demand: 2D list [city][product] - demand at each city for each product
        ShipmentCost: 3D list [city][city][product] - cost to ship product from city i to city j
        Capacity: 3D list [city][city][product] - capacity for each product on each link
        JointCapacity: 2D list [city][city] - joint capacity across all products on each link
        ContractCosts: 2D list [city][city] - fixed cost to establish contract on each link

    Returns:
        total_cost: float, minimized total cost (including contract fixed costs)
    """

    # Create model
    model = Model()

    # Create indices
    cities = range(n)
    products = range(P)
    links = [(i, j) for i in cities for j in cities if i != j]  # All possible links

    # Decision vars
    x = model.addVars(links, products,
                      lb=0,
                      name="x")
    z = model.addVars(links,
                      vtype=GRB.BINARY,
                      name="z")

    # Objective
    model.setObjective(
        quicksum(
            (1 - 0.2*z[i,j])
            * ShipmentCost[i][j][p]
            * x[i,j,p]
            + ContractCosts[i][j] * z[i,j]
            for (i,j) in links for p in products
        ),
        GRB.MINIMIZE
    )

    # Build adjacency for flow
    in_links  = {i: [] for i in cities}
    out_links = {i: [] for i in cities}
    for (i,j) in links:
        out_links[i].append(j)
        in_links[j].append(i)

    # Flow conservation
    for i in cities:
        for p in products:
            net_supply = Supply[i][p] - Demand[i][p]
            inflow  = quicksum(x[j,i,p] for j in in_links[i])
            outflow = quicksum(x[i,j,p] for j in out_links[i])
            model.addConstr(net_supply + inflow - outflow == 0,
                            name=f"flow_{i}_{p}")

    # Per‐product link capacities
    for (i,j) in links:
        for p in products:
            model.addConstr(
                x[i,j,p] <= Capacity[i][j][p],
                name=f"cap_{i}_{j}_{p}"
            )

    # Joint link capacities
    for (i,j) in links:
        model.addConstr(
            quicksum(x[i,j,p] for p in products)
            <= JointCapacity[i][j],
            name=f"jointcap_{i}_{j}"
        )

    # Optimize
    model.optimize()

    if model.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(i, j, p): x[i, j, p].X for (i, j) in links for p in products}
        z_sol = {(i, j): z[i, j].X for (i, j) in links}
        
        all_vars = {
            "shipments": x_sol,
            "contracts": z_sol
        }
        return model.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")