import json
from gurobipy import Model, GRB, quicksum

def netmcol(C, P, L, Supply, Demand, ShipmentCost, Capacity, JointCapacity, r, link_from, link_to):
    """
    Args:
        C: number of cities
        P: number of products
        L: number of links
        Supply: 2D list [city][product] - supply at each city for each product
        Demand: 2D list [city][product] - demand at each city for each product
        ShipmentCost: 2D list [link][product] - cost to ship product on each link
        Capacity: 2D list [link][product] - capacity for each product on each link
        JointCapacity: 1D list [link] - joint capacity across all products on each link
        r: 1D list [product] - revenue per unit for each product
        link_from: 1D list [link] - source city for each link (1-based indexing)
        link_to: 1D list [link] - destination city for each link (1-based indexing)
    Returns:
        float: maximized revenue / shipment-cost ratio
    """
    # Create model
    model = Model()
    model.Params.NonConvex = 2

    # Create indices
    cities = range(C)
    products = range(P)
    links = range(L)
    
    # Convert 1-based city indices to 0-based
    link_from_0 = [link_from[l] - 1 for l in links]
    link_to_0 = [link_to[l] - 1 for l in links]
    
    # Create link mapping
    link_pairs = [(link_from_0[l], link_to_0[l]) for l in links]

    # Decision vars
    x = model.addVars(links, products, lb=0, name="x")
    y = model.addVars(cities, products, lb=0, name="y")

    # Objective: (∑ r_p y_{i,p}) / (∑ cost_{l,p} x_{l,p})
    revenue = quicksum(
        r[p] * y[i, p]
        for i in cities for p in products
    )
    cost = quicksum(
        ShipmentCost[l][p] * x[l, p]
        for l in links for p in products
    )
    
    # Use auxiliary variables for fractional objective
    num = model.addVar(lb=0, name="numerator")
    denom = model.addVar(lb=1e-6, name="denominator")  # Small positive bound
    ratio = model.addVar(lb=0, name="ratio")
    
    # Define numerator and denominator
    model.addConstr(num == revenue, name="num_def")
    model.addConstr(denom == cost, name="denom_def")
    
    # Define ratio: num = ratio * denom
    model.addConstr(num == ratio * denom, name="ratio_def")
    
    model.setObjective(ratio, GRB.MAXIMIZE)

    # Build link‐lists for flow conservation
    in_links  = {i: [] for i in cities}
    out_links = {i: [] for i in cities}
    for l in links:
        i, j = link_from_0[l], link_to_0[l]
        out_links[i].append(l)
        in_links[j].append(l)

    # Flow conservation: inflow + Supply = outflow + y
    for i in cities:
        for p in products:
            inflow  = quicksum(x[l, p] for l in in_links[i])
            outflow = quicksum(x[l, p] for l in out_links[i])
            model.addConstr(
                inflow + Supply[i][p]
                == outflow + y[i, p],
                name=f"flow_{i}_{p}"
            )

    # Demand bounds
    for i in cities:
        for p in products:
            model.addConstr(
                y[i, p] <= Demand[i][p],
                name=f"demand_{i}_{p}"
            )

    # Per‐product capacity + joint capacity
    for l in links:
        for p in products:
            model.addConstr(
                x[l, p] <= Capacity[l][p],
                name=f"cap_{l}_{p}"
            )
        model.addConstr(
            quicksum(x[l, p] for p in products)
            <= JointCapacity[l],
            name=f"joint_cap_{l}"
        )

    # Solve
    model.optimize()

    if model.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(l, p): x[l, p].X for l in links for p in products}
        y_sol = {(i, p): y[i, p].X for i in cities for p in products}
        num_sol = model.getVarByName("numerator").X
        denom_sol = model.getVarByName("denominator").X
        ratio_sol = model.getVarByName("ratio").X
        
        all_vars = {
            "shipments": x_sol,
            "demand": y_sol,
            "numerator": num_sol,
            "denominator": denom_sol,
            "ratio": ratio_sol
        }
        return model.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")
    