def nltrans(I, J, S, D, r, r_tilde, ell, C):
    """
    Args:
        I: number of origins
        J: number of destinations
        S: a list of integers, each indicates the amount of goods available at an origin
        D: a list of integers, each indicates the amount of goods required at a destination
        r: a 2D list of integers, the shipment costs per unit from each origin to each destination
        r_tilde: a 2D list of integers, the reduced shipment costs per unit if investment is made
        ell: a 2D list of integers, the limit on units shipped from each origin to each destination
        C: a scalar integer, the fixed cost of making the investment

    Returns:
        total_cost: an integer, denotes the minimum total cost of shipping goods (including investment if made)
    """
    import gurobipy as gp
    from gurobipy import GRB

    num_origins = I
    num_destinations = J

    # Create model
    m = gp.Model()
    # Allow non-convex/quadratic constraints
    m.Params.NonConvex = 2

    # Create variables for units shipped from each origin to each destination
    x = m.addVars(num_origins, num_destinations, lb=0, name="x")
    # Binary investment decision
    z = m.addVar(vtype=GRB.BINARY, name="z")

    # Supply constraints: total units shipped from each origin cannot exceed its supply
    m.addConstrs(
        (gp.quicksum(x[i, j] for j in range(num_destinations)) <= S[i]
         for i in range(num_origins)),
        name="Supply"
    )

    # Demand constraints: total units received at each destination must meet its demand
    m.addConstrs(
        (gp.quicksum(x[i, j] for i in range(num_origins)) >= D[j]
         for j in range(num_destinations)),
        name="Demand"
    )

    # Shipping limit constraints: units shipped from origin i to destination j cannot exceed ell[i][j]
    m.addConstrs(
        (x[i, j] <= ell[i][j]
         for i in range(num_origins)
         for j in range(num_destinations)),
        name="Limit"
    )

    # Objective: minimize total cost
    #   (1-z)*sum(r * x) + z*sum(r_tilde * x) + z*C
    # = sum(r[i][j] * x[i,j]) + sum((r_tilde[i][j] - r[i][j]) * z * x[i,j]) + C*z
    total_cost_expr = (
        gp.quicksum(r[i][j] * x[i, j]
                    for i in range(num_origins)
                    for j in range(num_destinations))
        + gp.quicksum((r_tilde[i][j] - r[i][j]) * z * x[i, j]
                      for i in range(num_origins)
                      for j in range(num_destinations))
        + C * z
    )
    m.setObjective(total_cost_expr, GRB.MINIMIZE)

    # Optimize the model
    m.optimize()

    # Retrieve the total cost
    if m.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(i, j): x[i, j].X for i in range(num_origins) for j in range(num_destinations)}
        z_sol = z.X
        
        all_vars = {
            "shipments": x_sol,
            "investment": z_sol
        }
        total_cost = m.objVal
        return total_cost, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")