import gurobipy as gp
from gurobipy import GRB

def ntrans(I, J, S, D, r, ell, c):
    """
    Args:
        I: number of origins
        J: number of destinations
        S: list of supply amounts at each origin, length I
        D: list of demand amounts at each destination, length J
        r: 2D list of shipping costs r[i][j], shape I×J
        ell: 2D list of soft‐capacity limits ell[i][j], shape I×J
        c: penalty cost per unit above each limit

    Returns:
        total_cost: integer, the minimum total cost of shipping
    """
    num_origins = I
    num_destinations = J

    # Create model
    m = gp.Model()

    # Decision vars: x[i,j] = units shipped; y[i,j] = units in excess of limit
    x = m.addVars(num_origins, num_destinations, lb=0, name="x")
    y = m.addVars(num_origins, num_destinations, lb=0, name="y")

    # Variables for differences: diff[i,j] = x[i,j] - ell[i][j]
    diff = m.addVars(num_origins, num_destinations, lb=-float('inf'), ub=float('inf'), name="diff")

    # Supply constraints: sum_j x[i,j] <= S_i
    m.addConstrs(
        (gp.quicksum(x[i, j] for j in range(num_destinations)) <= S[i]
         for i in range(num_origins)),
        name="Supply"
    )

    # Demand constraints: sum_i x[i,j] >= D_j
    m.addConstrs(
        (gp.quicksum(x[i, j] for i in range(num_origins)) >= D[j]
         for j in range(num_destinations)),
        name="Demand"
    )

    # Define differences: diff[i,j] = x[i,j] - ell[i][j]
    for i in range(num_origins):
        for j in range(num_destinations):
            m.addConstr(diff[i, j] == x[i, j] - ell[i][j], name=f"diff_def_{i}_{j}")

    # Soft‐capacity penalty: y[i,j] = max{0, diff[i,j]}
    for i in range(num_origins):
        for j in range(num_destinations):
            m.addGenConstrMax(
                y[i, j],
                [diff[i, j], 0],
                name=f"Exceed_{i}_{j}"
            )

    # Objective: minimize sum of (r_ij * x_ij + c * y_ij)
    total_cost = gp.quicksum(
        r[i][j] * x[i, j] + c * y[i, j]
        for i in range(num_origins)
        for j in range(num_destinations)
    )
    m.setObjective(total_cost, GRB.MINIMIZE)

    # Solve
    m.optimize()

    # Return the optimal cost
    if m.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {(i, j): x[i, j].X for i in range(num_origins) for j in range(num_destinations)}
        y_sol = {(i, j): y[i, j].X for i in range(num_origins) for j in range(num_destinations)}
        diff_sol = {(i, j): diff[i, j].X for i in range(num_origins) for j in range(num_destinations)}
        
        all_vars = {
            "shipments": x_sol,
            "excess": y_sol,
            "differences": diff_sol
        }
        return m.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")
