def prod_with_campaign(n, a, c, u, c_tilde, b, C):
    """
    Args:
        n: number of products
        a: list of ints, parameter for each element in set P
        c: list of ints, base profit coefficient for each element in set P
        u: list of ints, upper limit for each element in set P
        c_tilde: list of ints, increased profit coefficient for each element in set P
        b: int or float, global resource limit
        C: int or float, cost of media campaign

    Returns:
        total_profit: int, maximum total profit achieved
    """
    # Import gurobipy
    from gurobipy import Model, GRB

    # Check that a, c, u, and c_tilde have the same length
    if not (len(a) == len(c) == len(u) == len(c_tilde) == n):
        raise ValueError("Input lists a, c, u, and c_tilde must have length n")

    # Check that a[j] != 0 and u[j] >= 0
    for idx in range(n):
        if a[idx] == 0:
            raise ValueError(f"Parameter a[{idx}] is zero, division by zero not allowed.")
        if u[idx] < 0:
            raise ValueError(f"Upper limit u[{idx}] cannot be negative.")

    # Create the model
    m = Model("profit_maximization_with_campaign")
    m.setParam('OutputFlag', 0)  # Suppress Gurobi output

    # Number of elements in P
    P = range(n)

    # Add continuous variables x_j
    X = {}
    for j in P:
        X[j] = m.addVar(lb=0, ub=u[j], name=f"x_{j}")

    # Add binary variable z for campaign decision
    z = m.addVar(vtype=GRB.BINARY, name="z")

    # Build the two profit expressions
    expr_base     = sum(c[j]        * X[j] for j in P)
    expr_campaign = sum(c_tilde[j]  * X[j] for j in P)

    # Objective: (1 - z)*expr_base + z*expr_campaign - z*C
    m.setObjective((1 - z) * expr_base + z * expr_campaign - z * C, GRB.MAXIMIZE)

    # Resource constraint: sum((1/a[j]) * x_j) <= b
    m.addConstr(sum((1.0 / a[j]) * X[j] for j in P) <= b, name="resource_limit")

    # Optimize
    m.optimize()

    # Retrieve the total profit
    if m.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {j: X[j].X for j in P}
        z_sol = z.X
        
        all_vars = {
            "production": x_sol,
            "campaign": z_sol
        }
        total_profit = int(round(m.objVal))
        return total_profit, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")