def prod(n, a, c, u, b, c_penalty):
    """
    Args:
        n: number of products
        a: a list of numbers, parameter for each element in set P
        c: a list of numbers, profit coefficient for each element in set P
        u: a list of numbers, upper limit for each element in set P
        b: a number, the global resource limit
        c_penalty: a number, cost per unit of exceeding the resource limit

    Returns:
        total_profit: the maximum total profit accounting for penalty
    """
    # Import gurobipy
    from gurobipy import Model, GRB

    # Basic input checks
    if not (len(a) == len(c) == len(u) == n):
        raise ValueError("Input lists a, c, and u must have length n")
    for j in range(n):
        if a[j] == 0:
            raise ValueError(f"Parameter a[{j}] is zero, division by zero not allowed.")
        if u[j] < 0:
            raise ValueError(f"Upper limit u[{j}] cannot be negative.")

    # Create the model
    m = Model("prod_with_penalty")
    m.setParam('OutputFlag', 0)  # Suppress Gurobi output
    m.Params.NonConvex = 2  # Enable non-convex optimization

    # Index set
    P = range(n)

    # Decision variables x_j
    X = {}
    for j in P:
        X[j] = m.addVar(lb=0, ub=u[j], name=f"X_{j}")

    # Expression for total normalized consumption minus b
    consumption = sum((1.0 / a[j]) * X[j] for j in P)
    
    # Create variable for consumption - b
    consumption_minus_b = m.addVar(name="consumption_minus_b")
    m.addConstr(consumption_minus_b == consumption - b, name="consumption_def")

    # Auxiliary variable for the positive part: max{0, consumption - b}
    Excess = m.addVar(lb=0, name="Excess")
    m.addGenConstrMax(Excess, [consumption_minus_b, 0], name="excess_def")

    # Objective: maximize sum(c_j * x_j) - c_penalty * Excess
    m.setObjective(
        sum(c[j] * X[j] for j in P) - c_penalty * Excess,
        GRB.MAXIMIZE
    )

    # Optimize
    m.optimize()

    # Retrieve the optimal objective
    if m.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {j: X[j].X for j in P}
        consumption_minus_b_sol = consumption_minus_b.X
        excess_sol = Excess.X
        
        all_vars = {
            "production": x_sol,
            "consumption_minus_b": consumption_minus_b_sol,
            "excess": excess_sol
        }
        return m.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")