# filename: revenue_maximization.py

import gurobipy as gp
from gurobipy import GRB

def revenue_maximization(n_packages, n_legs, c, r, r_tilde, x_lb, x_ub, delta, C):
    """
    Args:
        n_packages: Number of packages
        n_legs: Number of flight legs
        c: List of ints, available seats per flight leg (length n_legs)
        r: List of ints, revenue per package without campaign (length n_packages)
        r_tilde: List of ints, revenue per package with campaign (length n_packages)
        x_lb: List of ints, lower bound on packages sold (length n_packages)
        x_ub: List of ints, upper bound on packages sold (length n_packages)
        delta: 2D list (n_packages × n_legs), 1 if package i uses leg j
        C: Int, fixed cost of marketing campaign

    Returns:
        objective_value: maximum revenue achieved
    """
    # Create model
    m = gp.Model("Revenue_Maximization")
    m.Params.OutputFlag = 0   # turn off solver output
    m.Params.NonConvex = 2    # enable nonconvex for bilinear terms

    # Indices
    packages = range(n_packages)
    legs = range(n_legs)

    # Decision variables
    x = m.addVars(
        packages,
        lb={i: x_lb[i] for i in packages},
        ub={i: x_ub[i] for i in packages},
        vtype=GRB.INTEGER,
        name="x"
    )
    z = m.addVar(vtype=GRB.BINARY, name="z")

    # Capacity constraints for each leg
    for j in legs:
        m.addConstr(
            gp.quicksum(delta[i][j] * x[i] for i in packages)
            <= c[j],
            name=f"Capacity_Leg_{j}"
        )

    # Objective: (1-z)*∑ r[i] x[i] + z*∑ r_tilde[i] x[i] - z*C
    # = ∑ r[i] x[i] + z*∑ (r_tilde[i] - r[i]) x[i] - z*C
    obj = gp.quicksum(r[i] * x[i] for i in packages) + \
          z * gp.quicksum((r_tilde[i] - r[i]) * x[i] for i in packages) - \
          C * z
    m.setObjective(obj, GRB.MAXIMIZE)

    # Solve
    m.optimize()

    if m.status == GRB.OPTIMAL:
        # Extract ALL solution variables
        x_sol = {i: x[i].X for i in packages}
        z_sol = z.X
        
        all_vars = {
            "package_sales": x_sol,
            "campaign": z_sol
        }
        return m.objVal, all_vars
    else:
        raise RuntimeError("Model did not solve to optimality.")