# LLM Model Manager

The LLM Model Manager provides a unified interface for using different LLM providers (OpenAI, Google) with their specific configurations and parameters.

## Features

- **Multi-Provider Support**: OpenAI, Google
- **Flexible Configuration**: Use simple model names or detailed configurations
- **Backward Compatibility**: Existing code continues to work with minimal changes
- **Error Handling**: Graceful handling of missing dependencies and API keys
- **Custom Parameters**: Support for provider-specific parameters

## Quick Start

### Basic Usage

```python
from llm_model_manager import create_llm

# Use a simple model name
llm = create_llm("gpt-4o")

# Use a custom configuration
from llm_model_manager import LLMConfig
config = LLMConfig(
    provider="openai",
    model_name="gpt-4o",
    temperature=0.1,
    max_tokens=2000
)
llm = create_llm(config)
```

### Available Models

```python
from llm_model_manager import get_available_models

available = get_available_models()
print(available)
```

## Configuration Options

### LLMConfig Parameters

- `provider`: Provider name ("openai", "google")
- `model_name`: Specific model name
- `temperature`: Sampling temperature (0.0-1.0)
- `max_tokens`: Maximum tokens to generate
- `top_p`: Nucleus sampling parameter
- `frequency_penalty`: Frequency penalty for OpenAI
- `presence_penalty`: Presence penalty for OpenAI
- `timeout`: Request timeout in seconds
- `max_retries`: Maximum retry attempts
- `custom_parameters`: Provider-specific parameters

### Provider-Specific Parameters

#### OpenAI
- `frequency_penalty`: -2.0 to 2.0
- `presence_penalty`: -2.0 to 2.0
- `request_timeout`: Request timeout

#### Google
- `safety_settings`: Safety filter settings
- `max_output_tokens`: Alternative to max_tokens

## Usage in Workflow Components

### NonLinearPatternExtractor

```python
from nonlinear_detector import NonLinearPatternExtractor

# Use different models for different tasks
extractor = NonLinearPatternExtractor("gpt-4o")  # For pattern detection

```

### PatternReformulationAgents

```python
from pattern_reformulation_agents import PatternReformulationAgents

# Use Gemini for reformulation tasks
agents = PatternReformulationAgents("gemini-2.5-flash")
```

### CodeConverter

```python
from code_converter import CodeConverter

# Use Gemini for code generation
converter = CodeConverter("gemini-2.5-pro")
```

### MIPCoordinator

```python
from mip_coordinator import MIPCoordinator

# Use GPT-4 for coordination
coordinator = MIPCoordinator("gpt-4o", verbose=True)
```

## Command Line Usage

### Using the LinearizeLLM Workflow

```bash
# Use default model (o3/gpt-4o)
python run_linearizellm_data.py --file blend_problem.tex

# Use specific model
python run_linearizellm_data.py --file blend_problem.tex --model gemini-2.5-flash

# Use custom configuration file
python run_linearizellm_data.py --file blend_problem.tex --model-config example_model_config.json
```

### Configuration File Format

Create a JSON file with your model configuration:

```json
{
  "provider": "openai",
  "model_name": "gpt-4o",
  "temperature": 0.0,
  "max_tokens": 4000,
  "timeout": 60,
  "max_retries": 3
}
```

## API Key Management

The system automatically handles API keys for different providers:

### Environment Variables

```bash
# OpenAI
export OPENAI_API_KEY="your-openai-key"

# Google
export GOOGLE_API_KEY="your-google-key"
```

### Programmatic Setting

```python
from api_key_manager import APIKeyManager

manager = APIKeyManager("OpenAI")
manager.set_api_key("your-key")
```

## Error Handling

The system provides comprehensive error handling:

```python
try:
    llm = create_llm("gpt-4o")
except ImportError as e:
    print(f"Missing dependency: {e}")
except ValueError as e:
    print(f"Invalid configuration: {e}")
except Exception as e:
    print(f"Unexpected error: {e}")
```

## Installation

### Required Dependencies

```bash
# For OpenAI
pip install langchain-openai

# For Google
pip install langchain-google-genai
```

## Migration Guide

### From Old Code

**Before:**
```python
from langchain_openai import ChatOpenAI
from api_key_manager import get_openai_api_key

llm = ChatOpenAI(
    model="gpt-4o",
    openai_api_key=get_openai_api_key()
)
```

**After:**
```python
from llm_model_manager import create_llm

llm = create_llm("gpt-4o")
```

### Component Updates

All components now accept either a string (model name) or LLMConfig object:

```python
# String (backward compatible)
extractor = NonLinearPatternExtractor("o3")

# LLMConfig object
from llm_model_manager import LLMConfig
config = LLMConfig("openai", "gpt-4o", temperature=0.1)
extractor = NonLinearPatternExtractor(config)
```

## Troubleshooting

### Common Issues

1. **ImportError**: Missing provider package
   - Install the required package: `pip install langchain-openai`

2. **ValueError**: Invalid model name
   - Check available models with `get_available_models()`

3. **API Key Error**: Missing or invalid API key
   - Set environment variable: `export OPENAI_API_KEY="your-key"`
   - Or let the system prompt you interactively
   - **Note**: No default API keys are provided - you must obtain your own

4. **Timeout Error**: Request taking too long
   - Increase timeout in configuration

### Debug Mode

Enable verbose logging to see detailed information:

```python
from llm_model_manager import llm_manager

# Check available providers
print(llm_manager.providers.keys())

# Check default configurations
print(llm_manager.default_configs.keys())
```

## Contributing

To add a new provider:

1. Create a new provider class inheriting from `LLMProvider`
2. Implement `create_llm()` and `get_api_key()` methods
3. Add the provider to `LLMModelManager.providers`
4. Add default configurations to `LLMModelManager.default_configs`
5. Update the `get_available_models()` method

## License

This project is part of the LinearizeLLM framework. 