#!/usr/bin/env python3
"""
Enhanced LinearizeLLM Experiments Runner with Interactive Menu

This script provides an interactive menu to run LinearizeLLM experiments with:
- All the same options as run_linearizellm.py
- Additional context scenarios (no_context, partial_info)
- Seed configuration options
- Both single runs and experimental runs
"""

import subprocess
import sys
import argparse
from pathlib import Path
import random

# Add the src directory to the Python path
sys.path.insert(0, str(Path(__file__).parent / "src"))

# Check dependencies before importing any other modules
from utils.dependency_checker import check_dependencies

if not check_dependencies():
    print("\n❌ LinearizeLLM Experiments cannot start due to missing dependencies.")
    print("Please install all required packages and try again.")
    sys.exit(1)


def get_available_instances():
    """Get all available problem instances."""
    instances_dir = Path("data/LinearizeLLM_data/instances_linearizellm")
    instances = []
    if instances_dir.exists():
        for item in instances_dir.iterdir():
            if item.is_dir() and (item / f"{item.name}.tex").exists():
                instances.append(item.name)
    return sorted(instances)


def run_single_linearizellm(tex_file_path, llm_model="gemini-2.5-flash"):
    """Run a single LinearizeLLM workflow on a specific file."""
    print(f"🚀 Running LinearizeLLM on: {tex_file_path.name}")
    
    cmd = [
        sys.executable, "run_linearizellm.py",
        "--file", tex_file_path.parent.name,
        "--model", llm_model
    ]
    
    print(f"Running: {' '.join(cmd)}")
    subprocess.run(cmd)


def run_single_context_experiment(instance, scenario, seed, llm_model="gemini-2.5-flash"):
    """Run a single context experiment for testing."""
    print(f"🧪 Running context experiment: {instance} - {scenario} - seed {seed} - {llm_model}")
    
    cmd = [
        sys.executable, "experimental_setup.py",
        "--single-experiment",
        "--instance", instance,
        "--scenario", scenario,
        "--seed", str(seed),
        "--results-dir", "data/context_experiment_results",
        "--llm-model", llm_model
    ]
    
    print(f"Running: {' '.join(cmd)}")
    subprocess.run(cmd)


def run_context_experiment_suite(instances, scenarios, seeds, llm_model="gemini-2.5-flash"):
    """Run a full context experiment suite."""
    print(f"🚀 Running context experiment suite...")
    print(f"   📊 Instances: {len(instances)}")
    print(f"   🔍 Scenarios: {scenarios}")
    print(f"   🎲 Seeds: {seeds}")
    print(f"   🤖 LLM Model: {llm_model}")
    print(f"   📈 Total experiments: {len(instances) * len(scenarios) * len(seeds)}")
    
    response = input("\nContinue? (y/N): ")
    if response.lower() != 'y':
        print("Experiment cancelled.")
        return
    
    cmd = [
        sys.executable, "experimental_setup.py",
        "--instances"] + instances + [
        "--scenarios"] + scenarios + [
        "--seeds"] + [str(s) for s in seeds] + [
        "--results-dir", "data/context_experiment_results",
        "--llm-model", llm_model
    ]
    
    print(f"Running: {' '.join(cmd)}")
    subprocess.run(cmd)


def select_llm_model():
    """Let user select LLM model."""
    print("\n🤖 LLM Model Selection:")
    print("1. gemini-2.5-flash (recommended)")
    print("2. o3")
    
    choice = input("Enter your choice (1-2) [default: 1]: ").strip()
    if choice == "2":
        return "o3"
    else:
        return "gemini-2.5-flash"


def select_context_scenarios():
    """Let user select context scenarios."""
    print("\n🔍 Context Scenario Selection:")
    print("="*50)
    print("1. no_context only")
    print("   • Only LaTeX model provided")
    print("   • No parameter context or decision variable info")
    print("   • Tests baseline performance with minimal information")
    print()
    print("2. partial_info only")
    print("   • LaTeX model + decision variable context")
    print("   • Knows which symbols are decision variables")
    print("   • No concrete parameter values")
    print("   • Tests impact of variable type knowledge")
    print()
    print("3. both scenarios")
    print("   • Runs both no_context and partial_info")
    print("   • Compares performance across information levels")
    print()
    
    choice = input("Enter your choice (1-3) [default: 3]: ").strip()
    if choice == "1":
        return ["no_context"]
    elif choice == "2":
        return ["partial_info"]
    else:
        return ["no_context", "partial_info"]


def select_seeds():
    """Let user select seeds."""
    print("\n🎲 Seed Selection:")
    print("1. Single seed (seed 1)")
    print("2. Multiple seeds (1, 2, 3)")
    print("3. All seeds (1, 2, 3, 4, 5)")
    print("4. Custom seeds")
    
    choice = input("Enter your choice (1-4) [default: 1]: ").strip()
    if choice == "2":
        return [1, 2, 3]
    elif choice == "3":
        return [1, 2, 3, 4, 5]
    elif choice == "4":
        seeds_input = input("Enter seeds separated by commas (e.g., 1,3,5): ").strip()
        try:
            return [int(s.strip()) for s in seeds_input.split(',')]
        except ValueError:
            print("❌ Invalid seed format, using seed 1")
            return [1]
    else:
        return [1]


def interactive_menu():
    """Main interactive menu."""
    instances = get_available_instances()
    if not instances:
        print("❌ No problem instances found!")
        return
    
    print(f"\n🔬 LINEARIZELLM EXPERIMENTS RUNNER")
    print("="*50)
    print(f"📊 Available problems: {len(instances)}")
    print(f"📋 Problems: {instances}")
    print()
    print("🧪 Context Experiments: Test how different information levels affect LLM performance")
    print("   • no_context: Only LaTeX model (minimal information)")
    print("   • partial_info: LaTeX + decision variable types (medium information)")
    
    print("\n🎯 Experiment Types:")
    print("1. Single LinearizeLLM run (like run_linearizellm.py)")
    print("2. Single context experiment")
    print("   • Test how information level affects LLM performance")
    print("   • Compare no_context vs partial_info scenarios")
    print("3. Multiple context experiments")
    print("   • Run context experiments across multiple problems")
    print("4. Full experimental suite")
    print("   • Complete evaluation across all problems/scenarios/seeds")
    print("5. Quick test (single instance, both scenarios, seed 1)")
    print("   • Fast validation with media_selection_nonlinear_bincon")
    
    choice = input("\nEnter your choice (1-5): ").strip()
    
    if choice == "1":
        # Single LinearizeLLM run
        run_single_linearizellm_menu(instances)
    
    elif choice == "2":
        # Single context experiment
        run_single_context_experiment_menu(instances)
    
    elif choice == "3":
        # Multiple context experiments
        run_multiple_context_experiments_menu(instances)
    
    elif choice == "4":
        # Full experimental suite
        run_full_experimental_suite_menu(instances)
    
    elif choice == "5":
        # Quick test
        run_quick_test()
    
    else:
        print("❌ Invalid choice")


def run_single_linearizellm_menu(instances):
    """Menu for single LinearizeLLM run."""
    print(f"\n📄 SINGLE LINEARIZELLM RUN")
    print("="*30)
    
    print("\nProblem Selection Options:")
    print("1. Process all problems")
    print("2. Process specific problem")
    print("3. Process first N problems")
    print("4. Process range of problems")
    print("5. Process multiple specific problems")
    print("6. Process random sample")
    
    choice = input("\nEnter your choice (1-6): ").strip()
    
    llm_model = select_llm_model()
    instances_dir = Path("data/LinearizeLLM_data/instances_linearizellm")
    
    if choice == "1":
        # Process all
        for instance in instances:
            tex_file = instances_dir / instance / f"{instance}.tex"
            run_single_linearizellm(tex_file, llm_model)
    
    elif choice == "2":
        # Specific problem
        print(f"\n📋 Available problems:")
        for i, instance in enumerate(instances, 1):
            print(f"   {i:2d}. {instance}")
        print()
        problem_name = input("Enter problem name: ").strip().strip("'\"")
        tex_file = instances_dir / problem_name / f"{problem_name}.tex"
        if tex_file.exists():
            run_single_linearizellm(tex_file, llm_model)
        else:
            print(f"❌ Problem {problem_name} not found")
    
    elif choice == "3":
        # First N
        try:
            n = int(input("Enter number of problems: ").strip())
            for instance in instances[:n]:
                tex_file = instances_dir / instance / f"{instance}.tex"
                run_single_linearizellm(tex_file, llm_model)
        except ValueError:
            print("❌ Invalid number")
    
    elif choice == "4":
        # Range
        range_input = input("Enter range (e.g., 0-2): ").strip()
        try:
            if '-' in range_input:
                start, end = map(int, range_input.split('-'))
                for instance in instances[start:end+1]:
                    tex_file = instances_dir / instance / f"{instance}.tex"
                    run_single_linearizellm(tex_file, llm_model)
            else:
                print("❌ Invalid range format")
        except ValueError:
            print("❌ Invalid range format")
    
    elif choice == "5":
        # Multiple specific
        print(f"\n📋 Available problems:")
        for i, instance in enumerate(instances, 1):
            print(f"   {i:2d}. {instance}")
        print()
        problems_input = input("Enter problem names separated by commas: ").strip()
        try:
            problem_names = [name.strip().strip("'\"") for name in problems_input.split(',')]
            for name in problem_names:
                tex_file = instances_dir / name / f"{name}.tex"
                if tex_file.exists():
                    run_single_linearizellm(tex_file, llm_model)
                else:
                    print(f"⚠️ Problem {name} not found, skipping...")
        except ValueError:
            print("❌ Invalid format")
    
    elif choice == "6":
        # Random sample
        try:
            n = int(input("Enter number of random problems: ").strip())
            selected = random.sample(instances, min(n, len(instances)))
            for instance in selected:
                tex_file = instances_dir / instance / f"{instance}.tex"
                run_single_linearizellm(tex_file, llm_model)
        except ValueError:
            print("❌ Invalid number")


def run_single_context_experiment_menu(instances):
    """Menu for single context experiment."""
    print(f"\n🧪 SINGLE CONTEXT EXPERIMENT")
    print("="*30)
    
    # Select instance
    print(f"📋 Available problems:")
    for i, instance_name in enumerate(instances, 1):
        print(f"   {i:2d}. {instance_name}")
    print()
    instance = input("Enter problem name: ").strip().strip("'\"")
    if instance not in instances:
        print(f"❌ Problem {instance} not found")
        return
    
    # Select scenario
    scenarios = select_context_scenarios()
    
    # Select seed
    seeds = select_seeds()
    
    # Select LLM model
    llm_model = select_llm_model()
    
    # Run experiments
    for scenario in scenarios:
        for seed in seeds:
            run_single_context_experiment(instance, scenario, seed, llm_model)


def run_multiple_context_experiments_menu(instances):
    """Menu for multiple context experiments."""
    print(f"\n🧪 MULTIPLE CONTEXT EXPERIMENTS")
    print("="*35)
    
    # Select instances
    print("Instance Selection Options:")
    print("1. All instances")
    print("2. Specific instances")
    print("3. First N instances")
    print("4. Random N instances")
    
    choice = input("Enter your choice (1-4): ").strip()
    
    if choice == "1":
        selected_instances = instances
    elif choice == "2":
        print(f"\n📋 Available problems:")
        for i, instance in enumerate(instances, 1):
            print(f"   {i:2d}. {instance}")
        print()
        problems_input = input("Enter problem names separated by commas: ").strip()
        try:
            selected_instances = [name.strip().strip("'\"") for name in problems_input.split(',')]
            selected_instances = [i for i in selected_instances if i in instances]
        except ValueError:
            print("❌ Invalid format")
            return
    elif choice == "3":
        try:
            n = int(input("Enter number of instances: ").strip())
            selected_instances = instances[:n]
        except ValueError:
            print("❌ Invalid number")
            return
    elif choice == "4":
        try:
            n = int(input("Enter number of random instances: ").strip())
            selected_instances = random.sample(instances, min(n, len(instances)))
        except ValueError:
            print("❌ Invalid number")
            return
    else:
        print("❌ Invalid choice")
        return
    
    # Select scenarios and seeds
    scenarios = select_context_scenarios()
    seeds = select_seeds()
    llm_model = select_llm_model()
    
    # Run experiments
    run_context_experiment_suite(selected_instances, scenarios, seeds, llm_model)


def run_full_experimental_suite_menu(instances):
    """Menu for full experimental suite."""
    print(f"\n🚀 FULL EXPERIMENTAL SUITE")
    print("="*30)
    print("This will run all instances with all scenarios and all seeds.")
    print(f"Total experiments: {len(instances)} * 2 scenarios * 5 seeds = {len(instances) * 2 * 5}")
    
    llm_model = select_llm_model()
    
    run_context_experiment_suite(instances, ["no_context", "partial_info"], [1, 2, 3, 4, 5], llm_model)


def run_quick_test(llm_model="gemini-2.5-flash"):
    """Run quick test with single instance."""
    print(f"\n⚡ QUICK TEST")
    print("="*15)
    print("Running: media_selection_nonlinear_bincon with both scenarios, seed 1")
    
    test_instance = "media_selection_nonlinear_bincon"
    scenarios = ["no_context", "partial_info"]
    seed = 1
    
    for scenario in scenarios:
        run_single_context_experiment(test_instance, scenario, seed, llm_model)


def main():
    parser = argparse.ArgumentParser(description='Enhanced LinearizeLLM Experiments Runner')
    parser.add_argument('--interactive', action='store_true', default=True,
                       help='Run in interactive mode (default)')
    parser.add_argument('--quick-test', action='store_true',
                       help='Run quick test')
    parser.add_argument('--llm-model', type=str, default='gemini-2.5-flash',
                       choices=['o3', 'gemini-2.5-flash'],
                       help='LLM model to use')
    
    args = parser.parse_args()
    
    if args.quick_test:
        run_quick_test(args.llm_model)
    else:
        interactive_menu()


if __name__ == "__main__":
    main() 