#!/usr/bin/env python3
"""
Generalized Optimization Framework
Runs any code.py file with its associated parameters.json
"""

import json
import sys
import os
import importlib.util
import traceback
from pathlib import Path
from typing import Dict, Any, Optional, List, Tuple
from datetime import datetime

def convert_tuple_keys_to_strings(obj):
    """Recursively convert all tuple keys to strings in nested structures."""
    if isinstance(obj, dict):
        return {str(k) if isinstance(k, tuple) else k: convert_tuple_keys_to_strings(v) for k, v in obj.items()}
    elif isinstance(obj, list):
        return [convert_tuple_keys_to_strings(item) for item in obj]
    elif isinstance(obj, tuple):
        return str(obj)
    else:
        return obj

class TupleKeyEncoder(json.JSONEncoder):
    """Custom JSON encoder that converts tuple keys to strings recursively."""
    def default(self, obj):
        if isinstance(obj, dict):
            return {str(k) if isinstance(k, tuple) else k: self._convert_value(v) for k, v in obj.items()}
        elif isinstance(obj, list):
            return [self._convert_value(item) for item in obj]
        return super().default(obj)
    
    def _convert_value(self, value):
        """Recursively convert values that might contain tuples."""
        if isinstance(value, dict):
            return {str(k) if isinstance(k, tuple) else k: self._convert_value(v) for k, v in value.items()}
        elif isinstance(value, list):
            return [self._convert_value(item) for item in value]
        elif isinstance(value, tuple):
            return str(value)
        return value

class OptimizationFramework:
    """
    A framework to run optimization problems with their associated parameters.
    """
    
    def __init__(self, base_dir: str = "data/LinearizeLLM_data/instances_linearizellm"):
        """
        Initialize the framework.
        
        Args:
            base_dir: Base directory containing problem instances
        """
        self.base_dir = Path(base_dir)
        
        # Define problem-specific parameter mappings
        self.problem_handlers = {
            'diet_problem_min_abs': self._handle_diet_problem_min_abs,
            'diet_problem_nonlinear_frac': self._handle_diet_problem_nonlinear_frac,
            'diet_problem_nonlinear_frac_': self._handle_diet_problem_nonlinear_frac,
            'diet_problem_nonlinear_frac_min': self._handle_diet_problem_nonlinear_frac_min,
            'diet_problem_monotone': self._handle_diet_problem_monotone,
            'blend_problem_abs': self._handle_blend_problem_abs,
            'blend_problem_frac': self._handle_blend_problem_frac,
            'knapsack_problem_nonlinear_min_1': self._handle_knapsack_problem_min_1,
            'knapsack_problem_nonlinear_min_2': self._handle_knapsack_problem_min_2,
            'netmcol_nonlinear_frac': self._handle_network_problem,
            'netmcol_nonlinear_bincon': self._handle_network_problem,
            'netasgn_nonlinear_abs': self._handle_network_problem,
            'netasgn_nonlinear_max': self._handle_network_problem,
            'nltrans_nonlinear_bincon': self._handle_network_problem,
            'nltrans_nonlinear_max': self._handle_network_problem,
            'media_selection_nonlinear_binbin': self._handle_media_selection_binbin,
            'media_selection_nonlinear_bincon': self._handle_media_selection_bincon,
            'multi_nonlinear_abs': self._handle_multi_problem,
            'multi_nonlinear_bincon': self._handle_multi_problem,
            'prod_nonlinear_bincon': self._handle_production_problem,
            'prod_nonlinear_max': self._handle_production_problem,
            'revenue_maximization_nonlinear_bincon': self._handle_revenue_problem,
            'aircraft_monotone_test': self._handle_aircraft_problem,
            'aircraft_problem_nonlinear_max': self._handle_aircraft_problem,
            'aircraft_monotone_max': self._handle_aircraft_problem,
        }
        
    def list_available_problems(self) -> list:
        """
        List all available optimization problems.
        
        Returns:
            List of problem names that have both code.py and parameters.json
        """
        problems = []
        
        if not self.base_dir.exists():
            print(f"❌ Base directory {self.base_dir} does not exist")
            return problems
            
        for problem_dir in self.base_dir.iterdir():
            if problem_dir.is_dir():
                code_file = problem_dir / "code.py"
                params_file = problem_dir / "parameters.json"
                
                if code_file.exists() and params_file.exists():
                    problems.append(problem_dir.name)
        
        return sorted(problems)
    
    def load_parameters(self, problem_name: str) -> Dict[str, Any]:
        """
        Load parameters from parameters.json for a given problem.
        
        Args:
            problem_name: Name of the problem directory
            
        Returns:
            Dictionary containing the parameters
        """
        params_file = self.base_dir / problem_name / "parameters.json"
        
        if not params_file.exists():
            raise FileNotFoundError(f"Parameters file not found: {params_file}")
        
        with open(params_file, 'r') as f:
            return json.load(f)
    
    def load_optimization_code(self, problem_name: str):
        """
        Load the optimization code from code.py for a given problem.
        
        Args:
            problem_name: Name of the problem directory
            
        Returns:
            The loaded module containing the optimization code
        """
        code_file = self.base_dir / problem_name / "code.py"
        
        if not code_file.exists():
            raise FileNotFoundError(f"Code file not found: {code_file}")
        
        # Load the module dynamically
        spec = importlib.util.spec_from_file_location(f"optimization_{problem_name}", code_file)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        
        return module
    
    def analyze_code_functions(self, module) -> list:
        """
        Analyze the loaded module to find optimization functions.
        
        Args:
            module: The loaded optimization module
            
        Returns:
            List of function names that could be optimization functions
        """
        functions = []
        
        for name in dir(module):
            obj = getattr(module, name)
            if callable(obj) and not name.startswith('_'):
                # Check if it looks like an optimization function
                if any(keyword in name.lower() for keyword in ['optimize', 'solve', 'run', 'diet', 'network', 'blend', 'knapsack', 'media', 'production', 'revenue', 'aircraft', 'netasgn', 'netmcol', 'multi', 'nltrans', 'ntrans', 'prod']):
                    functions.append(name)
        
        return functions
    
    # Problem-specific handlers
    def _handle_diet_problem_min_abs(self, optimization_func, params):
        """Handle diet_problem_min_abs specifically."""
        return optimization_func(
            p=params.get('p', []), 
            a=params.get('a', []), 
            m_lower=params.get('m', []),
            M=params.get('M', []), 
            x_lb=params.get('underline_x', []), 
            x_ub=params.get('overline_x', []), 
            B=params.get('B', 0)
        )

    def _handle_diet_problem_nonlinear_frac(self, optimization_func, params):
        """Handle diet_problem_nonlinear_frac specifically."""
        nutrients = [f"nutrient_{i}" for i in range(params.get('I', 0))]
        foods = [f"food_{j}" for j in range(params.get('J', 0))]
        return optimization_func(
            nutrients=nutrients,
            foods=foods,
            a=params.get('a', []),
            p=params.get('p', []),
            x_min=params.get('underline_x', []),
            x_max=params.get('overline_x', []),
            M_req=params.get('M', [])
        )

    def _handle_diet_problem_nonlinear_frac_min(self, optimization_func, params):
        """Handle diet_problem_nonlinear_frac_min specifically."""
        nutrients = [f"nutrient_{i}" for i in range(params.get('I', 0))]
        foods = [f"food_{j}" for j in range(params.get('J', 0))]
        return optimization_func(
            nutrients=nutrients,
            foods=foods,
            a=params.get('a', []),
            p=params.get('p', []),
            x_min=params.get('underline_x', []),
            x_max=params.get('overline_x', []),
            m_req=params.get('m', []),
            M_req=params.get('M', [])
        )
    
    def _handle_diet_problem_monotone(self, optimization_func, params):
        """Handle diet_problem_monotone specifically."""
        return optimization_func(
            p=params.get('p', []),
            a=params.get('a', []),
            m=params.get('m', []),
            M=params.get('M', []),
            underline_x=params.get('underline_x', []),
            overline_x=params.get('overline_x', [])
        )
    
    def _handle_blend_problem_abs(self, optimization_func, params):
        """Handle blend_problem_abs specifically."""
        return optimization_func(
            J=params.get('J', 0),
            I=params.get('I', 0),
            p=params.get('p', []),
            c=params.get('c', []),
            d=params.get('d', []),
            lam=params.get('lam', [])
        )

    def _handle_blend_problem_frac(self, optimization_func, params):
        """Handle blend_problem_frac specifically."""
        return optimization_func(
            unit_costs=params.get('c', []),
            heat_contents=params.get('h', []),
            min_heat_kcal=params.get('heat_min', 7200),
            total_mass_kg=params.get('mix_total', 1000)
        )
    
    def _handle_knapsack_problem_min_1(self, optimization_func, params):
        """Handle knapsack_problem_nonlinear_min_1 specifically."""
        return optimization_func(
            item_values=params.get('v', []),
            item_weights=params.get('w', []),
            lb_x=[0] * len(params.get('v', [])),  # Default lower bounds
            ub_x=[1] * len(params.get('v', [])),  # Default upper bounds
            max_weight_knapsack=params.get('W', 0)
        )

    def _handle_knapsack_problem_min_2(self, optimization_func, params):
        """Handle knapsack_problem_nonlinear_min_2 specifically."""
        return optimization_func(
            n=params.get('n', 0),
            v=params.get('v', []),
            w=params.get('w', []),
            W=params.get('W', 0),
            b=params.get('b', 0)
        )
    
    def _handle_network_problem(self, optimization_func, params):
        """Handle network optimization problems."""
        # Check function signature to determine which network function
        import inspect
        sig = inspect.signature(optimization_func)
        param_names = list(sig.parameters.keys())
        
        if 'I' in param_names and 'J' in param_names and 'S' in param_names and 'D' in param_names and 'r_tilde' in param_names:  # nltrans function
            return optimization_func(
                I=params.get('I', 0),
                J=params.get('J', 0),
                S=params.get('S', []),
                D=params.get('D', []),
                r=params.get('r', []),
                r_tilde=params.get('r_tilde', []),
                ell=params.get('ell', []),
                C=params.get('C', 0)
            )
        elif 'I' in param_names and 'J' in param_names and 'S' in param_names and 'D' in param_names and 'c' in param_names and 'ell' in param_names:  # ntrans function
            return optimization_func(
                I=params.get('I', 0),
                J=params.get('J', 0),
                S=params.get('S', []),
                D=params.get('D', []),
                r=params.get('r', []),
                ell=params.get('ell', []),
                c=params.get('c', 0)
            )
        elif 'I' in param_names and 'J' in param_names and 'S' in param_names and 'D' in param_names:  # netasgn functions
            return optimization_func(
                I=params.get('I', 0),
                J=params.get('J', 0),
                S=params.get('S', []),
                D=params.get('D', []),
                c=params.get('c', []),
                ell=params.get('ell', []),
                r=params.get('r', 0)
            )
        elif 'n' in param_names and 'P' in param_names and 'Supply' in param_names and 'Demand' in param_names:  # netmcol function
            return optimization_func(
                n=params.get('n', 0),
                P=params.get('P', 0),
                Supply=params.get('Supply', []),
                Demand=params.get('Demand', []),
                ShipmentCost=params.get('ShipmentCost', []),
                Capacity=params.get('Capacity', []),
                JointCapacity=params.get('JointCapacity', []),
                ContractCosts=params.get('ContractCosts', [])
            )
        elif 'C' in param_names and 'P' in param_names and 'L' in param_names and 'link_from' in param_names:  # netmcol fractional function
            return optimization_func(
                C=params.get('C', 0),
                P=params.get('P', 0),
                L=params.get('L', 0),
                Supply=params.get('Supply', []),
                Demand=params.get('Demand', []),
                ShipmentCost=params.get('ShipmentCost', []),
                Capacity=params.get('Capacity', []),
                JointCapacity=params.get('JointCapacity', []),
                r=params.get('r', []),
                link_from=params.get('link_from', []),
                link_to=params.get('link_to', [])
            )

        else:  # other network functions
            return optimization_func(
                C=params.get('C', 0),
                P=params.get('P', 0),
                L=params.get('L', 0),
                Supply=params.get('Supply', []),
                Demand=params.get('Demand', []),
                ShipmentCost=params.get('ShipmentCost', []),
                Capacity=params.get('Capacity', []),
                JointCapacity=params.get('JointCapacity', []),
                r=params.get('r', []),
                link_from=params.get('link_from', []),
                link_to=params.get('link_to', [])
            )
    
    def _handle_media_selection_binbin(self, optimization_func, params):
        """Handle media_selection_nonlinear_binbin specifically."""
        return optimization_func(
            c=params.get('c', []),
            d=params.get('d', []),
            a=params.get('a', []),
            m=params.get('m', [])
        )

    def _handle_media_selection_bincon(self, optimization_func, params):
        """Handle media_selection_nonlinear_bincon specifically."""
        return optimization_func(
            c=params.get('c', []),
            r=params.get('r', []),
            a=params.get('a', []),
            m=params.get('m', []),
            budget=params.get('budget', 0)
        )
    
    def _handle_multi_problem(self, optimization_func, params):
        """Handle multi-objective optimization problems."""
        # Check function signature to determine which multi function
        import inspect
        sig = inspect.signature(optimization_func)
        param_names = list(sig.parameters.keys())
        
        if 'Origins' in param_names and 'Destinations' in param_names and 'Products' in param_names:  # multi function
            return optimization_func(
                Origins=params.get('Origins', 0),
                Destinations=params.get('Destinations', 0),
                Products=params.get('Products', 0),
                Supply=params.get('Supply', []),
                Demand=params.get('Demand', []),
                Limit=params.get('Limit', []),
                Cost=params.get('Cost', []),
                r=params.get('r', 0)
            )
        elif 'I' in param_names and 'J' in param_names and 'P' in param_names and 'l' in param_names:  # multi bincon function
            return optimization_func(
                I=params.get('I', 0),
                J=params.get('J', 0),
                P=params.get('P', 0),
                supply=params.get('supply', []),
                demand=params.get('demand', []),
                limit=params.get('limit', []),
                cost=params.get('cost', []),
                l=params.get('l', 0),
                b=params.get('b', 0),
                B=params.get('B', 0)
            )
        else:  # generic multi-objective
            return optimization_func(
                objectives=params.get('objectives', []),
                constraints=params.get('constraints', []),
                variables=params.get('variables', [])
            )
    
    def _handle_production_problem(self, optimization_func, params):
        """Handle production optimization problems."""
        # Check function signature to determine which production function
        import inspect
        sig = inspect.signature(optimization_func)
        param_names = list(sig.parameters.keys())
        
        if 'n' in param_names and 'a' in param_names and 'c_tilde' in param_names:  # prod_with_campaign function
            return optimization_func(
                n=params.get('n', 0),
                a=params.get('a', []),
                c=params.get('c', []),
                u=params.get('u', []),
                c_tilde=params.get('c_tilde', []),
                b=params.get('b', 0),
                C=params.get('C', 0)
            )
        elif 'n' in param_names and 'a' in param_names and 'c_penalty' in param_names:  # prod function
            return optimization_func(
                n=params.get('n', 0),
                a=params.get('a', []),
                c=params.get('c', []),
                u=params.get('u', []),
                b=params.get('b', 0),
                c_penalty=params.get('c_penalty', 0)
            )
        else:  # generic production function
            return optimization_func(
                products=params.get('products', []),
                resources=params.get('resources', []),
                costs=params.get('costs', []),
                capacities=params.get('capacities', []),
                demands=params.get('demands', [])
            )

    def _handle_revenue_problem(self, optimization_func, params):
        """Handle revenue maximization problems."""
        return optimization_func(
            n_packages=params.get('n_packages', 0),
            n_legs=params.get('n_legs', 0),
            c=params.get('c', []),
            r=params.get('r', []),
            r_tilde=params.get('r_tilde', []),
            x_lb=params.get('x_lb', []),
            x_ub=params.get('x_ub', []),
            delta=params.get('delta', []),
            C=params.get('C', 0)
        )
    

    
    def _handle_aircraft_problem(self, optimization_func, params):
        """Handle aircraft optimization problems."""
        return optimization_func(
            n=params.get('n', 0),
            E=params.get('E', []),
            T=params.get('T', []),
            L=params.get('L', []),
            alpha=params.get('alpha', []),
            beta=params.get('beta', []),
            S=params.get('S', [])
        )
    
    def _handle_generic_problem(self, optimization_func, params):
        """Generic handler for problems without specific handlers."""
        try:
            # Try keyword arguments first
            return optimization_func(**params)
        except TypeError:
            try:
                # Try passing as a single parameter
                return optimization_func(params)
            except TypeError:
                # Try passing parameters individually based on function signature
                import inspect
                sig = inspect.signature(optimization_func)
                param_names = list(sig.parameters.keys())
                
                # Create arguments dict with available parameters
                args = {}
                for param_name in param_names:
                    if param_name in params:
                        args[param_name] = params[param_name]
                
                return optimization_func(**args)
    
    def _extract_optimal_solution(self, result, problem_name: str, function_name: str):
        """
        Extract optimal point and objective value from various return formats.
        
        Args:
            result: The result returned by the optimization function
            problem_name: Name of the problem for debugging
            function_name: Name of the function for debugging
            
        Returns:
            tuple: (objective_value, optimal_point)
        """
        if result is None:
            return None, None
            
        # Handle different return formats
        if isinstance(result, (int, float)):
            # Scalar return - only objective value
            return result, None
            
        elif isinstance(result, tuple):
            if len(result) == 2:
                # (objective_value, optimal_point) or (optimal_point, objective_value)
                if isinstance(result[0], (int, float)) and isinstance(result[1], dict):
                    # (objective_value, all_vars_dict)
                    return result[0], result[1]
                elif isinstance(result[1], (int, float)) and isinstance(result[0], (list, dict)):
                    # (optimal_point, objective_value) - old format
                    return result[1], result[0]
                else:
                    # Try to determine which is which
                    if isinstance(result[0], (int, float)):
                        return result[0], result[1]
                    else:
                        return result[1], result[0]
            elif len(result) == 3:
                # (optimal_point, objective_value, all_vars_dict) - new format for some functions
                return result[1], result[2]  # objective_value, all_vars_dict
            else:
                # Unknown tuple format
                return result[0] if isinstance(result[0], (int, float)) else None, result
                
        elif isinstance(result, dict):
            # Dictionary return - assume it contains both
            obj_val = result.get('objective_value', result.get('objVal', None))
            opt_point = result.get('optimal_point', result.get('solution', None))
            return obj_val, opt_point
            
        elif isinstance(result, list):
            # List return - assume it's the optimal point
            return None, result
            
        else:
            # Unknown format
            return None, result
    
    def _get_standardized_result_info(self, result, problem_name: str, function_name: str) -> dict:
        """
        Get standardized information about the optimization result.
        
        Args:
            result: The result from the optimization function
            problem_name: Name of the problem
            function_name: Name of the function that was called
            
        Returns:
            dict: Standardized result information
        """
        optimal_point, objective_value = self._extract_optimal_solution(result, problem_name, function_name)
        
        # Determine what information is available
        has_objective = objective_value is not None
        has_optimal_point = optimal_point is not None
        
        # Create standardized info
        info = {
            'has_objective_value': has_objective,
            'has_optimal_point': has_optimal_point,
            'objective_value': objective_value,
            'optimal_point': optimal_point,
            'result_type': type(result).__name__,
            'raw_result': result
        }
        
        # Add problem-specific information
        if has_optimal_point:
            if isinstance(optimal_point, list):
                info['num_variables'] = len(optimal_point)
                info['variable_values'] = optimal_point
            elif isinstance(optimal_point, dict):
                info['num_variables'] = len(optimal_point)
                info['variable_names'] = list(optimal_point.keys())
                info['variable_values'] = optimal_point
        
        return info
    
    def run_optimization(self, problem_name: str, function_name: Optional[str] = None, verbose: bool = True) -> Dict[str, Any]:
        """
        Run optimization for a given problem.
        
        Args:
            problem_name: Name of the problem to run
            function_name: Specific function to call (if None, will try to auto-detect)
            verbose: Whether to print detailed output
            
        Returns:
            Dictionary containing results and metadata
        """
        if verbose:
            print(f"🚀 Running optimization for problem: {problem_name}")
            print("="*60)
        
        try:
            # Load parameters
            if verbose:
                print("📊 Loading parameters...")
            params = self.load_parameters(problem_name)
            
            if verbose:
                print("Parameters loaded:")
                print(json.dumps(params, indent=2))
                print()
            
            # Load optimization code
            if verbose:
                print("💻 Loading optimization code...")
            module = self.load_optimization_code(problem_name)
            
            # Find optimization functions
            available_functions = self.analyze_code_functions(module)
            
            if verbose:
                print(f"Available functions: {available_functions}")
            
            # Determine which function to call
            if function_name is None:
                if len(available_functions) == 1:
                    function_name = available_functions[0]
                elif len(available_functions) > 1:
                    # Try to find the most likely optimization function
                    for func in available_functions:
                        if any(keyword in func.lower() for keyword in ['optimize', 'solve']):
                            function_name = func
                            break
                    if function_name is None:
                        function_name = available_functions[0]
                else:
                    raise ValueError("No optimization functions found in the code")
            
            if verbose:
                print(f"🎯 Using function: {function_name}")
            
            # Get the optimization function
            optimization_func = getattr(module, function_name)
            
            # Run the optimization
            if verbose:
                print("⚡ Running optimization...")
                print("-" * 40)
            
            # Use problem-specific handler if available
            if problem_name in self.problem_handlers:
                if verbose:
                    print(f"🔧 Using specialized handler for {problem_name}")
                result = self.problem_handlers[problem_name](optimization_func, params)
            else:
                if verbose:
                    print(f"🔧 Using generic handler for {problem_name}")
                result = self._handle_generic_problem(optimization_func, params)
            
            # Extract optimal point and objective value
            optimal_point, objective_value = self._extract_optimal_solution(result, problem_name, function_name)
            
            # Get standardized result information
            result_info = self._get_standardized_result_info(result, problem_name, function_name)
            
            # Prepare results
            results = {
                'success': True,
                'problem_name': problem_name,
                'function_name': function_name,
                'result': result,
                'optimal_point': optimal_point,
                'objective_value': objective_value,
                'result_info': result_info,
                'parameters': params,
                'error': None
            }
            
            if verbose:
                print("-" * 40)
                print("✅ OPTIMIZATION SUCCESSFUL!")
                print(f"Result: {result}")
                if objective_value is not None:
                    print(f"Objective Value: {objective_value}")
                if optimal_point is not None:
                    print(f"Optimal Point: {optimal_point}")
            
            return results
            
        except Exception as e:
            error_info = {
                'success': False,
                'problem_name': problem_name,
                'function_name': function_name if 'function_name' in locals() else None,
                'result': None,
                'parameters': params if 'params' in locals() else None,
                'error': str(e),
                'traceback': traceback.format_exc()
            }
            
            if verbose:
                print("-" * 40)
                print("❌ OPTIMIZATION FAILED!")
                print(f"Error: {e}")
                print("\nFull traceback:")
                print(traceback.format_exc())
            
            return error_info
    
    def save_results_to_json(self, results: Dict[str, Dict], filename: str = "optimization_results.json"):
        """Save results to JSON file with proper handling of tuple keys."""
        # Convert all tuple keys to strings before JSON serialization
        converted_results = convert_tuple_keys_to_strings(results)
        
        json_data = {
            "timestamp": datetime.now().isoformat(),
            "total_problems": len(results),
            "results": converted_results
        }
        
        with open(filename, 'w') as f:
            json.dump(json_data, f, indent=2)
        
        print(f"Results saved to {filename}")
    
    def run_all_problems(self, verbose: bool = True, save_to_json: bool = True) -> Dict[str, Dict]:
        """
        Run optimization for all available problems.
        
        Args:
            verbose: Whether to print detailed output
            save_to_json: Whether to save results to JSON file
            
        Returns:
            Dictionary mapping problem names to their results
        """
        import datetime
        
        problems = self.list_available_problems()
        
        if verbose:
            print(f"🔍 Found {len(problems)} problems to run:")
            for problem in problems:
                print(f"  - {problem}")
            print()
        
        results = {}
        
        for problem in problems:
            if verbose:
                print(f"\n{'='*60}")
                print(f"PROBLEM: {problem}")
                print('='*60)
            
            result = self.run_optimization(problem, verbose=verbose)
            result['timestamp'] = datetime.datetime.now().isoformat()
            results[problem] = result
            
            if verbose:
                print(f"\n{'='*60}")
                print(f"SUMMARY FOR {problem}:")
                print(f"  Success: {result['success']}")
                if result['success']:
                    print(f"  Result: {result['result']}")
                    if result.get('objective_value') is not None:
                        print(f"  Objective Value: {result['objective_value']}")
                    if result.get('optimal_point') is not None:
                        print(f"  Optimal Point: {result['optimal_point']}")
                else:
                    print(f"  Error: {result['error']}")
                print('='*60)
        
        # Save results to JSON if requested
        if save_to_json:
            self.save_results_to_json(results)
        
        # Print final summary
        if verbose:
            print(f"\n{'='*80}")
            print("FINAL SUMMARY")
            print('='*80)
            for problem_name, result in results.items():
                status = "✅" if result['success'] else "❌"
                if result['success']:
                    obj_val = result.get('objective_value', 'N/A')
                    print(f"{status} {problem_name}: {obj_val}")
                else:
                    print(f"{status} {problem_name}: {result.get('error', 'Unknown error')}")
        
        return results

def main():
    """
    Main function to run the optimization framework.
    """
    import argparse
    
    parser = argparse.ArgumentParser(description="Run optimization problems with their parameters")
    parser.add_argument("--problem", "-p", help="Specific problem to run")
    parser.add_argument("--function", "-f", help="Specific function to call")
    parser.add_argument("--all", "-a", action="store_true", help="Run all available problems")
    parser.add_argument("--list", "-l", action="store_true", help="List all available problems")
    parser.add_argument("--quiet", "-q", action="store_true", help="Suppress verbose output")
    parser.add_argument("--json", "-j", help="Save results to JSON file (default: optimization_results.json)")
    parser.add_argument("--no-json", action="store_true", help="Don't save results to JSON file")
    
    args = parser.parse_args()
    
    framework = OptimizationFramework()
    
    if args.list:
        problems = framework.list_available_problems()
        print(f"Available problems ({len(problems)}):")
        for problem in problems:
            print(f"  - {problem}")
        return
    
    if args.all:
        # Determine JSON saving options
        save_to_json = not args.no_json
        json_filename = args.json if args.json else "optimization_results.json"
        
        results = framework.run_all_problems(verbose=not args.quiet, save_to_json=save_to_json)
        
        # If custom JSON filename is specified, save with that name
        if save_to_json and args.json:
            framework.save_results_to_json(results, json_filename)
        
        # Print summary
        print("\n" + "="*80)
        print("FINAL SUMMARY")
        print("="*80)
        
        successful = 0
        failed = 0
        
        for problem, result in results.items():
            if result['success']:
                successful += 1
                obj_val = result.get('objective_value', result.get('result', 'N/A'))
                print(f"✅ {problem}: {obj_val}")
            else:
                failed += 1
                print(f"❌ {problem}: {result['error']}")
        
        print(f"\nTotal: {successful} successful, {failed} failed")
        
    elif args.problem:
        result = framework.run_optimization(args.problem, args.function, verbose=not args.quiet)
        
        if result['success']:
            print(f"\n✅ {args.problem} completed successfully!")
            print(f"Result: {result['result']}")
        else:
            print(f"\n❌ {args.problem} failed!")
            print(f"Error: {result['error']}")
    
    else:
        # Interactive mode
        problems = framework.list_available_problems()
        
        if not problems:
            print("❌ No problems found!")
            return
        
        print("Available problems:")
        for i, problem in enumerate(problems, 1):
            print(f"  {i}. {problem}")
        
        try:
            choice = input(f"\nSelect problem (1-{len(problems)}) or 'all': ").strip()
            
            if choice.lower() == 'all':
                results = framework.run_all_problems()
            else:
                idx = int(choice) - 1
                if 0 <= idx < len(problems):
                    problem = problems[idx]
                    result = framework.run_optimization(problem)
                    
                    if result['success']:
                        print(f"\n✅ {problem} completed successfully!")
                        print(f"Result: {result['result']}")
                    else:
                        print(f"\n❌ {problem} failed!")
                        print(f"Error: {result['error']}")
                else:
                    print("❌ Invalid choice!")
        except (ValueError, KeyboardInterrupt):
            print("\n👋 Goodbye!")

if __name__ == "__main__":
    main() 