#!/usr/bin/env python3
"""
Script for managing and testing pattern reformulation agents.
"""

import argparse
import json
from pathlib import Path
from pattern_reformulation_agents import PatternReformulationAgents
from nonlinear_detector import NonLinearPatternExtractor


def list_agents():
    """List all available reformulation agents."""
    agents = PatternReformulationAgents()
    available_agents = agents.list_available_agents()
    
    print("="*60)
    print("AVAILABLE REFORMULATION AGENTS")
    print("="*60)
    
    for i, agent_name in enumerate(available_agents, 1):
        agent_method = getattr(agents, agent_name)
        doc = agent_method.__doc__.split('\n')[0] if agent_method.__doc__ else "No description"
        print(f"{i}. {agent_name}")
        print(f"   {doc}")
        print()
    
    print(f"Total: {len(available_agents)} agents available")


def test_agent(agent_type: str, latex_model: str, pattern: str, parameters=None):
    """
    Test a specific reformulation agent.
    
    Args:
        agent_type (str): Type of agent to test
        latex_model (str): LaTeX model to reformulate
        pattern (str): Pattern to reformulate
        parameters (dict): Optional parameters
    """
    print(f"🧪 Testing {agent_type} reformulation agent...")
    print("="*60)
    
    try:
        agents = PatternReformulationAgents()
        agent = agents.get_agent(agent_type)
        
        print(f"📝 Input Pattern: {pattern}")
        print(f"📊 Parameters: {len(parameters) if parameters else 0} parameters")
        print()
        
        # Run the agent
        result = agent(latex_model, pattern, parameters=parameters)
        
        print("✅ Agent execution completed successfully!")
        print("📄 Result:")
        print("-" * 40)
        print(result)
        print("-" * 40)
        
        return result
        
    except Exception as e:
        print(f"❌ Agent test failed: {str(e)}")
        return None


def add_custom_agent(agent_name: str, agent_file: str):
    """
    Add a custom agent from a file.
    
    Args:
        agent_name (str): Name for the new agent
        agent_file (str): Path to the file containing the agent function
    """
    print(f"➕ Adding custom agent '{agent_name}' from {agent_file}...")
    
    try:
        # Import the custom agent function
        import importlib.util
        spec = importlib.util.spec_from_file_location("custom_agent", agent_file)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        
        # Look for the agent function in the module
        agent_function = None
        for attr_name in dir(module):
            if attr_name.endswith('_agent') or 'agent' in attr_name.lower():
                attr = getattr(module, attr_name)
                if callable(attr):
                    agent_function = attr
                    break
        
        if agent_function is None:
            print("❌ No agent function found in the file")
            return False
        
        # Add the agent to the PatternReformulationAgents class
        agents = PatternReformulationAgents()
        agents.add_custom_agent(agent_name, agent_function)
        
        print(f"✅ Custom agent '{agent_name}' added successfully!")
        return True
        
    except Exception as e:
        print(f"❌ Failed to add custom agent: {str(e)}")
        return False


def interactive_test():
    """Interactive testing mode."""
    print("🧪 INTERACTIVE AGENT TESTING")
    print("="*60)
    
    # List available agents
    agents = PatternReformulationAgents()
    available_agents = agents.list_available_agents()
    
    print("Available agents:")
    for i, agent_name in enumerate(available_agents, 1):
        print(f"  {i}. {agent_name}")
    
    # Get user input
    try:
        choice = int(input(f"\nSelect agent (1-{len(available_agents)}): ")) - 1
        if choice < 0 or choice >= len(available_agents):
            print("❌ Invalid choice")
            return
        
        agent_type = available_agents[choice].replace('_pattern_reformulation_agent', '')
        
        print(f"\nSelected agent: {agent_type}")
        
        # Get LaTeX model
        print("\nEnter LaTeX model (or 'file:path/to/file.tex' to load from file):")
        latex_input = input().strip()
        
        if latex_input.startswith('file:'):
            file_path = latex_input[5:]
            try:
                with open(file_path, 'r', encoding='utf-8') as f:
                    latex_model = f.read()
                print(f"✅ Loaded LaTeX model from {file_path}")
            except Exception as e:
                print(f"❌ Failed to load file: {str(e)}")
                return
        else:
            latex_model = latex_input
        
        # Get pattern
        print("\nEnter pattern to reformulate:")
        pattern = input().strip()
        
        # Get parameters (optional)
        print("\nEnter parameters as JSON (or press Enter to skip):")
        params_input = input().strip()
        parameters = None
        if params_input:
            try:
                parameters = json.loads(params_input)
                print("✅ Parameters loaded")
            except json.JSONDecodeError:
                print("❌ Invalid JSON format, skipping parameters")
        
        # Run test
        test_agent(agent_type, latex_model, pattern, parameters)
        
    except (ValueError, KeyboardInterrupt):
        print("\n❌ Test cancelled")


def create_agent_template(agent_name: str, output_file: str):
    """
    Create a template for a new custom agent.
    
    Args:
        agent_name (str): Name for the new agent
        output_file (str): Output file path
    """
    template = f'''#!/usr/bin/env python3
"""
Custom reformulation agent: {agent_name}
"""

def {agent_name}_reformulation_agent(self, latex_model: str, pattern: str, problem_id=None, parameters=None):
    """
    Custom reformulation agent for {agent_name} patterns.
    
    Args:
        latex_model: The complete LaTeX model
        pattern: The pattern description
        problem_id: Problem identifier for parameter loading
        parameters: Pre-loaded parameters
    """
    # Use provided parameters or load them if needed
    param_info = ""
    if parameters:
        import json
        param_info = f"""
AVAILABLE CONCRETE PARAMETERS:
{{json.dumps(parameters, indent=2)}}

Use these concrete values for variable bounds and parameter definitions.
"""
    
    prompt = f"""You are a specialized expert in {agent_name} pattern reformulation for mixed-integer programming.

**FOCUS ON THIS PATTERN:**
{{pattern}}

**YOUR TASK:**
1. Identify the general {agent_name} pattern and find ALL instances in the model
2. Choose the most appropriate reformulation technique for this specific pattern
3. Determine suitable variable bounds and auxiliary variables needed
4. Create the necessary linear constraints to replace ALL instances of this pattern
5. Ensure the reformulation maintains mathematical equivalence
6. **CRITICAL**: Preserve ALL existing constraints and reformulations from the input model

**AVAILABLE TECHNIQUES:**
[Describe the specific techniques for {agent_name} patterns]

**OTHER CONSIDERATIONS:**
- Process all instances of the pattern together for consistency
- Use systematic naming for auxiliary variables and constraints
- Use the MIP reformulation technique suitable for the variable type
- If you use big-M constants, define them first in the model

{{param_info}}

**ORIGINAL LATEX MODEL:**
{{latex_model}}

**PATTERN TO REFORMULATE:**
{{pattern}}

Provide your reformulation in this format:

{agent_name.upper()} PATTERN REFORMULATION:

Pattern Identified: [general pattern description]
All Instances Found: [list all specific instances found in the model]
Technique Selected: [chosen linearization technique and reasoning]

Auxiliary Variables Created:
[List all auxiliary variables with clear naming]

Reformulation Constraints:
[All constraints needed for this pattern reformulation]

UPDATED MODEL:
[Complete LaTeX model with ALL instances of this pattern replaced by auxiliary variables]
[IMPORTANT: Include ALL existing constraints from the input model, preserving any previous reformulations]
    """
    
    response = self.llm.invoke(prompt)
    return response.content
'''
    
    try:
        with open(output_file, 'w', encoding='utf-8') as f:
            f.write(template)
        print(f"✅ Agent template created: {output_file}")
        print(f"📝 Edit the template and use 'add-custom-agent' to add it")
    except Exception as e:
        print(f"❌ Failed to create template: {str(e)}")


def main():
    """Main function for the agent management script."""
    parser = argparse.ArgumentParser(description='Manage and test pattern reformulation agents')
    parser.add_argument('command', choices=['list', 'test', 'add-custom', 'interactive', 'create-template'],
                       help='Command to execute')
    parser.add_argument('--agent-type', type=str, help='Type of agent (for test command)')
    parser.add_argument('--latex-model', type=str, help='LaTeX model (for test command)')
    parser.add_argument('--pattern', type=str, help='Pattern to reformulate (for test command)')
    parser.add_argument('--parameters', type=str, help='Parameters as JSON string (for test command)')
    parser.add_argument('--agent-name', type=str, help='Name for custom agent')
    parser.add_argument('--agent-file', type=str, help='File containing custom agent')
    parser.add_argument('--output-file', type=str, help='Output file for template')
    
    args = parser.parse_args()
    
    if args.command == 'list':
        list_agents()
    
    elif args.command == 'test':
        if not all([args.agent_type, args.latex_model, args.pattern]):
            print("❌ Test command requires --agent-type, --latex-model, and --pattern")
            return
        
        parameters = None
        if args.parameters:
            try:
                parameters = json.loads(args.parameters)
            except json.JSONDecodeError:
                print("❌ Invalid JSON format for parameters")
                return
        
        test_agent(args.agent_type, args.latex_model, args.pattern, parameters)
    
    elif args.command == 'add-custom':
        if not all([args.agent_name, args.agent_file]):
            print("❌ Add-custom command requires --agent-name and --agent-file")
            return
        
        add_custom_agent(args.agent_name, args.agent_file)
    
    elif args.command == 'interactive':
        interactive_test()
    
    elif args.command == 'create-template':
        if not all([args.agent_name, args.output_file]):
            print("❌ Create-template command requires --agent-name and --output-file")
            return
        
        create_agent_template(args.agent_name, args.output_file)


if __name__ == "__main__":
    main() 