#!/usr/bin/env python3
"""
Optimization Executor for running Gurobi optimization code.
"""

from gurobipy import *
import numpy as np
import traceback
import sys
import re
import math
from datetime import datetime
from pathlib import Path


class OptimizationExecutor:
    """
    Class responsible for executing Gurobi optimization code and handling results.
    """
    
    def __init__(self, save_error_log=True):
        """
        Initialize the optimization executor.
        
        Args:
            save_error_log (bool): Whether to save error logs to files
        """
        self.save_error_log = save_error_log
    
    def _detect_monotone_transformation(self, gurobi_code: str):
        """
        Detect if the code contains monotone transformation and extract post-processing instructions.
        
        Args:
            gurobi_code (str): The generated Gurobi code
            
        Returns:
            dict: Information about monotone transformation if detected, None otherwise
        """
        # Check for monotone transformation markers
        if 'MONOTONE_TRANSFORMATION' not in gurobi_code:
            return None
        
        # Extract post-processing instructions
        post_processing_info = {}
        
        # Look for post-processing patterns in the code
        lines = gurobi_code.split('\n')
        
        # Pattern 1: Look for post-processing comments in the code
        for i, line in enumerate(lines):
            if 'POST-PROCESSING' in line or 'post-processing' in line.lower():
                # Extract the next few lines for context
                context_lines = []
                for j in range(i, min(i + 10, len(lines))):
                    context_lines.append(lines[j])
                    if 'print(' in lines[j] or 'original_objective_value' in lines[j]:
                        break
                
                post_processing_info['context'] = '\n'.join(context_lines)
                break
        
        # Pattern 2: Look for specific transformation types
        transformation_types = []
        
        # Check for common monotone transformations
        if 'log(' in gurobi_code and 'exp(' in gurobi_code:
            transformation_types.append('log_exp')
        if 'sqrt(' in gurobi_code and '** 0.5' in gurobi_code:
            transformation_types.append('sqrt_power')
        if 'exp(' in gurobi_code and 'log(' in gurobi_code:
            transformation_types.append('exp_log')
        
        # Pattern 3: Extract the actual transformation from comments or code
        original_objective_pattern = r'Original objective:\s*(.+)'
        original_match = re.search(original_objective_pattern, gurobi_code)
        if original_match:
            post_processing_info['original_objective'] = original_match.group(1).strip()
        
        # Pattern 4: Look for the actual reverse mapping code
        reverse_mapping_patterns = [
            r'original_objective_value\s*=\s*([^#\n]+)',
            r'math\.exp\(([^)]+)\)',
            r'math\.log\(([^)]+)\)',
            r'\*\* 0\.5',
            r'\*\* 2',
        ]
        
        reverse_mapping_code = None
        for pattern in reverse_mapping_patterns:
            match = re.search(pattern, gurobi_code)
            if match:
                reverse_mapping_code = match.group(0)
                break
        
        # Pattern 5: Look for specific transformation patterns in comments
        if 'log(sum(' in gurobi_code or 'log(' in gurobi_code:
            transformation_types.append('log_transformation')
        if 'sqrt(' in gurobi_code:
            transformation_types.append('sqrt_transformation')
        if 'exp(' in gurobi_code:
            transformation_types.append('exp_transformation')
        
        # Pattern 6: Look for specific reverse mapping patterns in the code
        if 'math.exp(' in gurobi_code:
            post_processing_info['reverse_mapping_type'] = 'exp'
        elif 'math.log(' in gurobi_code:
            post_processing_info['reverse_mapping_type'] = 'log'
        elif '** 0.5' in gurobi_code:
            post_processing_info['reverse_mapping_type'] = 'sqrt'
        elif '** 2' in gurobi_code:
            post_processing_info['reverse_mapping_type'] = 'square'
        
        # Pattern 7: Extract the actual reverse mapping line
        for i, line in enumerate(lines):
            if 'original_objective_value' in line and '=' in line:
                post_processing_info['reverse_mapping_line'] = line.strip()
                break
        
        if reverse_mapping_code:
            post_processing_info['reverse_mapping_code'] = reverse_mapping_code
        
        if transformation_types or post_processing_info:
            return {
                'detected': True,
                'transformation_types': transformation_types,
                'post_processing_info': post_processing_info
            }
        
        return None
    
    def _apply_reverse_mapping(self, objective_value: float, transformation_info: dict, variables: dict):
        """
        Apply reverse mapping to recover the original objective value.
        
        Args:
            objective_value (float): The transformed objective value from the solver
            transformation_info (dict): Information about the monotone transformation
            variables (dict): Dictionary of variable values from the solution
            
        Returns:
            float: The original objective value after reverse mapping
        """
        if not transformation_info or not transformation_info.get('detected'):
            return objective_value
        
        # Extract transformation type and apply appropriate reverse mapping
        transformation_types = transformation_info.get('transformation_types', [])
        post_processing_info = transformation_info.get('post_processing_info', {})
        
        # Method 1: Use the reverse mapping code if available
        reverse_mapping_code = post_processing_info.get('reverse_mapping_code')
        if reverse_mapping_code:
            print(f"   🔧 Attempting reverse mapping using code: {reverse_mapping_code}")
            try:
                # Create a safe environment for evaluation
                safe_dict = {
                    'math': math,
                    'model': type('Model', (), {'objVal': objective_value})(),
                    'z_star': objective_value,
                    'objVal': objective_value
                }
                # Add variables to the safe environment
                for var_name, var_value in variables.items():
                    safe_dict[var_name] = var_value
                
                # Extract the expression from the reverse mapping code
                if '=' in reverse_mapping_code:
                    expression = reverse_mapping_code.split('=')[1].strip()
                else:
                    expression = reverse_mapping_code
                
                # Clean up the expression
                expression = expression.replace('math.', '')
                expression = expression.replace('objVal', str(objective_value))
                expression = expression.replace('z_star', str(objective_value))
                
                print(f"   🔧 Cleaned expression: {expression}")
                
                # Apply common transformations
                if 'exp(' in expression:
                    # Handle exponential transformation
                    inner_expr = expression.replace('exp(', '').replace(')', '')
                    try:
                        inner_value = eval(inner_expr, {"__builtins__": {}}, safe_dict)
                        result = math.exp(inner_value)
                        print(f"   ✅ Applied exp transformation: {result}")
                        return result
                    except:
                        pass
                
                if 'log(' in expression:
                    # Handle logarithmic transformation
                    inner_expr = expression.replace('log(', '').replace(')', '')
                    try:
                        inner_value = eval(inner_expr, {"__builtins__": {}}, safe_dict)
                        result = math.log(inner_value)
                        print(f"   ✅ Applied log transformation: {result}")
                        return result
                    except:
                        pass
                
                if '** 0.5' in expression:
                    # Handle square root transformation
                    base_expr = expression.replace('** 0.5', '')
                    try:
                        base_value = eval(base_expr, {"__builtins__": {}}, safe_dict)
                        result = math.sqrt(base_value)
                        print(f"   ✅ Applied sqrt transformation: {result}")
                        return result
                    except:
                        pass
                
                if '** 2' in expression:
                    # Handle square transformation
                    base_expr = expression.replace('** 2', '')
                    try:
                        base_value = eval(base_expr, {"__builtins__": {}}, safe_dict)
                        result = base_value ** 2
                        print(f"   ✅ Applied square transformation: {result}")
                        return result
                    except:
                        pass
                
            except Exception as e:
                print(f"⚠️ Warning: Failed to apply reverse mapping using code: {e}")
        
        # Method 2: Apply based on transformation types
        for trans_type in transformation_types:
            if trans_type == 'log_exp':
                result = math.exp(objective_value)
                print(f"   ✅ Applied log_exp transformation: {result}")
                return result
            elif trans_type == 'sqrt_power':
                result = math.sqrt(objective_value)
                print(f"   ✅ Applied sqrt_power transformation: {result}")
                return result
            elif trans_type == 'exp_log':
                result = math.log(objective_value)
                print(f"   ✅ Applied exp_log transformation: {result}")
                return result
            elif trans_type == 'log_transformation':
                result = math.exp(objective_value)
                print(f"   ✅ Applied log_transformation: {result}")
                return result
            elif trans_type == 'sqrt_transformation':
                result = math.sqrt(objective_value)
                print(f"   ✅ Applied sqrt_transformation: {result}")
                return result
            elif trans_type == 'exp_transformation':
                result = math.log(objective_value)
                print(f"   ✅ Applied exp_transformation: {result}")
                return result
        
        # Method 3: Apply based on reverse mapping type
        reverse_mapping_type = post_processing_info.get('reverse_mapping_type')
        if reverse_mapping_type == 'exp':
            result = math.exp(objective_value)
            print(f"   ✅ Applied exp reverse mapping: {result}")
            return result
        elif reverse_mapping_type == 'log':
            result = math.log(objective_value)
            print(f"   ✅ Applied log reverse mapping: {result}")
            return result
        elif reverse_mapping_type == 'sqrt':
            result = math.sqrt(objective_value)
            print(f"   ✅ Applied sqrt reverse mapping: {result}")
            return result
        elif reverse_mapping_type == 'square':
            result = objective_value ** 2
            print(f"   ✅ Applied square reverse mapping: {result}")
            return result
        
        # Method 4: Try to infer from the original objective description
        original_objective = post_processing_info.get('original_objective', '')
        if 'log(' in original_objective:
            result = math.exp(objective_value)
            print(f"   ✅ Applied log->exp inference: {result}")
            return result
        elif 'sqrt(' in original_objective:
            result = math.sqrt(objective_value)
            print(f"   ✅ Applied sqrt inference: {result}")
            return result
        elif 'exp(' in original_objective:
            result = math.log(objective_value)
            print(f"   ✅ Applied exp->log inference: {result}")
            return result
        
        # If no reverse mapping could be applied, return the original value
        print(f"⚠️ Warning: Could not determine reverse mapping for monotone transformation")
        return objective_value
    
    def execute(self, gurobi_code: str, problem_id=None):
        """
        Executes the gurobipy optimization code.
        
        Args:
            gurobi_code (str): Gurobi Python code as string
            problem_id (str): Problem identifier for logging
            
        Returns:
            dict: Dictionary with optimization results
        """
        def save_error_to_file(error_info, gurobi_code, problem_id=None):
            """Save detailed error information to a log file."""
            if not self.save_error_log:
                return
                
            try:
                # Create error log directory
                error_log_dir = Path("data/error_logs")
                error_log_dir.mkdir(parents=True, exist_ok=True)
                
                # Create timestamped error log file
                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                if problem_id:
                    log_filename = f"error_problem_{problem_id}_{timestamp}.log"
                else:
                    log_filename = f"error_{timestamp}.log"
                
                log_path = error_log_dir / log_filename
                
                with open(log_path, 'w', encoding='utf-8') as f:
                    f.write("="*80 + "\n")
                    f.write("GUROBI CODE EXECUTION ERROR LOG\n")
                    f.write("="*80 + "\n")
                    f.write(f"Timestamp: {datetime.now().isoformat()}\n")
                    f.write(f"Problem ID: {problem_id or 'Unknown'}\n")
                    f.write(f"Error Type: {error_info.get('error_type', 'Unknown')}\n")
                    f.write(f"Error Message: {error_info.get('error', 'Unknown')}\n")
                    f.write("\n" + "="*80 + "\n")
                    f.write("FULL ERROR DETAILS:\n")
                    f.write("="*80 + "\n")
                    f.write(error_info.get('full_traceback', 'No traceback available') + "\n")
                    
                    if 'line_number' in error_info:
                        f.write(f"\nError Line Number: {error_info['line_number']}\n")
                        f.write(f"Error Text: {error_info.get('error_text', 'N/A')}\n")
                    
                    f.write("\n" + "="*80 + "\n")
                    f.write("GENERATED GUROBI CODE:\n")
                    f.write("="*80 + "\n")
                    f.write(gurobi_code + "\n")
                    
                    f.write("\n" + "="*80 + "\n")
                    f.write("CODE ANALYSIS:\n")
                    f.write("="*80 + "\n")
                    
                    # Analyze the code for common issues
                    lines = gurobi_code.split('\n')
                    f.write(f"Total lines: {len(lines)}\n")
                    f.write("Line-by-line analysis:\n")
                    for i, line in enumerate(lines, 1):
                        f.write(f"{i:3d}: {line}\n")
                        # Check for common issues
                        if 'SET1' in line or 'SET2' in line:
                            f.write(f"     ⚠️  WARNING: Abstract set name found on line {i}\n")
                        if ' i ' in line and 'for i in' not in line:
                            f.write(f"     ⚠️  WARNING: Variable 'i' used without proper loop context on line {i}\n")
                        if 'quicksum(' in line and ('SET' in line or ' i ' in line):
                            f.write(f"     ⚠️  WARNING: Potential undefined variable in quicksum on line {i}\n")
                    
                    f.write("\n" + "="*80 + "\n")
                    f.write("SUGGESTIONS:\n")
                    f.write("="*80 + "\n")
                    f.write(error_info.get('suggestion', 'No specific suggestions available') + "\n")
                    f.write("="*80 + "\n")
                
                print(f"📝 Detailed error log saved to: {log_path}")
                return str(log_path)
                
            except Exception as log_error:
                print(f"⚠️ Failed to save error log: {str(log_error)}")
                return None
        
        try:
            # Create a local namespace for execution
            local_namespace = {
                'Model': Model,
                'GRB': GRB,
                'quicksum': quicksum,
                'np': np,
                '__builtins__': __builtins__
            }
            
            # Add all gurobipy imports to namespace
            for name in dir():
                if not name.startswith('_'):
                    try:
                        obj = eval(name)
                        if hasattr(obj, '__module__') and obj.__module__ == 'gurobipy':
                            local_namespace[name] = obj
                    except:
                        pass
            
            # First, try to compile the code to catch syntax errors
            try:
                compiled_code = compile(gurobi_code, '<generated_code>', 'exec')
            except SyntaxError as syntax_error:
                error_info = {
                    'success': False,
                    'error': 'Python syntax error in generated code',
                    'error_type': 'SYNTAX_ERROR',
                    'details': str(syntax_error),
                    'line_number': syntax_error.lineno,
                    'error_text': syntax_error.text,
                    'full_traceback': traceback.format_exc(),
                    'suggestion': 'The Code Converter generated invalid Python syntax. This needs to be fixed in the code generation step.'
                }
                save_error_to_file(error_info, gurobi_code, problem_id)
                return error_info
            except Exception as compile_error:
                error_info = {
                    'success': False,
                    'error': 'Python compilation error in generated code',
                    'error_type': 'COMPILATION_ERROR',
                    'details': str(compile_error),
                    'full_traceback': traceback.format_exc(),
                    'suggestion': 'The generated code has compilation issues. Check for undefined variables, imports, or syntax problems.'
                }
                save_error_to_file(error_info, gurobi_code, problem_id)
                return error_info
            
            # Execute the compiled code
            try:
                exec(compiled_code, local_namespace)
            except NameError as name_error:
                error_info = {
                    'success': False,
                    'error': 'Python runtime error: undefined variable or name',
                    'error_type': 'NAME_ERROR',
                    'details': str(name_error),
                    'full_traceback': traceback.format_exc(),
                    'suggestion': 'The generated code uses undefined variables. This typically happens when loop variables or parameters are not properly defined.'
                }
                save_error_to_file(error_info, gurobi_code, problem_id)
                return error_info
            except ImportError as import_error:
                error_info = {
                    'success': False,
                    'error': 'Python import error',
                    'error_type': 'IMPORT_ERROR',
                    'details': str(import_error),
                    'full_traceback': traceback.format_exc(),
                    'suggestion': 'Missing required imports in the generated code.'
                }
                save_error_to_file(error_info, gurobi_code, problem_id)
                return error_info
            except Exception as runtime_error:
                error_info = {
                    'success': False,
                    'error': 'Python runtime error during code execution',
                    'error_type': 'RUNTIME_ERROR',
                    'details': str(runtime_error),
                    'full_traceback': traceback.format_exc(),
                    'suggestion': 'The generated code has runtime issues before reaching Gurobi optimization.'
                }
                save_error_to_file(error_info, gurobi_code, problem_id)
                return error_info
            
            # The code should have created a 'model' variable
            model = None

            # Try to find the model variable (could be named 'model', 'm', etc.)
            for var_name, var_value in local_namespace.items():
                if hasattr(var_value, 'Status') and hasattr(var_value, 'optimize'):
                    # This looks like a Gurobi model
                    model = var_value
                    break

            # Check if model was solved successfully
            model_type = 'unknown'
            if model is not None:
                try:
                    is_mip = model.getAttr(GRB.Attr.IsMIP)
                    is_qp = model.getAttr(GRB.Attr.IsQP)
                    is_qcp = model.getAttr(GRB.Attr.IsQCP)
                    if is_qcp:
                        model_type = 'MIQCP' if is_mip else 'QCP'
                    elif is_qp:
                        model_type = 'MIQP' if is_mip else 'QP'
                    else:
                        model_type = 'MILP' if is_mip else 'LP'
                except Exception:
                    model_type = 'unknown'

            if model is not None and model.Status == GRB.OPTIMAL:
                # Collect variable values
                variables = {}
                for var in model.getVars():
                    variables[var.VarName] = var.X
                
                # Detect monotone transformation
                transformation_info = self._detect_monotone_transformation(gurobi_code)
                
                # Apply reverse mapping if transformation is detected
                if transformation_info and transformation_info.get('detected'):
                    objective_value = model.ObjVal
                    print(f"🔄 Detected monotone transformation: {transformation_info.get('transformation_types', [])}")
                    print(f"   Transformed objective value: {objective_value}")
                    original_objective_value = self._apply_reverse_mapping(objective_value, transformation_info, variables)
                    print(f"   Original objective value: {original_objective_value}")
                    results = {
                        'status': 'optimal',
                        'status_code': model.Status,
                        'objective_value': original_objective_value,
                        'transformed_objective_value': objective_value,
                        'variables': variables,
                        'solve_time': model.Runtime,
                        'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                        'model_type': model_type,
                        'monotone_transformation_applied': True,
                        'transformation_info': transformation_info
                    }
                else:
                    results = {
                        'status': 'optimal',
                        'status_code': model.Status,
                        'objective_value': model.ObjVal,
                        'variables': variables,
                        'solve_time': model.Runtime,
                        'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                        'model_type': model_type,
                        'monotone_transformation_applied': False
                    }
            elif model is not None and model.Status == GRB.INFEASIBLE:
                # Compute IIS for infeasible models
                iis_info = None
                try:
                    print("🔍 Model is infeasible. Computing IIS (Irreducible Inconsistent Subsystem)...")
                    model.computeIIS()
                    
                    # Extract IIS information
                    iis_constraints = []
                    iis_variables = []
                    
                    # Get IIS constraints
                    for constr in model.getConstrs():
                        if constr.IISConstr:
                            iis_constraints.append({
                                'name': constr.ConstrName,
                                'sense': constr.Sense,
                                'rhs': constr.RHS
                            })
                    
                    # Get IIS variable bounds
                    for var in model.getVars():
                        if var.IISLB:
                            iis_variables.append({
                                'name': var.VarName,
                                'bound_type': 'lower_bound',
                                'bound_value': var.LB
                            })
                        if var.IISUB:
                            iis_variables.append({
                                'name': var.VarName,
                                'bound_type': 'upper_bound',
                                'bound_value': var.UB
                            })
                    
                    iis_info = {
                        'constraints': iis_constraints,
                        'variables': iis_variables,
                        'total_iis_constraints': len(iis_constraints),
                        'total_iis_variables': len(iis_variables)
                    }
                    
                    print(f"📊 IIS computed: {len(iis_constraints)} constraints, {len(iis_variables)} variable bounds")
                    
                except Exception as iis_error:
                    print(f"⚠️ Failed to compute IIS: {str(iis_error)}")
                    iis_info = {'error': f'Failed to compute IIS: {str(iis_error)}'}
                
                results = {
                    'status': 'infeasible',
                    'status_code': model.Status,
                    'objective_value': None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': None,
                    'iis': iis_info,
                    'error_details': 'Model is infeasible - no feasible solution exists'
                }
            elif model is not None and model.Status == GRB.UNBOUNDED:
                results = {
                    'status': 'unbounded',
                    'status_code': model.Status,
                    'objective_value': None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': None,
                    'error_details': 'Model is unbounded - objective can be improved without limit'
                }
            elif model is not None and model.Status == GRB.INF_OR_UNBD:
                results = {
                    'status': 'infeasible_or_unbounded',
                    'status_code': model.Status,
                    'objective_value': None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': None,
                    'error_details': 'Model is either infeasible or unbounded'
                }
            elif model is not None and model.Status == GRB.TIME_LIMIT:
                # Collect best solution found if available
                variables = {}
                if model.SolCount > 0:
                    for var in model.getVars():
                        variables[var.VarName] = var.X
                
                # Detect monotone transformation
                transformation_info = self._detect_monotone_transformation(gurobi_code)
                
                # Apply reverse mapping if transformation is detected
                if transformation_info and transformation_info.get('detected'):
                    objective_value = model.ObjVal
                    original_objective_value = self._apply_reverse_mapping(objective_value, transformation_info, variables)
                    results = {
                        'status': 'time_limit',
                        'status_code': model.Status,
                        'objective_value': original_objective_value,
                        'transformed_objective_value': objective_value,
                        'variables': variables,
                        'solve_time': model.Runtime,
                        'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                        'error_details': f'Time limit reached. Found {model.SolCount} solutions.',
                        'monotone_transformation_applied': True,
                        'transformation_info': transformation_info
                    }
                else:
                    results = {
                        'status': 'time_limit',
                        'status_code': model.Status,
                        'objective_value': model.ObjVal if model.SolCount > 0 else None,
                        'variables': variables,
                        'solve_time': model.Runtime,
                        'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                        'error_details': f'Time limit reached. Found {model.SolCount} solutions.',
                        'monotone_transformation_applied': False
                    }
            elif model is not None and model.Status == GRB.ITERATION_LIMIT:
                results = {
                    'status': 'iteration_limit',
                    'status_code': model.Status,
                    'objective_value': model.ObjVal if hasattr(model, 'ObjVal') else None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': None,
                    'error_details': 'Iteration limit reached'
                }
            elif model is not None and model.Status == GRB.NODE_LIMIT:
                results = {
                    'status': 'node_limit',
                    'status_code': model.Status,
                    'objective_value': model.ObjVal if hasattr(model, 'ObjVal') else None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                    'error_details': 'Node limit reached in branch-and-bound'
                }
            elif model is not None and model.Status == GRB.SOLUTION_LIMIT:
                # Collect best solution found
                variables = {}
                for var in model.getVars():
                    variables[var.VarName] = var.X
                
                # Detect monotone transformation
                transformation_info = self._detect_monotone_transformation(gurobi_code)
                
                # Apply reverse mapping if transformation is detected
                if transformation_info and transformation_info.get('detected'):
                    objective_value = model.ObjVal
                    original_objective_value = self._apply_reverse_mapping(objective_value, transformation_info, variables)
                    results = {
                        'status': 'solution_limit',
                        'status_code': model.Status,
                        'objective_value': original_objective_value,
                        'transformed_objective_value': objective_value,
                        'variables': variables,
                        'solve_time': model.Runtime,
                        'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                        'error_details': f'Solution limit reached. Found {model.SolCount} solutions.',
                        'monotone_transformation_applied': True,
                        'transformation_info': transformation_info
                    }
                else:
                    results = {
                        'status': 'solution_limit',
                        'status_code': model.Status,
                        'objective_value': model.ObjVal,
                        'variables': variables,
                        'solve_time': model.Runtime,
                        'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                        'error_details': f'Solution limit reached. Found {model.SolCount} solutions.',
                        'monotone_transformation_applied': False
                    }
            elif model is not None and model.Status == GRB.NUMERIC:
                results = {
                    'status': 'numeric_issues',
                    'status_code': model.Status,
                    'objective_value': None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': None,
                    'error_details': 'Numerical issues encountered during optimization'
                }
            else:
                # Handle other status codes
                status_names = {
                    GRB.LOADED: 'loaded',
                    GRB.CUTOFF: 'cutoff',
                    GRB.SUBOPTIMAL: 'suboptimal',
                    GRB.INPROGRESS: 'in_progress',
                    GRB.USER_OBJ_LIMIT: 'user_obj_limit'
                }
                
                status_name = status_names.get(model.Status, 'unknown')
                
                results = {
                    'status': status_name,
                    'status_code': model.Status,
                    'objective_value': model.ObjVal if hasattr(model, 'ObjVal') else None,
                    'variables': {},
                    'solve_time': model.Runtime,
                    'mip_gap': model.MIPGap if hasattr(model, 'MIPGap') else None,
                    'error_details': f'Optimization ended with status: {status_name} (code: {model.Status})'
                }
            
            # Create empty error log file for successful runs
            if self.save_error_log:
                try:
                    error_log_dir = Path("data/error_logs")
                    error_log_dir.mkdir(parents=True, exist_ok=True)
                    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                    if problem_id:
                        log_filename = f"success_problem_{problem_id}_{timestamp}.log"
                    else:
                        log_filename = f"success_{timestamp}.log"
                    log_path = error_log_dir / log_filename
                    
                    with open(log_path, 'w', encoding='utf-8') as f:
                        f.write("="*80 + "\n")
                        f.write("GUROBI CODE EXECUTION SUCCESS LOG\n")
                        f.write("="*80 + "\n")
                        f.write(f"Timestamp: {datetime.now().isoformat()}\n")
                        f.write(f"Problem ID: {problem_id or 'Unknown'}\n")
                        f.write(f"Status: SUCCESS - No errors occurred\n")
                        f.write(f"Optimization Status: {results['status']}\n")
                        if results.get('objective_value') is not None:
                            f.write(f"Objective Value: {results['objective_value']}\n")
                        f.write(f"Variables Solved: {len(results.get('variables', {}))}\n")
                        f.write(f"Solve Time: {results.get('solve_time', 'N/A')} seconds\n")
                        f.write("\n" + "="*80 + "\n")
                        f.write("GENERATED GUROBI CODE:\n")
                        f.write("="*80 + "\n")
                        f.write(gurobi_code + "\n")
                        f.write("="*80 + "\n")
                    
                    print(f"✅ Success log saved to: {log_path}")
                except Exception as log_error:
                    print(f"⚠️ Failed to save success log: {str(log_error)}")
            
            return {
                'success': True,
                'optimization_results': results
            }
            
        except Exception as e:
            error_info = {
                'success': False,
                'error': 'Unexpected error during optimization execution',
                'error_type': 'UNEXPECTED_ERROR',
                'details': str(e),
                'python_error_type': type(e).__name__,
                'full_traceback': traceback.format_exc(),
                'suggestion': 'This is an unexpected error that occurred during the optimization process.'
            }
            save_error_to_file(error_info, gurobi_code, problem_id)
            return error_info 