#!/usr/bin/env python3
"""
Script to browse and analyze saved results from the Sequential Optimization Workflow.
"""

import os
import json
import argparse
from pathlib import Path
from datetime import datetime

def list_problem_results(results_dir="data/results"):
    """
    List all problem results in the results directory.
    """
    results_path = Path(results_dir)
    
    if not results_path.exists():
        print(f"❌ Results directory not found: {results_path}")
        return
    
    print(f"📁 Results Directory: {results_path.absolute()}")
    print("="*80)
    
    problem_dirs = [d for d in results_path.iterdir() if d.is_dir() and d.name.startswith('problem_')]
    
    if not problem_dirs:
        print("No problem results found.")
        return
    
    problem_dirs.sort(key=lambda x: x.name)
    
    for problem_dir in problem_dirs:
        problem_id = problem_dir.name.replace('problem_', '')
        print(f"\n🔍 Problem {problem_id}:")
        
        run_dirs = [d for d in problem_dir.iterdir() if d.is_dir() and d.name.startswith('run_')]
        run_dirs.sort(key=lambda x: x.name, reverse=True)  # Most recent first
        
        for i, run_dir in enumerate(run_dirs):
            timestamp = run_dir.name.replace('run_', '')
            
            # Try to load summary
            summary_file = run_dir / "results_summary.json"
            if summary_file.exists():
                try:
                    with open(summary_file, 'r', encoding='utf-8') as f:
                        summary = json.load(f)
                    
                    success = summary['metadata']['success']
                    total_steps = summary['metadata']['total_steps']
                    status_icon = "✅" if success else "❌"
                    
                    # Get optimization results if available
                    opt_info = ""
                    if summary.get('optimization_results', {}).get('success'):
                        opt_results = summary['optimization_results']['optimization_results']
                        status = opt_results.get('status', 'unknown')
                        obj_val = opt_results.get('objective_value')
                        if obj_val is not None:
                            opt_info = f" | {status.upper()} (obj: {obj_val:.4f})"
                        else:
                            opt_info = f" | {status.upper()}"
                    
                    print(f"   {status_icon} Run {timestamp} | {total_steps} steps{opt_info}")
                    
                except Exception as e:
                    print(f"   ⚠️ Run {timestamp} | Error reading summary: {e}")
            else:
                print(f"   📁 Run {timestamp} | No summary available")

def show_problem_details(problem_id, run_timestamp=None, results_dir="data/results"):
    """
    Show detailed information about a specific problem run.
    """
    results_path = Path(results_dir)
    problem_dir = results_path / f"problem_{problem_id}"
    
    if not problem_dir.exists():
        print(f"❌ Problem {problem_id} not found in {results_path}")
        return
    
    # Find the run directory
    if run_timestamp:
        run_dir = problem_dir / f"run_{run_timestamp}"
        if not run_dir.exists():
            print(f"❌ Run {run_timestamp} not found for problem {problem_id}")
            return
    else:
        # Use the most recent run
        run_dirs = [d for d in problem_dir.iterdir() if d.is_dir() and d.name.startswith('run_')]
        if not run_dirs:
            print(f"❌ No runs found for problem {problem_id}")
            return
        run_dir = sorted(run_dirs, key=lambda x: x.name)[-1]
        run_timestamp = run_dir.name.replace('run_', '')
    
    print(f"📊 Problem {problem_id} - Run {run_timestamp}")
    print("="*80)
    
    # Load and display summary
    summary_file = run_dir / "results_summary.json"
    if summary_file.exists():
        try:
            with open(summary_file, 'r', encoding='utf-8') as f:
                summary = json.load(f)
            
            print(f"📋 Workflow Summary:")
            print(f"   Success: {'✅ Yes' if summary['metadata']['success'] else '❌ No'}")
            print(f"   Total Steps: {summary['metadata']['total_steps']}")
            print(f"   Workflow Version: {summary['metadata']['workflow_version']}")
            
            # Show optimization results
            if summary.get('optimization_results'):
                opt_results = summary['optimization_results']
                print(f"\n🚀 Optimization Results:")
                if opt_results.get('success'):
                    opt_data = opt_results['optimization_results']
                    print(f"   Status: {opt_data.get('status', 'unknown').upper()}")
                    if opt_data.get('objective_value') is not None:
                        print(f"   Objective Value: {opt_data['objective_value']}")
                    print(f"   Variables: {len(opt_data.get('variables', {}))}")
                    print(f"   Solve Time: {opt_data.get('solve_time', 'N/A')} seconds")
                    if opt_data.get('mip_gap') is not None:
                        print(f"   MIP Gap: {opt_data['mip_gap']}")
                    
                    # Show error details if present (for non-optimal solutions)
                    if 'error_details' in opt_data:
                        print(f"   Details: {opt_data['error_details']}")
                    
                    # Show IIS information for infeasible models
                    if opt_data.get('status') == 'infeasible' and 'iis' in opt_data:
                        iis = opt_data['iis']
                        if 'error' not in iis:
                            print(f"\n🔍 IIS (Irreducible Inconsistent Subsystem):")
                            print(f"   Conflicting Constraints: {iis['total_iis_constraints']}")
                            print(f"   Conflicting Variable Bounds: {iis['total_iis_variables']}")
                            
                            if iis['constraints']:
                                print(f"   Top Conflicting Constraints:")
                                for i, constr in enumerate(iis['constraints'][:3], 1):
                                    print(f"     {i}. {constr['name']}: {constr['sense']} {constr['rhs']}")
                                if len(iis['constraints']) > 3:
                                    print(f"     ... and {len(iis['constraints']) - 3} more")
                            
                            if iis['variables']:
                                print(f"   Top Conflicting Variable Bounds:")
                                for i, var in enumerate(iis['variables'][:3], 1):
                                    print(f"     {i}. {var['name']} {var['bound_type']}: {var['bound_value']}")
                                if len(iis['variables']) > 3:
                                    print(f"     ... and {len(iis['variables']) - 3} more")
                        else:
                            print(f"   ⚠️ IIS computation failed: {iis['error']}")
                else:
                    print(f"   ❌ Failed: {opt_results.get('error', 'Unknown error')}")
                    if 'details' in opt_results:
                        print(f"   Details: {opt_results['details']}")
                    if 'error_type' in opt_results:
                        print(f"   Error Type: {opt_results['error_type']}")
            
            # Show files created
            print(f"\n📁 Files Created:")
            for file_path in summary.get('files_created', []):
                print(f"   {file_path}")
            
            print(f"\n📂 Full Results Path: {run_dir.absolute()}")
            
        except Exception as e:
            print(f"❌ Error reading summary: {e}")
    else:
        print("❌ No summary file found")

def show_step_result(problem_id, step_name, run_timestamp=None, results_dir="data/results"):
    """
    Show the result of a specific workflow step.
    """
    results_path = Path(results_dir)
    problem_dir = results_path / f"problem_{problem_id}"
    
    if not problem_dir.exists():
        print(f"❌ Problem {problem_id} not found")
        return
    
    # Find the run directory
    if run_timestamp:
        run_dir = problem_dir / f"run_{run_timestamp}"
    else:
        run_dirs = [d for d in problem_dir.iterdir() if d.is_dir() and d.name.startswith('run_')]
        if not run_dirs:
            print(f"❌ No runs found for problem {problem_id}")
            return
        run_dir = sorted(run_dirs, key=lambda x: x.name)[-1]
    
    # Find the step file
    step_files = list(run_dir.rglob(f"{step_name}.*"))
    
    if not step_files:
        print(f"❌ Step '{step_name}' not found for problem {problem_id}")
        return
    
    step_file = step_files[0]
    
    print(f"📄 Problem {problem_id} - Step: {step_name}")
    try:
        relative_path = step_file.relative_to(Path.cwd())
        print(f"📂 File: {relative_path}")
    except ValueError:
        print(f"📂 File: {step_file}")
    print("="*80)
    
    try:
        with open(step_file, 'r', encoding='utf-8') as f:
            content = f.read()
        
        print(content)
        
    except Exception as e:
        print(f"❌ Error reading file: {e}")

def main():
    parser = argparse.ArgumentParser(description='Browse saved optimization results')
    parser.add_argument('--results-dir', type=str, default='data/results', 
                       help='Results directory (default: data/results)')
    
    subparsers = parser.add_subparsers(dest='command', help='Available commands')
    
    # List command
    list_parser = subparsers.add_parser('list', help='List all problem results')
    
    # Show command
    show_parser = subparsers.add_parser('show', help='Show detailed results for a problem')
    show_parser.add_argument('problem_id', type=str, help='Problem ID to show')
    show_parser.add_argument('--run', type=str, help='Specific run timestamp (default: most recent)')
    
    # Step command
    step_parser = subparsers.add_parser('step', help='Show specific workflow step result')
    step_parser.add_argument('problem_id', type=str, help='Problem ID')
    step_parser.add_argument('step_name', type=str, help='Step name (e.g., latex_model, gurobi_code)')
    step_parser.add_argument('--run', type=str, help='Specific run timestamp (default: most recent)')
    
    args = parser.parse_args()
    
    if args.command == 'list':
        list_problem_results(args.results_dir)
    elif args.command == 'show':
        show_problem_details(args.problem_id, args.run, args.results_dir)
    elif args.command == 'step':
        show_step_result(args.problem_id, args.step_name, args.run, args.results_dir)
    else:
        # Interactive mode
        print("📊 OPTIMIZATION RESULTS BROWSER")
        print("="*50)
        print("Commands:")
        print("  python browse_results.py list")
        print("  python browse_results.py show <problem_id>")
        print("  python browse_results.py step <problem_id> <step_name>")
        print("\nOr use --help for more options")

if __name__ == "__main__":
    main() 