#!/usr/bin/env python3
"""
API Key Management for OpenAI and other services.
"""

import os
import getpass
import json
from pathlib import Path


class APIKeyManager:
    """
    Class responsible for managing API keys from multiple sources.
    """
    
    def __init__(self, service_name="OpenAI"):
        """
        Initialize the API key manager.
        
        Args:
            service_name (str): Name of the service (e.g., "OpenAI", "Anthropic")
        """
        self.service_name = service_name
        self.api_key = None
        self.config_file = Path.home() / ".linearizellm_api_keys.json"
    
    def _load_saved_keys(self):
        """Load saved API keys from config file."""
        try:
            if self.config_file.exists():
                with open(self.config_file, 'r') as f:
                    return json.load(f)
        except:
            pass
        return {}
    
    def _save_key(self, service, api_key):
        """Save API key to config file."""
        try:
            keys = self._load_saved_keys()
            keys[service.lower()] = api_key
            
            # Ensure directory exists
            self.config_file.parent.mkdir(parents=True, exist_ok=True)
            
            with open(self.config_file, 'w') as f:
                json.dump(keys, f, indent=2)
            
            # Also set in environment for current session
            os.environ[f"{service.upper()}_API_KEY"] = api_key
            print(f"✅ {service} API key saved permanently")
            return True
        except Exception as e:
            print(f"⚠️ Could not save API key permanently: {e}")
            # Still set in environment for current session
            os.environ[f"{service.upper()}_API_KEY"] = api_key
            print(f"✅ {service} API key saved for this session only")
            return False
    
    def get_openai_api_key(self):
        """
        Get OpenAI API key from multiple sources in order of preference:
        1. Environment variable OPENAI_API_KEY
        2. Environment variable OPENAI_API_KEY_LINEARIZELLM (project-specific)
        3. Saved config file
        4. Interactive prompt (if running interactively)
        5. Default fallback key (for backward compatibility)
        
        Returns:
            str: OpenAI API key
        """
        # Try environment variables first
        api_key = os.getenv("OPENAI_API_KEY")
        if api_key:
            return api_key
        
        # Try project-specific environment variable
        api_key = os.getenv("OPENAI_API_KEY_LINEARIZELLM")
        if api_key:
            return api_key
        
        # Skip saved config file - always prompt for fresh key
        # saved_keys = self._load_saved_keys()
        # api_key = saved_keys.get("openai")
        # if api_key:
        #     # Set in environment for current session
        #     os.environ["OPENAI_API_KEY"] = api_key
        #     return api_key
        
        # If running interactively and no env var is set, prompt user
        try:
            if hasattr(__builtins__, '__IPYTHON__') or os.isatty(0):
                print("⚠️ OpenAI API key not found in environment variables.")
                print("You can set it using: export OPENAI_API_KEY=your_key_here")
                print("🔐 Enter your OpenAI API key (input will be hidden for security):")
                api_key = getpass.getpass("OpenAI API key: ")
                if api_key.strip():
                    # Set in environment for current session only (don't save permanently)
                    os.environ["OPENAI_API_KEY"] = api_key.strip()
                    print("✅ OpenAI API key set for this session only")
                    return api_key.strip()
        except:
            pass
        
        # No default key available - user must provide their own
        raise ValueError("OpenAI API key not found. Please set OPENAI_API_KEY environment variable or provide your API key.")
    
    def get_google_api_key(self):
        """
        Get Google API key from multiple sources in order of preference:
        1. Environment variable GOOGLE_API_KEY
        2. Environment variable GOOGLE_API_KEY_LINEARIZELLM (project-specific)
        3. Saved config file
        4. Interactive prompt (if running interactively)
        5. Default fallback key (for backward compatibility)
        
        Returns:
            str: Google API key
        """
        # Try environment variables first
        api_key = os.getenv("GOOGLE_API_KEY")
        if api_key:
            return api_key
        
        # Try project-specific environment variable
        api_key = os.getenv("GOOGLE_API_KEY_LINEARIZELLM")
        if api_key:
            return api_key
        
        # Skip saved config file - always prompt for fresh key
        # saved_keys = self._load_saved_keys()
        # api_key = saved_keys.get("google")
        # if api_key:
        #     # Set in environment for current session
        #     os.environ["GOOGLE_API_KEY"] = api_key
        #     return api_key
        
        # If running interactively and no env var is set, prompt user
        try:
            if hasattr(__builtins__, '__IPYTHON__') or os.isatty(0):
                print("⚠️ Google API key not found in environment variables.")
                print("You can set it using: export GOOGLE_API_KEY=your_key_here")
                print("🔐 Enter your Google API key (input will be hidden for security):")
                api_key = getpass.getpass("Google API key: ")
                if api_key.strip():
                    # Set in environment for current session only (don't save permanently)
                    os.environ["GOOGLE_API_KEY"] = api_key.strip()
                    print("✅ Google API key set for this session only")
                    return api_key.strip()
        except:
            pass
        
        # No default key available - user must provide their own
        raise ValueError("Google API key not found. Please set GOOGLE_API_KEY environment variable or provide your API key.")
    
    def get_api_key(self, service_name=None):
        """
        Generic method to get API key for any service.
        
        Args:
            service_name (str): Name of the service (defaults to self.service_name)
            
        Returns:
            str: API key for the specified service
        """
        service = service_name or self.service_name
        
        if service.lower() == "openai":
            return self.get_openai_api_key()
        elif service.lower() == "google":
            return self.get_google_api_key()
        else:
            # Try environment variable with service name
            env_var_name = f"{service.upper()}_API_KEY"
            api_key = os.getenv(env_var_name)
            if api_key:
                return api_key
            
            # Skip saved config file - always prompt for fresh key
            # saved_keys = self._load_saved_keys()
            # api_key = saved_keys.get(service.lower())
            # if api_key:
            #     # Set in environment for current session
            #     os.environ[env_var_name] = api_key
            #     return api_key
            
            # Try interactive prompt
            try:
                if hasattr(__builtins__, '__IPYTHON__') or os.isatty(0):
                    print(f"⚠️ {service} API key not found in environment variables.")
                    print(f"You can set it using: export {env_var_name}=your_key_here")
                    print(f"🔐 Enter your {service} API key (input will be hidden for security):")
                    api_key = getpass.getpass(f"{service} API key: ")
                    if api_key.strip():
                        # Set in environment for current session only (don't save permanently)
                        os.environ[env_var_name] = api_key.strip()
                        print(f"✅ {service} API key set for this session only")
                        return api_key.strip()
            except:
                pass
            
            raise ValueError(f"No API key found for {service}")
    
    def set_api_key(self, api_key, service_name=None):
        """
        Set API key for a service.
        
        Args:
            api_key (str): The API key to set
            service_name (str): Name of the service (defaults to self.service_name)
        """
        service = service_name or self.service_name
        self._save_key(service, api_key)
    
    def validate_api_key(self, api_key, service_name=None):
        """
        Basic validation of API key format.
        
        Args:
            api_key (str): The API key to validate
            service_name (str): Name of the service (defaults to self.service_name)
            
        Returns:
            bool: True if key appears valid, False otherwise
        """
        service = service_name or self.service_name
        
        if not api_key or not isinstance(api_key, str):
            return False
        
        if service.lower() == "openai":
            # OpenAI keys typically start with 'sk-'
            return api_key.startswith('sk-') and len(api_key) > 20
        else:
            # Generic validation - non-empty string
            return len(api_key.strip()) > 0


# Convenience function for backward compatibility
def get_openai_api_key():
    """
    Convenience function to get OpenAI API key.
    Maintains backward compatibility with existing code.
    
    Returns:
        str: OpenAI API key
    """
    manager = APIKeyManager("OpenAI")
    return manager.get_openai_api_key() 