#!/usr/bin/env python3
"""
Dependency checker for LinearizeLLM framework.
Checks if all required packages from requirements.txt are installed.
"""

import importlib
import sys
from pathlib import Path


def get_package_import_mapping():
    """Map package names to their actual import names."""
    return {
        'numpy': 'numpy',
        'pandas': 'pandas',
        'langchain': 'langchain',
        'langchain-core': 'langchain_core',
        'langchain-openai': 'langchain_openai',
        'langchain-google-genai': 'langchain_google_genai',
        'openai': 'openai',
        'gurobipy': 'gurobipy',
        'matplotlib': 'matplotlib',
        'seaborn': 'seaborn'
    }


def parse_requirements():
    """Parse requirements.txt and extract package names."""
    requirements_file = Path(__file__).parent.parent.parent / "requirements.txt"
    packages = []
    
    if not requirements_file.exists():
        print("❌ Error: requirements.txt not found!")
        return packages
    
    with open(requirements_file, 'r') as f:
        for line in f:
            line = line.strip()
            # Skip comments and empty lines
            if line.startswith('#') or not line:
                continue
            # Extract package name (remove version constraints)
            package_name = line.split('>=')[0].split('==')[0].split('<=')[0].split('~=')[0].split('!=')[0].strip()
            if package_name:
                packages.append(package_name)
    
    return packages


def check_dependencies():
    """Check if all required dependencies are installed."""
    print("🔍 Checking LinearizeLLM dependencies...")
    
    required_packages = parse_requirements()
    if not required_packages:
        print("❌ Error: Could not parse requirements.txt")
        return False
    
    # Get the mapping of package names to import names
    import_mapping = get_package_import_mapping()
    
    missing_packages = []
    installed_packages = []
    
    for package in required_packages:
        # Get the import name for this package
        import_name = import_mapping.get(package, package)
        
        try:
            importlib.import_module(import_name)
            installed_packages.append(package)
            print(f"✅ {package}")
        except ImportError:
            missing_packages.append(package)
            print(f"❌ {package} - NOT INSTALLED")
    
    print(f"\n📊 Summary:")
    print(f"   ✅ Installed: {len(installed_packages)}/{len(required_packages)}")
    
    if missing_packages:
        print(f"   ❌ Missing: {len(missing_packages)} packages")
        print(f"\n❌ CRITICAL ERROR: Missing required dependencies!")
        print(f"Please install missing packages using:")
        print(f"   pip install -r requirements.txt")
        print(f"\nMissing packages: {', '.join(missing_packages)}")
        return False
    else:
        print(f"   🎉 All dependencies are installed!")
        return True


def main():
    """Main function for standalone dependency checking."""
    if check_dependencies():
        print("\n✅ All dependencies are satisfied. LinearizeLLM is ready to run!")
        sys.exit(0)
    else:
        print("\n❌ Please install missing dependencies before running LinearizeLLM.")
        sys.exit(1)


if __name__ == "__main__":
    main() 