#!/usr/bin/env python3
"""
LLM Model Manager for supporting different LLM providers and their configurations.
"""

import os
from typing import Dict, Any, Optional, Union
from dataclasses import dataclass
from abc import ABC, abstractmethod

# Import different LLM providers
try:
    from langchain_openai import ChatOpenAI
    OPENAI_AVAILABLE = True
except ImportError:
    OPENAI_AVAILABLE = False

try:
    from langchain_anthropic import ChatAnthropic
    ANTHROPIC_AVAILABLE = True
except ImportError:
    ANTHROPIC_AVAILABLE = False

try:
    from langchain_google_genai import ChatGoogleGenerativeAI
    GOOGLE_AVAILABLE = True
except ImportError:
    GOOGLE_AVAILABLE = False

from src.utils.api_key_manager import APIKeyManager


@dataclass
class LLMConfig:
    """Configuration for an LLM model."""
    provider: str  # "openai", "anthropic", "google", etc.
    model_name: str  # "gpt-4o", "claude-3-opus", "gemini-pro", etc.
    temperature: float = 0.0
    max_tokens: Optional[int] = None
    top_p: Optional[float] = None
    frequency_penalty: Optional[float] = None
    presence_penalty: Optional[float] = None
    timeout: Optional[int] = None
    max_retries: int = 3
    custom_parameters: Optional[Dict[str, Any]] = None


class LLMProvider(ABC):
    """Abstract base class for LLM providers."""
    
    @abstractmethod
    def create_llm(self, config: LLMConfig) -> Any:
        """Create an LLM instance with the given configuration."""
        pass
    
    @abstractmethod
    def get_api_key(self, config: LLMConfig) -> str:
        """Get the API key for this provider."""
        pass


class OpenAIProvider(LLMProvider):
    """OpenAI LLM provider implementation."""
    
    def create_llm(self, config: LLMConfig):
        """Create an OpenAI ChatOpenAI instance."""
        if not OPENAI_AVAILABLE:
            raise ImportError("langchain_openai is not installed. Install with: pip install langchain-openai")
        
        # Get API key
        api_key = self.get_api_key(config)
        
        # Build parameters
        params = {
            "model": config.model_name,
            "openai_api_key": api_key,
            "temperature": config.temperature,
            "max_retries": config.max_retries
        }
        
        # Add optional parameters
        if config.max_tokens:
            params["max_tokens"] = config.max_tokens
        if config.top_p:
            params["top_p"] = config.top_p
        if config.frequency_penalty:
            params["frequency_penalty"] = config.frequency_penalty
        if config.presence_penalty:
            params["presence_penalty"] = config.presence_penalty
        if config.timeout:
            params["timeout"] = config.timeout
        
        # Add custom parameters
        if config.custom_parameters:
            params.update(config.custom_parameters)
        
        # Handle model_kwargs specifically for OpenAI
        if config.custom_parameters and "model_kwargs" in config.custom_parameters:
            params["model_kwargs"] = config.custom_parameters["model_kwargs"]
        
        return ChatOpenAI(**params)
    
    def get_api_key(self, config: LLMConfig) -> str:
        """Get OpenAI API key."""
        manager = APIKeyManager("OpenAI")
        return manager.get_openai_api_key()


class AnthropicProvider(LLMProvider):
    """Anthropic LLM provider implementation."""
    
    def create_llm(self, config: LLMConfig):
        """Create an Anthropic ChatAnthropic instance."""
        if not ANTHROPIC_AVAILABLE:
            raise ImportError("langchain_anthropic is not installed. Install with: pip install langchain-anthropic")
        
        # Get API key
        api_key = self.get_api_key(config)
        
        # Build parameters
        params = {
            "model": config.model_name,
            "anthropic_api_key": api_key,
            "temperature": config.temperature,
            "max_retries": config.max_retries
        }
        
        # Add optional parameters
        if config.max_tokens:
            params["max_tokens"] = config.max_tokens
        if config.top_p:
            params["top_p"] = config.top_p
        if config.timeout:
            params["timeout"] = config.timeout
        
        # Add custom parameters
        if config.custom_parameters:
            params.update(config.custom_parameters)
        
        return ChatAnthropic(**params)
    
    def get_api_key(self, config: LLMConfig) -> str:
        """Get Anthropic API key."""
        manager = APIKeyManager("Anthropic")
        return manager.get_api_key("Anthropic")


class GoogleProvider(LLMProvider):
    """Google LLM provider implementation."""
    
    def create_llm(self, config: LLMConfig):
        """Create a Google ChatGoogleGenerativeAI instance."""
        if not GOOGLE_AVAILABLE:
            raise ImportError("langchain_google_genai is not installed. Install with: pip install langchain-google-genai")
        
        # Get API key
        api_key = self.get_api_key(config)
        
        # Build parameters
        params = {
            "model": config.model_name,
            "google_api_key": api_key,
            "temperature": config.temperature,
            "max_retries": config.max_retries
        }
        
        # Add optional parameters
        if config.max_tokens:
            params["max_output_tokens"] = config.max_tokens
        if config.top_p:
            params["top_p"] = config.top_p
        if config.timeout:
            params["timeout"] = config.timeout
        
        # Add custom parameters
        if config.custom_parameters:
            params.update(config.custom_parameters)
        
        return ChatGoogleGenerativeAI(**params)
    
    def get_api_key(self, config: LLMConfig) -> str:
        """Get Google API key."""
        manager = APIKeyManager("Google")
        return manager.get_api_key("Google")


class LLMModelManager:
    """
    Manager for creating and configuring LLM models from different providers.
    """
    
    def __init__(self):
        """Initialize the LLM model manager."""
        self.providers = {
            "openai": OpenAIProvider(),
            "anthropic": AnthropicProvider(),
            "google": GoogleProvider()
        }
        
        # Default configurations for common models
        self.default_configs = {
            # OpenAI models with comprehensive parameters
            "gpt-4o": LLMConfig(
                "openai", "gpt-4o", 
                temperature=0.0,
                max_tokens=16000,  # Increased for large models
                top_p=1.0,
                frequency_penalty=0.0,
                presence_penalty=0.0,
                timeout=120,  # Increased timeout for large responses
                max_retries=3,
                custom_parameters={
                    "request_timeout": 120,
                    "model_kwargs": {
                        "response_format": {"type": "text"}
                    }
                }
            ),
            "gpt-4o-mini": LLMConfig(
                "openai", "gpt-4o-mini", 
                temperature=0.0,
                max_tokens=16000,  # Increased for large models
                top_p=1.0,
                frequency_penalty=0.0,
                presence_penalty=0.0,
                timeout=120,  # Increased timeout for large responses
                max_retries=3
            ),
            "gpt-4": LLMConfig(
                "openai", "gpt-4", 
                temperature=0.0,
                max_tokens=16000,  # Increased for large models
                top_p=1.0,
                frequency_penalty=0.0,
                presence_penalty=0.0,
                timeout=120,  # Increased timeout for large responses
                max_retries=3
            ),
            "gpt-3.5-turbo": LLMConfig(
                "openai", "gpt-3.5-turbo", 
                temperature=0.0,
                max_tokens=16000,  # Increased for large models
                top_p=1.0,
                frequency_penalty=0.0,
                presence_penalty=0.0,
                timeout=120,  # Increased timeout for large responses
                max_retries=3
            ),
            # Enhanced o3 configuration with comprehensive parameters
            "o3": LLMConfig(
                "openai", "gpt-4o", 
                temperature=0.1,
                max_tokens=16000,  # Increased for large models
                top_p=0.9,
                frequency_penalty=0.0,
                presence_penalty=0.0,
                timeout=120,  # Increased timeout for large responses
                max_retries=3,
                custom_parameters={
                    "request_timeout": 120,
                    "model_kwargs": {
                        "response_format": {"type": "text"}
                    }
                }
            ),  # Alias for backward compatibility
            
            # Anthropic models
            "claude-3-opus": LLMConfig("anthropic", "claude-3-opus-20240229", temperature=0.0, max_tokens=16000, timeout=120),
            "claude-3-sonnet": LLMConfig("anthropic", "claude-3-sonnet-20240229", temperature=0.0, max_tokens=16000, timeout=120),
            "claude-3-haiku": LLMConfig("anthropic", "claude-3-haiku-20240307", temperature=0.0, max_tokens=16000, timeout=120),
            
            # Google models
            "gemini-1.5-pro": LLMConfig("google", "gemini-1.5-pro", temperature=0.0, max_tokens=16000, timeout=120),
            "gemini-1.5-flash": LLMConfig("google", "gemini-1.5-flash", temperature=0.0, max_tokens=16000, timeout=120),
            "gemini-2.5-flash": LLMConfig("google", "gemini-2.5-flash", temperature=0.15, max_tokens=16000, timeout=120),
            "gemini-2.5-pro": LLMConfig(
                "google", "gemini-2.5-pro", 
                temperature=0.15,  # Slightly more creative
                max_tokens=32000,  # Higher token limit
                timeout=180,  # Longer timeout
                top_p=0.9  # Nucleus sampling
            ),
        }
    
    def create_llm(self, model_spec: Union[str, LLMConfig]) -> Any:
        """
        Create an LLM instance from a model specification.
        
        Args:
            model_spec: Either a string (model name) or LLMConfig object
            
        Returns:
            LLM instance (ChatOpenAI, ChatAnthropic, etc.)
        """
        if isinstance(model_spec, str):
            # Use default configuration for this model
            if model_spec in self.default_configs:
                config = self.default_configs[model_spec]
            else:
                # Try to infer provider from model name
                if model_spec.startswith("gpt-") or model_spec in ["o3"]:
                    config = LLMConfig("openai", model_spec, temperature=0.0)
                elif model_spec.startswith("claude-"):
                    config = LLMConfig("anthropic", model_spec, temperature=0.0)
                elif model_spec.startswith("gemini-"):
                    config = LLMConfig("google", model_spec, temperature=0.0)
                else:
                    raise ValueError(f"Unknown model '{model_spec}'. Please provide a full LLMConfig object.")
        else:
            config = model_spec
        
        # Get the provider
        provider = self.providers.get(config.provider.lower())
        if not provider:
            raise ValueError(f"Unsupported provider '{config.provider}'. Supported providers: {list(self.providers.keys())}")
        
        # Create the LLM instance
        return provider.create_llm(config)
    
    def get_available_models(self) -> Dict[str, list]:
        """Get list of available models by provider."""
        available = {}
        
        if OPENAI_AVAILABLE:
            available["openai"] = [
                "gpt-4o", "gpt-4o-mini", "gpt-4", "gpt-3.5-turbo", "o3"
            ]
        
        if ANTHROPIC_AVAILABLE:
            available["anthropic"] = [
                "claude-3-opus", "claude-3-sonnet", "claude-3-haiku"
            ]
        
        if GOOGLE_AVAILABLE:
            available["google"] = [
                "gemini-pro", "gemini-1.5-pro", "gemini-1.5-flash", "gemini-2.5-flash", "gemini-2.5-pro"
            ]
        
        return available
    
    def create_custom_config(self, provider: str, model_name: str, **kwargs) -> LLMConfig:
        """
        Create a custom LLM configuration.
        
        Args:
            provider: Provider name ("openai", "anthropic", "google")
            model_name: Model name
            **kwargs: Additional configuration parameters
            
        Returns:
            LLMConfig object
        """
        return LLMConfig(provider=provider, model_name=model_name, **kwargs)


# Global instance for convenience
llm_manager = LLMModelManager()


# Convenience functions for backward compatibility
def create_llm(model_spec: Union[str, LLMConfig]) -> Any:
    """Convenience function to create an LLM instance."""
    return llm_manager.create_llm(model_spec)


def get_available_models() -> Dict[str, list]:
    """Convenience function to get available models."""
    return llm_manager.get_available_models() 