#!/usr/bin/env python3
"""
Prompt Template Manager for loading and formatting prompt templates from files.
"""

from pathlib import Path


class PromptTemplateManager:
    """
    Manages prompt templates loaded from external files.
    """
    
    def __init__(self, prompts_dir="src/configs/prompts"):
        """
        Initialize the prompt template manager.
        
        Args:
            prompts_dir (str): Directory containing prompt template files
        """
        # Handle both relative and absolute paths
        if Path(prompts_dir).is_absolute():
            self.prompts_dir = Path(prompts_dir)
        else:
            # Try to resolve relative to current working directory first
            cwd_path = Path.cwd() / prompts_dir
            if cwd_path.exists():
                self.prompts_dir = cwd_path
            else:
                # Try relative to the script location
                script_dir = Path(__file__).parent.parent
                script_path = script_dir / prompts_dir
                if script_path.exists():
                    self.prompts_dir = script_path
                else:
                    # Fall back to original behavior
                    self.prompts_dir = Path(prompts_dir)
        self.templates = {}
        self._load_templates()
    
    def _load_templates(self):
        """Load all prompt templates from the prompts directory."""
        if not self.prompts_dir.exists():
            print(f"⚠️ Prompts directory not found: {self.prompts_dir}")
            print(f"   Current working directory: {Path.cwd()}")
            print(f"   Script location: {Path(__file__).parent}")
            return
        
        # Load all .txt files in the prompts directory
        for template_file in self.prompts_dir.glob("*.txt"):
            template_name = template_file.stem
            try:
                with open(template_file, 'r', encoding='utf-8') as f:
                    template_content = f.read()
                self.templates[template_name] = template_content
                print(f"📄 Loaded prompt template: {template_name}")
            except Exception as e:
                print(f"⚠️ Failed to load template {template_name}: {str(e)}")
    
    def get_template(self, template_name):
        """
        Get a prompt template by name.
        
        Args:
            template_name (str): Name of the template (without .txt extension)
            
        Returns:
            str: Template content or None if not found
        """
        return self.templates.get(template_name)
    
    def format_template(self, template_name, **kwargs):
        """
        Format a prompt template with the provided parameters.
        
        Args:
            template_name (str): Name of the template
            **kwargs: Parameters to substitute in the template
            
        Returns:
            str: Formatted prompt template
            
        Raises:
            KeyError: If template is not found
            ValueError: If template formatting fails
        """
        template_content = self.get_template(template_name)
        if template_content is None:
            raise KeyError(f"Template '{template_name}' not found. Available templates: {list(self.templates.keys())}")
        
        try:
            # Use str.format() for {} placeholder substitution
            return template_content.format(**kwargs)
        except KeyError as e:
            missing_key = str(e).strip("'")
            raise ValueError(f"Missing required parameter '{missing_key}' for template '{template_name}'")
        except Exception as e:
            raise ValueError(f"Failed to format template '{template_name}': {str(e)}")
    
    def list_templates(self):
        """
        List all available templates.
        
        Returns:
            list: List of available template names
        """
        return list(self.templates.keys())
    
    def reload_templates(self):
        """Reload all templates from files."""
        self.templates.clear()
        self._load_templates()
    
    def add_template(self, template_name, template_content):
        """
        Add a template programmatically.
        
        Args:
            template_name (str): Name of the template
            template_content (str): Template content
        """
        self.templates[template_name] = template_content
    
    def save_template(self, template_name, template_content):
        """
        Save a template to a file.
        
        Args:
            template_name (str): Name of the template
            template_content (str): Template content
        """
        self.prompts_dir.mkdir(exist_ok=True)
        template_file = self.prompts_dir / f"{template_name}.txt"
        
        try:
            with open(template_file, 'w', encoding='utf-8') as f:
                f.write(template_content)
            self.templates[template_name] = template_content
            print(f"💾 Saved template: {template_name}")
        except Exception as e:
            print(f"⚠️ Failed to save template {template_name}: {str(e)}") 