import logging

from fastbo.try_import import (
    try_import_gpsearchers_message,
    try_import_kde_message,
    try_import_bore_message,
    try_import_botorch_message,
)
from fastbo.optimizer.schedulers.searchers import (
    BaseSearcher,
    RandomSearcher,
    GridSearcher,
)

__all__ = ["searcher_factory"]

logger = logging.getLogger(__name__)


SUPPORTED_SEARCHERS_FIFO = {
    "random",
    "grid",
    "kde",
    "bore",
    "bayesopt",
    "fastbo",
}


SUPPORTED_SEARCHERS_HYPERBAND = {
    "random",
    "grid",
    "kde",
    "bayesopt",
    "hypertune",
    "fastbo",
}


_OUR_MULTIFIDELITY_SCHEDULERS = {
    "hyperband_stopping",
    "hyperband_promotion",
    "hyperband_cost_promotion",
    "hyperband_pasha",
    "hyperband_synchronous",
}


def searcher_factory(searcher_name: str, **kwargs) -> BaseSearcher:
    """Factory for searcher objects

    This function creates searcher objects from string argument name and
    additional kwargs. It is typically called in the constructor of a
    scheduler (see :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`),
    which provides most of the required ``kwargs``.

    :param searcher_name: Value of ``searcher`` argument to scheduler (see
        :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`)
    :param kwargs: Argument to
        :class:`~syne_tune.optimizer.schedulers.searchers.BaseSearcher` constructor
    :return: New searcher object
    """
    supported_schedulers = None
    scheduler = kwargs.get("scheduler")
    model = kwargs.get("model", "gp_multitask")
    if searcher_name == "random":
        searcher_cls = RandomSearcher
    elif searcher_name == "grid":
        searcher_cls = GridSearcher
    elif searcher_name == "kde":
        try:
            from syne_tune.optimizer.schedulers.searchers.kde import (
                KernelDensityEstimator,
                MultiFidelityKernelDensityEstimator,
            )
        except ImportError:
            logger.info(try_import_kde_message())
            raise

        if scheduler == "fifo":
            searcher_cls = KernelDensityEstimator
        else:
            supported_schedulers = _OUR_MULTIFIDELITY_SCHEDULERS
            searcher_cls = MultiFidelityKernelDensityEstimator
    elif searcher_name == "bore":
        try:
            from syne_tune.optimizer.schedulers.searchers.bore import (
                Bore,
                MultiFidelityBore,
            )
        except ImportError:
            logger.info(try_import_bore_message())
            raise

        if scheduler == "fifo":
            searcher_cls = Bore
        else:
            supported_schedulers = _OUR_MULTIFIDELITY_SCHEDULERS
            searcher_cls = MultiFidelityBore
    else:
        gp_searchers = {
            "bayesopt",
            "hypertune",
            "fastbo",
        }
        assert (
            searcher_name in gp_searchers
        ), f"searcher '{searcher_name}' is not supported"
        try:
            from syne_tune.optimizer.schedulers.searchers import (
                GPFIFOSearcher,
                GPMultiFidelitySearcher,
            )
            from syne_tune.optimizer.schedulers.searchers.hypertune import (
                HyperTuneSearcher,
            )
            from syne_tune.optimizer.schedulers.searchers.fastbo.fastbo_searcher import (
                FastBOSearcher,
            )
        except ImportError:
            logger.info(try_import_gpsearchers_message())
            raise

        if searcher_name == "bayesopt":
            if scheduler == "fifo":
                searcher_cls = GPFIFOSearcher # here for BayesianOptimization
        elif searcher_name == "fastbo" and scheduler == "fifo":
            searcher_cls = FastBOSearcher
        elif searcher_name == "hypertune":
            supported_schedulers = _OUR_MULTIFIDELITY_SCHEDULERS
            searcher_cls = HyperTuneSearcher

    if supported_schedulers is not None:
        assert scheduler is not None, "Scheduler must set search_options['scheduler']"
        assert scheduler in supported_schedulers, (
            f"Searcher '{searcher_name}' only works with schedulers "
            + f"{supported_schedulers} (not with '{scheduler}')"
        )
    searcher = searcher_cls(**kwargs)
    return searcher
