from typing import Dict, Optional, Any, List, Union
import logging
import copy

from fastbo.optimizer.schedulers import (
    FIFOScheduler,
    HyperbandScheduler,
    PopulationBasedTraining,
)
from fastbo.optimizer.schedulers.multiobjective import MOASHA
from fastbo.optimizer.schedulers.searchers.regularized_evolution import (
    RegularizedEvolution,
)
from fastbo.optimizer.schedulers.random_seeds import RandomSeedGenerator
from fastbo.try_import import (
    try_import_blackbox_repository_message,
    try_import_bore_message,
    try_import_botorch_message,
)
from fastbo.util import dict_get

logger = logging.getLogger(__name__)


def _random_seed_from_generator(random_seed: int) -> int:
    """
    This helper makes sure that a searcher within
    :class:`~syne_tune.optimizer.schedulers.FIFOScheduler` is seeded in the same
    way whether it is created by the searcher factory, or by hand.

    :param random_seed: Random seed for scheduler
    :return: Random seed to be used for searcher created by hand
    """
    return RandomSeedGenerator(random_seed)()


def _assert_searcher_must_be(kwargs: Dict[str, Any], name: str):
    searcher = kwargs.get("searcher")
    assert searcher is None or searcher == name, f"Must have searcher='{name}'"


class RandomSearch(FIFOScheduler):
    """Random search.

    See :class:`~syne_tune.optimizer.schedulers.searchers.RandomSearcher`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`
    """

    def __init__(self, config_space: Dict[str, Any], metric: str, **kwargs):
        searcher_name = "random"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(RandomSearch, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            **kwargs,
        )


class GridSearch(FIFOScheduler):
    """Grid search.

    See :class:`~syne_tune.optimizer.schedulers.searchers.GridSearcher`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`
    """

    def __init__(self, config_space: Dict[str, Any], metric: str, **kwargs):
        searcher_name = "grid"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(GridSearch, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            **kwargs,
        )


class BayesianOptimization(FIFOScheduler):
    """Gaussian process based Bayesian optimization.

    See :class:`~syne_tune.optimizer.schedulers.searchers.GPFIFOSearcher`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`
    """

    def __init__(self, config_space: Dict[str, Any], metric: str, **kwargs):
        searcher_name = "bayesopt"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(BayesianOptimization, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            **kwargs,
        )


def _assert_need_one(kwargs: Dict[str, Any], need_one: Optional[set] = None):
    if need_one is None:
        need_one = {"max_t", "max_resource_attr"}
    assert need_one.intersection(kwargs.keys()), f"Need one of these: {need_one}"


class ASHA(HyperbandScheduler):
    """Asynchronous Sucessive Halving (ASHA).

    One of ``max_t``, ``max_resource_attr`` needs to be in ``kwargs``. For
    ``type="promotion"``, the latter is more useful, see also
    :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`
    """

    def __init__(
        self, config_space: Dict[str, Any], metric: str, resource_attr: str, **kwargs
    ):
        _assert_need_one(kwargs)
        searcher_name = "random"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(ASHA, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            resource_attr=resource_attr,
            **kwargs,
        )


class MOBSTER(HyperbandScheduler):
    """Model-based Asynchronous Multi-fidelity Optimizer (MOBSTER).

    One of ``max_t``, ``max_resource_attr`` needs to be in ``kwargs``. For
    ``type="promotion"``, the latter is more useful, see also
    :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    MOBSTER can be run with different surrogate models. The model is selected
    by ``search_options["model"]`` in ``kwargs``. The default is ``"gp_multitask"``
    (jointly dependent multi-task GP model), another useful choice is
    ``"gp_independent"`` (independent GP models at each rung level, with shared
    ARD kernel).

    See :class:`~syne_tune.optimizer.schedulers.searchers.GPMultifidelitySearcher`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`
    """

    def __init__(
        self, config_space: Dict[str, Any], metric: str, resource_attr: str, **kwargs
    ):
        _assert_need_one(kwargs)
        searcher_name = "bayesopt"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(MOBSTER, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            resource_attr=resource_attr,
            **kwargs,
        )


class HyperTune(HyperbandScheduler):
    """
    One of ``max_t``, ``max_resource_attr`` needs to be in ``kwargs``. For
    ``type="promotion"``, the latter is more useful, see also
    :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    Hyper-Tune is a model-based variant of ASHA with more than one bracket.
    It can be seen as extension of MOBSTER and can be used with
    ``search_options["model"]`` in ``kwargs`` being ``"gp_independent"`` or
    ``"gp_multitask"``. It has a model-based way to sample the bracket for every
    new trial, as well as an ensemble predictive distribution feeding into the
    acquisition function. Our implementation is based on:

        | Yang Li et al
        | Hyper-Tune: Towards Efficient Hyper-parameter Tuning at Scale
        | VLDB 2022
        | https://arxiv.org/abs/2201.06834

    See also
    :class:`~syne_tune.optimizer.schedulers.searchers.bayesopt.gpautograd.hypertune.gp_model.HyperTuneIndependentGPModel`,
    and see
    :class:`~syne_tune.optimizer.schedulers.searchers.hypertune.HyperTuneSearcher`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`
    """

    def __init__(self, config_space: Dict, metric: str, resource_attr: str, **kwargs):
        _assert_need_one(kwargs)
        searcher_name = "hypertune"
        _assert_searcher_must_be(kwargs, searcher_name)
        kwargs = copy.deepcopy(kwargs)
        search_options = dict_get(kwargs, "search_options", dict())
        k, v, supp = "model", "gp_independent", {"gp_independent", "gp_multitask"}
        model = search_options.get(k, v)
        assert model in supp, (
            f"HyperTune does not support search_options['{k}'] = '{model}'"
            f", must be in {supp}"
        )
        search_options[k] = model
        kwargs["search_options"] = search_options
        super(HyperTune, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            resource_attr=resource_attr,
            **kwargs,
        )


class PASHA(HyperbandScheduler):
    """Progressive ASHA.

    One of ``max_t``, ``max_resource_attr`` needs to be in ``kwargs``. The latter is
    more useful, see also :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`
    """

    def __init__(
        self, config_space: Dict[str, Any], metric: str, resource_attr: str, **kwargs
    ):
        _assert_need_one(kwargs)
        super(PASHA, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher="random",  # default, can be overwritten
            resource_attr=resource_attr,
            type="pasha",
            **kwargs,
        )


class BOHB(HyperbandScheduler):
    """Asynchronous BOHB

    Combines :class:`ASHA` with TPE-like Bayesian optimization, using kernel
    density estimators.

    One of ``max_t``, ``max_resource_attr`` needs to be in ``kwargs``. For
    ``type="promotion"``, the latter is more useful, see also
    :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    See
    :class:`~syne_tune.optimizer.schedulers.searchers.kde.MultiFidelityKernelDensityEstimator`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`
    """

    def __init__(
        self, config_space: Dict[str, Any], metric: str, resource_attr: str, **kwargs
    ):
        _assert_need_one(kwargs)
        searcher_name = "kde"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(BOHB, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            resource_attr=resource_attr,
            **kwargs,
        )


class SyncHyperband(SynchronousGeometricHyperbandScheduler):
    """Synchronous Hyperband.

    One of ``max_resource_level``, ``max_resource_attr`` needs to be in ``kwargs``.
    The latter is more useful, see also :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.synchronous.SynchronousGeometricHyperbandScheduler`
    """

    def __init__(
        self,
        config_space: Dict[str, Any],
        metric: str,
        resource_attr: str,
        **kwargs,
    ):
        _assert_need_one(kwargs, need_one={"max_resource_level", "max_resource_attr"})
        super(SyncHyperband, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher="random",  # default, can be overwritten
            resource_attr=resource_attr,
            **kwargs,
        )


class SyncBOHB(SynchronousGeometricHyperbandScheduler):
    """Synchronous BOHB.

    Combines :class:`SyncHyperband` with TPE-like Bayesian optimization, using
    kernel density estimators.

    One of ``max_resource_level``, ``max_resource_attr`` needs to be in ``kwargs``.
    The latter is more useful, see also
    :class:`~syne_tune.optimizer.schedulers.HyperbandScheduler`.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param resource_attr: Name of resource attribute
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.synchronous.SynchronousGeometricHyperbandScheduler`
    """

    def __init__(
        self,
        config_space: Dict[str, Any],
        metric: str,
        resource_attr: str,
        **kwargs,
    ):
        _assert_need_one(kwargs, need_one={"max_resource_level", "max_resource_attr"})
        searcher_name = "kde"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(SyncBOHB, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            resource_attr=resource_attr,
            **kwargs,
        )


def _create_searcher_kwargs(
    config_space: Dict[str, Any],
    metric: str,
    random_seed: Optional[int],
    kwargs: Dict[str, Any],
) -> Dict[str, Any]:
    searcher_kwargs = dict(
        config_space=config_space,
        metric=metric,
        points_to_evaluate=kwargs.get("points_to_evaluate"),
    )
    search_options = dict_get(kwargs, "search_options", dict())
    searcher_kwargs.update(search_options)
    if random_seed is not None:
        searcher_kwargs["random_seed"] = _random_seed_from_generator(random_seed)
    return searcher_kwargs


class BORE(FIFOScheduler):
    """Bayesian Optimization by Density-Ratio Estimation (BORE).

    See :class:`~syne_tune.optimizer.schedulers.searchers.bore.Bore`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param random_seed: Random seed, optional
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`
    """

    def __init__(
        self,
        config_space: Dict[str, Any],
        metric: str,
        random_seed: Optional[int] = None,
        **kwargs,
    ):
        try:
            from syne_tune.optimizer.schedulers.searchers.bore import Bore
        except ImportError:
            logging.info(try_import_bore_message())
            raise

        searcher_kwargs = _create_searcher_kwargs(
            config_space, metric, random_seed, kwargs
        )
        super(BORE, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=Bore(**searcher_kwargs),
            random_seed=random_seed,
            **kwargs,
        )


class REA(FIFOScheduler):
    """Regularized Evolution (REA).

    See :class:`~syne_tune.optimizer.schedulers.searchers.regularized_evolution.RegularizedEvolution`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param population_size: See
        :class:`~syne_tune.optimizer.schedulers.searchers.RegularizedEvolution`.
        Defaults to 100
    :param sample_size: See
        :class:`~syne_tune.optimizer.schedulers.searchers.RegularizedEvolution`.
        Defaults to 10
    :param random_seed: Random seed, optional
    :param kwargs: Additional arguments to
        :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`
    """

    def __init__(
        self,
        config_space: Dict[str, Any],
        metric: str,
        population_size: int = 100,
        sample_size: int = 10,
        random_seed: Optional[int] = None,
        **kwargs,
    ):
        searcher_kwargs = _create_searcher_kwargs(
            config_space, metric, random_seed, kwargs
        )
        searcher_kwargs["population_size"] = population_size
        searcher_kwargs["sample_size"] = sample_size
        super(REA, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=RegularizedEvolution(**searcher_kwargs),
            random_seed=random_seed,
            **kwargs,
        )


class KDE(FIFOScheduler):
    """Single-fidelity variant of BOHB

    Combines :class:`~syne_tune.optimizer.schedulers.FIFOScheduler` with TPE-like
    Bayesian optimization, using kernel density estimators.

    See
    :class:`~syne_tune.optimizer.schedulers.searchers.kde.KernelDensityEstimator`
    for ``kwargs["search_options"]`` parameters.

    :param config_space: Configuration space for evaluation function
    :param metric: Name of metric to optimize
    :param kwargs: Additional arguments to :class:`~syne_tune.optimizer.schedulers.FIFOScheduler`
    """

    def __init__(self, config_space: Dict[str, Any], metric: str, **kwargs):
        searcher_name = "kde"
        _assert_searcher_must_be(kwargs, searcher_name)
        super(KDE, self).__init__(
            config_space=config_space,
            metric=metric,
            searcher=searcher_name,
            **kwargs,
        )


# Dictionary that allows to also list baselines who don't need a wrapper class
# such as :class:`PopulationBasedTraining`
baselines_dict = {
    "Random Search": RandomSearch,
    "Grid Search": GridSearch,
    "Bayesian Optimization": BayesianOptimization,
    "ASHA": ASHA,
    "MOBSTER": MOBSTER,
    "PASHA": PASHA,
    "MOASHA": MOASHA,
    "PBT": PopulationBasedTraining,
    "BORE": BORE,
    "REA": REA,
    "SyncHyperband": SyncHyperband,
    "SyncBOHB": SyncBOHB,
}
