#!/usr/bin/env python3
"""
Created on 16:08, Aug. 31st, 2023

@author: Anonymous
"""
import copy as cp
import tensorflow as tf
import tensorflow.keras as K
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.path.join(os.pardir, os.pardir))
    from layers import *
else:
    from .layers import *
import utils.model

__all__ = [
    "clip_transformer",
]

class clip_transformer(K.Model):
    """
    `clip_transformer` model, with considering time information.
    """

    def __init__(self, params, **kwargs):
        """
        Initialize `clip_transformer` object.

        Args:
            params: Model parameters initialized by clip_transformer_params, updated by params.iteration.
            kwargs: The arguments related to initialize `tf.keras.Model`-style object.

        Returns:
            None
        """
        # First call super class init function to set up `K.Model`
        # style model and inherit it's functionality.
        super(clip_transformer, self).__init__(**kwargs)

        # Copy hyperparameters (e.g. network sizes) from parameter dotdict,
        # usually generated from clip_transformer_params() in params/clip_transformer_params.py.
        self.params = cp.deepcopy(params)

        # Create trainable vars.
        self._init_trainable()

    """
    init funcs
    """
    # def _init_trainable func
    def _init_trainable(self):
        """
        Initialize trainable variables.

        Args:
            None

        Returns:
            None
        """
        ## Construct encoder blocks.
        # Initialize encoder blocks.
        self.encoders = []
        # Create encoder for each modality.
        for modality_idx in range(self.params.n_modalities):
            # Initialize encoder block.
            model_encoder = K.models.Sequential(name="encoder-{:d}".format(modality_idx))
            # Add input embedding layer.
            # emb_input - (batch_size, seq_len, d_input) -> (batch_size, seq_len, d_model)
            model_encoder.add(K.layers.Dense(
                # Modified `Dense` layer parameters.
                units=self.params.encoder.d_model,
                kernel_initializer=K.initializers.random_normal(mean=0., stddev=0.01),
                bias_initializer=K.initializers.constant(value=0.01),
                # Default `Dense` layer parameters.
                activation=None, use_bias=True, kernel_regularizer=None, bias_regularizer=None,
                activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            ))
            # Add position embedding layer.
            # emb_pos - (batch_size, seq_len, d_model) -> (batch_size, seq_len, d_model)
            model_encoder.add(PositionEmbedding(self.params.encoder.max_len))
            # Add attention layers.
            # encoder - (batch_size, seq_len, d_model) -> (batch_size, seq_len, d_model)
            model_encoder.add(K.models.Sequential(layers=[TransformerBlock(
                n_heads=self.params.encoder.n_heads, d_head=self.params.encoder.d_head,
                mha_dropout_prob=self.params.encoder.mha_dropout_prob, d_ff=self.params.encoder.d_ff,
                ff_dropout_prob=self.params.encoder.ff_dropout_prob) for _ in range(self.params.encoder.n_blocks)
            ]))
            # Append `model_encoder` to `encoders`.
            self.encoders.append(model_encoder)
        ## Construct feature blocks.
        # Initialize feature blocks.
        self.feature_blocks = []
        # Create feature block for each modality.
        for modality_idx in range(self.params.n_modalities):
            # Construct feature block.
            model_feature = K.models.Sequential(name="feature-{:d}".format(modality_idx))
            model_feature.add(K.layers.Flatten())
            # Append `model_feature` to `feature_blocks`.
            self.feature_blocks.append(model_feature)
        ## Construct contrastive block.
        self.contrastive_block = LossLayer(d_contra=self.params.d_contra, loss_mode="clip_orig") 
        ## Construct classification blocks.
        # Initialize classification blocks.
        self.classification_blocks = []
        # Create classification block for each modality.
        for modality_idx in range(self.params.n_modalities):
            # Initialize classification block.
            model_classification = K.models.Sequential(name="classification-{:d}".format(modality_idx))
            # Add hidden `Dense` layers.
            for d_hidden_i in self.params.cls.d_hidden:
                model_classification.add(K.layers.Dense(
                    # Modified `Dense` layer parameters.
                    units=d_hidden_i, activation="relu",
                    kernel_initializer=K.initializers.random_normal(mean=0., stddev=0.01),
                    bias_initializer=K.initializers.constant(value=0.01),
                    # Default `Dense` layer parameters.
                    use_bias=True, kernel_regularizer=None, bias_regularizer=None,
                    activity_regularizer=None, kernel_constraint=None, bias_constraint=None
                ))
            # Add `Dropout` after hidden `Dense` layer.
            if self.params.cls.dropout > 0.:
                model_classification.add(K.layers.Dropout(rate=self.params.cls.dropout))
            # Add the final classification `Dense` layer.
            model_classification.add(K.layers.Dense(
                # Modified `Dense` layer parameters.
                units=self.params.cls.d_output,
                kernel_initializer=K.initializers.random_normal(mean=0., stddev=0.01),
                bias_initializer=K.initializers.constant(value=0.01),
                # Default `Dense` layer parameters.
                activation=None, use_bias=True, kernel_regularizer=None, bias_regularizer=None,
                activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            )); model_classification.add(K.layers.Softmax(axis=-1))
            # Append `model_classification` to `classification_blocks`.
            self.classification_blocks.append(model_classification)

    """
    network funcs
    """
    # def call func
    def call(self, inputs, training=None, mask=None):
        """
        Forward `clip_transformer` to get the final predictions.

        Args:
            inputs: tuple - The input data.
            training: Boolean or boolean scalar tensor, indicating whether to run
                the `Network` in training mode or inference mode.
            mask: A mask or list of masks. A mask can be either a tensor or None (no mask).

        Returns:
            outputs: (batch_size, n_labels) - The output labels.
            loss: tf.float32 - The corresponding loss.
        """
        # Initialize components of inputs.
        # X - (n_modalities[list], batch_size, seq_len, n_channels)
        # y - (batch_size,)
        X = inputs[0]; y_true = inputs[1]
        # Forward encoder layer to get extracted feature maps, then aggregate to get features.
        # X_f - (n_modalities[list], batch_size, n_features)
        X_f = [self.feature_blocks[modality_idx](self.encoders[modality_idx](X[modality_idx]))\
            for modality_idx in range(self.params.n_modalities)]
        # Forward contrastive block to calculate contrastive loss.
        # TODO: Make sure that the first modality is tmr, the second modality is image!!!
        # loss_contra - tf.float32
        # contra_matrix - (batch_size, batch_size), the first dimension is z, and the second dimension is y.
        loss_contra, contra_matrix = self.contrastive_block((X_f[0], X_f[1]))
        # If use `max_z` prediction mode, assign each label with the data item with max similarity.
        if self.params.contra_pred_mode == "max_z":
            # contra_idxs - (batch_size,), y_pred_contra - (batch_size, n_labels)
            contra_idxs = tf.squeeze(tf.argmax(contra_matrix, axis=0))
            y_pred_contra = tf.gather(y_true, indices=contra_idxs, axis=0)
        # If use `prob_z` prediction mode, assign each label with the weighted probability of all data items.
        elif self.params.contra_pred_mode == "prob_z":
            # contra_prob - (batch_size, batch_size), y_pred_contra - (batch_size, n_labels)
            contra_prob = tf.transpose(contra_matrix / tf.reduce_sum(contra_matrix, axis=0, keepdims=True), perm=[1,0])
            y_pred_contra = tf.matmul(contra_prob, y_true)
        # If use `max_y` prediction mode, assign each data item with the label with max similarity.
        elif self.params.contra_pred_mode == "max_y":
            # contra_idxs - (batch_size,), y_pred_contra - (batch_size, n_labels)
            contra_idxs = tf.squeeze(tf.argmax(contra_matrix, axis=-1))
            y_pred_contra = tf.gather(y_true, indices=contra_idxs, axis=0)
        # If use `prob_y` prediction mode, assign each data item with the weighted probability of all labels.
        elif self.params.contra_pred_mode == "prob_y":
            # contra_prob - (batch_size, batch_size), y_pred_contra - (batch_size, n_labels)
            contra_prob = contra_matrix / tf.reduce_sum(contra_matrix, axis=-1, keepdims=True)
            y_pred_contra = tf.matmul(contra_prob, y_true)
        # Get unknown contrastive prediction mode.
        else:
            raise ValueError((
                "ERROR: Unknown contrastive prediction mode {} in train.conv_net."
            ).format(self.params.contra_pred_mode))
        # Forward classification block to do classification.
        # Note: The layer before the final fc layers is considered as feature vectors.
        # y_pred_bce - (n_modalities[list], batch_size, n_labels)
        y_pred_bce = [self.classification_blocks[modality_idx](X_f[modality_idx])\
            for modality_idx in range(self.params.n_modalities)]
        # Calculate the binary cross entropy loss.
        # loss_bce - (n_modalities[list],)
        loss_bce = [tf.reduce_mean(self._loss_bce(y_pred_bce[modality_idx], y_true))\
            for modality_idx in range(self.params.n_modalities)]
        # Calculate the final `y_pred` & `loss`.
        # TODO: Make sure that the first modality is tmr!!!
        # y_pred - (batch_size, n_labels); loss - tf.float32
        y_pred = (1 - self.params.contra_pred_ratio) * y_pred_bce[0] + self.params.contra_pred_ratio * y_pred_contra
        loss = (1 - self.params.contra_loss_ratio) * tf.reduce_mean(tf.stack(loss_bce, axis=0)) +\
            self.params.contra_loss_ratio * self.params.contra_loss_scale * loss_contra
        # Return the final `outputs` & `loss`.
        return y_pred, loss

    # def _loss_bce func
    @utils.model.tf_scope
    def _loss_bce(self, value, target):
        """
        Calculates binary cross entropy between tensors value and target.
        Get mean over last dimension to keep losses of different batches separate.
        :param value: (batch_size,) - Value of the object.
        :param target: (batch_size,) - Target of the object.
        :return loss: (batch_size,) - Loss between value and target.
        """
        # Note: `tf.nn.softmax_cross_entropy_with_logits` needs unscaled log probabilities,
        # we must not add `tf.nn.Softmax` layer at the last of the model.
        # loss - (batch_size,)
        loss = tf.nn.softmax_cross_entropy_with_logits(labels=target,logits=value) if type(value) is not list else\
            [tf.nn.softmax_cross_entropy_with_logits(labels=target[i],logits=value[i]) for i in range(len(value))]
        # Return the final `loss`.
        return loss

    """
    tool funcs
    """
    # def summary func
    @utils.model.tf_scope
    def summary(self, print_fn=None):
        """
        Summary built model.
        :param print_fn: callable - Print function to use. Defaults to `print`. It will be called on each
            line of the summary. You can set it to a custom function in order to capture the string summary.
        """
        super(clip_transformer, self).summary(print_fn=print_fn)

if __name__ == "__main__":
    import numpy as np
    # local dep
    from params.clip_transformer_params import clip_transformer_params

    # macro
    dataset = "eeg_anonymous"; batch_size = 64; seq_len = 80; n_channels = 55; n_features = 128

    # Initialize training process.
    utils.model.set_seeds(42)

    # Instantiate params.
    clip_transformer_params_inst = clip_transformer_params(dataset=dataset)
    n_labels = clip_transformer_params_inst.model.n_labels
    # Instantiate clip_transformer.
    clip_transformer_inst = clip_transformer(clip_transformer_params_inst.model)
    # Initialize input X & label y_true.
    X = [tf.random.normal((batch_size, seq_len, n_channels), dtype=tf.float32),
        tf.random.normal((batch_size, seq_len, n_channels), dtype=tf.float32)]
    y_true = tf.cast(tf.one_hot(tf.cast(tf.range(batch_size), dtype=tf.int64), n_labels), dtype=tf.float32)
    # Forward layers in `clip_transformer_inst`.
    outputs, loss = clip_transformer_inst((X, y_true)); clip_transformer_inst.summary()

