#!/usr/bin/env python3
"""
Created on 01:59, Jul. 21st, 2023

@author: Anonymous
"""
import copy as cp
import numpy as np
import tensorflow as tf
import tensorflow.keras as K
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.pardir)
import utils.model

__all__ = [
    "cnn_ensemble",
]

class cnn_ensemble(K.Model):
    """
    `cnn_ensemble` model, which considering time information.
    """

    def __init__(self, params):
        """
        Initialize `cnn_ensemble` object.

        Args:
            params: Model parameters initialized by cnn_ensemble_params, updated by params.iteration.

        Returns:
            None
        """
        # First call super class init function to set up `K.Model`
        # style model and inherit it's functionality.
        super(cnn_ensemble, self).__init__()

        # Copy hyperparameters (e.g. network sizes) from parameter dotdict,
        # usually generated from cnn_ensemble_params() in params/cnn_ensemble_params.py.
        self.params = cp.deepcopy(params)

        # Create trainable vars.
        self._init_trainable()

    """
    init funcs
    """
    # def _init_trainable func
    def _init_trainable(self):
        """
        Initialize trainable variables.

        Args:
            None

        Returns:
            None
        """
        self.models = [self._init_cnn() for _ in range(self.params.n_models)]

    # def _init_cnn func
    def _init_cnn(self):
        """
        Initialize CNN module.

        Args:
            None

        Returns:
            None
        """
        ## Initialize trainable cnn layers.
        model_cnn = K.models.Sequential(name="CNN")
        # Add `Conv1D` & `MaxPool1D` layers.
        for cnn_idx in range(len(self.params.cnn.n_filters)):
            # Initialize `Conv1D` layer. `tf.keras.layers.Conv1D` is different from `torch.nn.Conv1d`. It doesn't have
            # `in_channels` argument. And `filters` argument equals to `out_channels` argument.
            out_channels, kernel_size = self.params.cnn.n_filters[cnn_idx], self.params.cnn.d_kernel[cnn_idx]
            strides, padding = self.params.cnn.strides[cnn_idx], self.params.cnn.padding[cnn_idx]
            dilation_rate = self.params.cnn.dilation_rate[cnn_idx]
            model_cnn.add(K.layers.Conv1D(
                # Modified `Conv1D` layer parameters.
                filters=out_channels, kernel_size=kernel_size, strides=strides,
                padding=padding, dilation_rate=dilation_rate, name="Conv1D_{:d}".format(cnn_idx),
                # Default `Conv1D` layer parameters.
                data_format="channels_last", groups=1, activation=None, use_bias=True,
                kernel_initializer="glorot_uniform", bias_initializer="zeros", kernel_regularizer=None,
                bias_regularizer=None, activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            ))
            # Initialize `MaxPool1D` layer.
            if isinstance(self.params.cnn.d_pooling_kernel, list):
                kernel_size = self.params.cnn.d_pooling_kernel[cnn_idx]
                model_cnn.add(K.layers.MaxPool1D(
                    # Modified `MaxPool1D` layer parameters.
                    pool_size=kernel_size, strides=1, name="MaxPool1D_{:d}".format(cnn_idx),
                    # Default `MaxPool1D` layer parameters.
                    padding="valid", data_format="channels_last"
                ))
            else:
                # Only add `MaxPool1D` layer at the last layer of cnn.
                if cnn_idx == len(self.params.cnn.n_filters) - 1:
                    kernel_size = self.params.cnn.d_pooling_kernel
                    model_cnn.add(K.layers.MaxPool1D(
                        # Modified `MaxPool1D` layer parameters.
                        pool_size=kernel_size, name="MaxPool1D_{:d}".format(cnn_idx),
                        # Default `MaxPool1D` layer parameters.
                        strides=None, padding="valid", data_format="channels_last"
                    ))
        # Add `Dropout` after `MaxPool1D` layer.
        if self.params.cnn.dropout > 0.:
            model_cnn.add(K.layers.Dropout(rate=self.params.cnn.dropout, name="Dropout_{}".format("cnn")))
        # Add `BatchNormalization` at the last layer of cnn layers.
        model_cnn.add(K.layers.BatchNormalization(
            # Modified `BatchNormalization` parameters.
            name="BatchNormalization_{}".format("cnn"),
            # Default `BatchNormalization` parameters.
            axis=-1, momentum=0.99, epsilon=0.001, center=True, scale=True, beta_initializer="zeros",
            gamma_initializer="ones", moving_mean_initializer="zeros", moving_variance_initializer="ones",
            beta_regularizer=None, gamma_regularizer=None, beta_constraint=None, gamma_constraint=None
        ))
        ## Initialize trainable fc layer. Then add FullConnect layer to do classification task.
        model_fc = K.models.Sequential(name="FullConnect")
        # Flatten convolved features to 1D-vector.
        model_fc.add(K.layers.Flatten(data_format="channels_last"))
        # Add hidden `Dense` layers.
        for d_hidden_i in self.params.fc.d_hidden:
             model_fc.add(K.layers.Dense(
                # Modified `Dense` parameters.
                units=d_hidden_i, activation="relu",
                # Default `Dense` parameters.
                use_bias=True, kernel_initializer="glorot_uniform", bias_initializer="zeros", kernel_regularizer=None,
                bias_regularizer=None, activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            ))
        # Add `Dropout` after hidden `Dense` layer.
        if self.params.fc.dropout > 0.:
            model_fc.add(K.layers.Dropout(rate=self.params.fc.dropout, name="Dropout_{}".format("fc")))
        # Add the final classification `Dense` layer.
        model_fc.add(K.layers.Dense(
            # Modified `Dense` parameters.
            units=self.params.fc.d_output, activation="sigmoid",
            # Default `Dense` parameters.
            use_bias=True, kernel_initializer="glorot_uniform", bias_initializer="zeros", kernel_regularizer=None,
            bias_regularizer=None, activity_regularizer=None, kernel_constraint=None, bias_constraint=None
        )); model_fc.add(K.layers.Softmax(axis=-1))
        ## Get the final `model`.
        model = K.models.Sequential([model_cnn, model_fc,])
        # Return the final `model`.
        return model

    """
    network funcs
    """
    # def call func
    @utils.model.tf_scope
    def call(self, inputs, training=None, mask=None):
        """
        Forward `cnn_ensemble` to get the final predictions.

        Args:
            inputs: (2[tuple],) - The dict of inputs, containing [X,y_true].
                X: (batch_size, seq_len, n_chennals) - The input data.
                y_true: (batch_size, n_labels) - The one-hot labels.
            training: Boolean or boolean scalar tensor, indicating whether to run
                the `Network` in training mode or inference mode.
            mask: A mask or list of masks. A mask can be either a tensor or None (no mask).

        Returns:
            y_pred: (n_models, batch_size, n_labels) - The predicted labels ensembled across models.
            loss: tf.float32 - The corresponding loss.
        """
        # Initialize `X` & `y_true` from `inputs`.
        X = inputs[0]; y_true = inputs[1]
        # Forward models defined in ensemble.
        # y_pred - (n_models, batch_size, n_labels)
        y_pred = tf.stack([model_i(X) for model_i in self.models], axis=0)
        # Calculate the binary cross entropy loss corresponding to each model.
        # loss_bce - (n_models,)
        loss_bce = tf.reduce_mean(tf.stack([self._loss_bce(y_pred_i, y_true)\
            for y_pred_i in tf.unstack(y_pred, axis=0)], axis=0), axis=-1)
        # Ensemble the predicted labels corresponding to ensemble type.
        # If use `average` ensemble type, directly average all prediction.
        # TODO: Average prediction before softmax to avoid miss leading.
        if self.params.ensemble_type == "average":
            # y_pred - (batch_size, n_labels)
            y_pred = tf.reduce_mean(y_pred, axis=0)
        # If use `min_loss` ensemble type, directly return the prediction with minimum loss according to target.
        # Note: This ensemble type may lead to information leak, we should not use this as the final prediction.
        elif self.params.ensemble_type == "min_loss":
            # y_pred - (batch_size, n_labels)
            y_pred = y_pred[tf.argmin(loss_bce),:,:]
        else:
            raise ValueError("ERROR: Get unknown ensemble type ({}).".format(self.params.ensemble_type))
        # Calculate the final `loss`. Every model has the same chance to approximate target.
        # loss - tf.float32
        loss = tf.reduce_mean(loss_bce)
        # Return the final `y_pred` & `loss`.
        return y_pred, loss

    # def _loss_bce func
    @utils.model.tf_scope
    def _loss_bce(self, value, target):
        """
        Calculates binary cross entropy between tensors value and target.
        Get mean over last dimension to keep losses of different batches separate.
        :param value: (batch_size,) - Value of the object.
        :param target: (batch_size,) - Target of the object.
        :return loss: (batch_size,) - Loss between value and target.
        """
        # Note: `tf.nn.softmax_cross_entropy_with_logits` needs unscaled log probabilities,
        # we must not add `tf.nn.Softmax` layer at the last of the model.
        # loss - (batch_size,)
        loss = tf.nn.softmax_cross_entropy_with_logits(labels=target,logits=value) if type(value) is not list else\
            [tf.nn.softmax_cross_entropy_with_logits(labels=target[i],logits=value[i]) for i in range(len(value))]
        # Return the final `loss`.
        return loss

    """
    tool funcs
    """
    # def summary func
    @utils.model.tf_scope
    def summary(self, print_fn=None):
        """
        Summary built model.

        Args:
            print_fn: callable - Print function to use. Defaults to `print`. It will be called on each
                line of the summary. You can set it to a custom function in order to capture the string summary.

        Returns:
            None
        """
        self.models[0].summary(
            # Modified summary parameters.
            print_fn=print_fn,
            # Default summary parameters.
            line_length=None, positions=None, expand_nested=True, show_trainable=True, layer_range=None
        )

if __name__ == "__main__":
    # local dep
    from params.cnn_ensemble_params import cnn_ensemble_params

    # Initialize training process.
    utils.model.set_seeds(42)
    # Initialize params.
    batch_size = 32; seq_len = 100; dataset = "eeg_anonymous"
    cnn_ensemble_params_inst = cnn_ensemble_params(dataset=dataset)
    n_channels = cnn_ensemble_params_inst.model.n_channels; n_labels = cnn_ensemble_params_inst.model.n_labels
    # Get cnn_ensemble_inst.
    cnn_ensemble_inst = cnn_ensemble(cnn_ensemble_params_inst.model)
    # Initialize inputs.
    X = tf.random.uniform((batch_size, seq_len, n_channels), dtype=tf.float32)
    y_pred = tf.cast(np.eye(n_labels)[np.random.randint(0, n_labels, size=(batch_size,))], dtype=tf.float32)
    # Forward cnn_ensemble_inst.
    y_pred, loss = cnn_ensemble_inst((X, y_pred))

