#!/usr/bin/env python3
"""
Created on 17:09, May. 17th, 2023

@author: Anonymous
"""
import copy as cp
import tensorflow as tf
import tensorflow.keras as K
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.path.join(os.pardir, os.pardir))
    from layers import *
else:
    from .layers import *
import utils.model

__all__ = [
    "conv_net",
    "conv_ensemble_net",
]

# def conv_net class
class conv_net(K.Model):
    """
    `conv_net` model, which considers time information.
    """

    def __init__(self, params, **kwargs):
        """
        Initialize `conv_net` object.

        Args:
            params: Model parameters initialized by conv_net_params, updated by params.iteration.
            kwargs: The arguments related to initialize `tf.keras.Model`-style object.

        Returns:
            None
        """
        # First call super class init function to set up `K.Model`
        # style model and inherit it's functionality.
        super(conv_net, self).__init__(**kwargs)

        # Copy hyperparameters (e.g. network sizes) from parameter dotdict,
        # usually generated from conv_net_params() in params/conv_net_params.py.
        self.params = cp.deepcopy(params)

        # Create trainable vars.
        self._init_trainable()

    """
    init funcs
    """
    # def _init_trainable func
    def _init_trainable(self):
        """
        Initialize trainable variables.

        Args:
            None

        Returns:
            None
        """
        ## Construct subject block.
        self.subject_block = SubjectBlock(self.params.subject)
        ## Construct CNN block.
        # Initialize CNN block.
        self.conv_block = K.models.Sequential(name="CNN")
        # Add `Conv1D` layers.
        for cnn_idx in range(len(self.params.cnn.n_filters)):
            # Initialize `Conv1D` layer.
            n_filters = self.params.cnn.n_filters[cnn_idx]; kernel_size = self.params.cnn.kernel_size[cnn_idx]
            padding = self.params.cnn.padding[cnn_idx]; dilation_rate = self.params.cnn.dilation_rate[cnn_idx]
            kernel_initializer = self.params.cnn.kernel_initializer[cnn_idx]
            self.conv_block.add(K.layers.Conv1D(
                # Modified `Conv1D` layer parameters.
                filters=n_filters, kernel_size=kernel_size, padding=padding,
                dilation_rate=dilation_rate, kernel_initializer=kernel_initializer,
                # Default `Conv1D` layer parameters.
                strides=1, data_format="channels_last", groups=1, activation=None, use_bias=True,
                bias_initializer="zeros", kernel_regularizer=None, bias_regularizer=None,
                activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            ))
        # Add `MaxPool1D` layer.
        self.conv_block.add(K.layers.MaxPool1D(
            # Modified `MaxPool1D` layer parameters.
            pool_size=self.params.cnn.pool_size,
            # Default `MaxPool1D` layer parameters.
            strides=None, padding="valid", data_format="channels_last"
        ))
        # Add `Dropout` layer.
        self.conv_block.add(K.layers.Dropout(rate=self.params.cnn.dropout_ratio))
        # Add `BatchNormalization` layer.
        self.conv_block.add(K.layers.BatchNormalization(
            # Default `BatchNormalization` parameters.
            axis=-1, momentum=0.99, epsilon=0.001, center=True, scale=True, beta_initializer="zeros",
            gamma_initializer="ones", moving_mean_initializer="zeros", moving_variance_initializer="ones",
            beta_regularizer=None, gamma_regularizer=None, beta_constraint=None, gamma_constraint=None
        ))
        # Initialize feature block.
        self.feature_block = K.models.Sequential(name="feature")
        self.feature_block.add(K.layers.Flatten())
        # Initialize contrastive block.
        self.contrastive_block = LossLayer(d_contra=self.params.d_contra, loss_mode="clip_orig")
        # Initialize classification block.
        self.classification_block = K.models.Sequential(name="classification")
        self.classification_block.add(K.layers.Dense(
            # Modified `Dense` layer parameters.
            units=self.params.n_labels, kernel_initializer="he_uniform",
            # Default `Dense` layer parameters.
            activation=None, use_bias=True, bias_initializer="zeros", kernel_regularizer=None,
            bias_regularizer=None, activity_regularizer=None, kernel_constraint=None, bias_constraint=None
        )); self.classification_block.add(K.layers.Softmax(axis=-1))

    """
    network funcs
    """
    # def call func
    def call(self, inputs, training=None, mask=None):
        """
        Forward `conv_net` to get the final predictions.

        Args:
            inputs: tuple - The input data.
            training: Boolean or boolean scalar tensor, indicating whether to run
                the `Network` in training mode or inference mode.
            mask: A mask or list of masks. A mask can be either a tensor or None (no mask).

        Returns:
            y_pred: (batch_size, n_labels) - The output labels.
            loss: tf.float32 - The corresponding loss.
        """
        # Initialize components of inputs.
        # X - (batch_size, seq_len, n_channels)
        # y - (batch_size,)
        # Y_f - (batch_size, seq_len, n_channels)
        X = inputs[0]; y_true = inputs[1]; Y_f = inputs[2]; subj_id = inputs[3]
        # Forward convolution layer to get extracted feature maps, then aggregate to get features.
        # X_f - (batch_size, n_features)
        X_f = self.feature_block(self.conv_block(self.subject_block((X, subj_id))))
        # Forward contrastive block to calculate contrastive loss.
        # loss_contra - tf.float32
        # contra_matrix - (batch_size, batch_size), the first dimension is z, and the second dimension is y.
        loss_contra, contra_matrix = self.contrastive_block((X_f, Y_f))
        # If use `max_z` prediction mode, assign each label with the data item with max similarity.
        if self.params.contra_pred_mode == "max_z":
            # contra_idxs - (batch_size,), y_pred_contra - (batch_size, n_labels)
            contra_idxs = tf.squeeze(tf.argmax(contra_matrix, axis=0))
            y_pred_contra = tf.gather(y_true, indices=contra_idxs, axis=0)
        # If use `prob_z` prediction mode, assign each label with the weighted probability of all data items.
        elif self.params.contra_pred_mode == "prob_z":
            # contra_prob - (batch_size, batch_size), y_pred_contra - (batch_size, n_labels)
            contra_prob = tf.transpose(contra_matrix / tf.reduce_sum(contra_matrix, axis=0, keepdims=True), perm=[1,0])
            y_pred_contra = tf.matmul(contra_prob, y_true)
        # If use `max_y` prediction mode, assign each data item with the label with max similarity.
        elif self.params.contra_pred_mode == "max_y":
            # contra_idxs - (batch_size,), y_pred_contra - (batch_size, n_labels)
            contra_idxs = tf.squeeze(tf.argmax(contra_matrix, axis=-1))
            y_pred_contra = tf.gather(y_true, indices=contra_idxs, axis=0)
        # If use `prob_y` prediction mode, assign each data item with the weighted probability of all labels.
        elif self.params.contra_pred_mode == "prob_y":
            # contra_prob - (batch_size, batch_size), y_pred_contra - (batch_size, n_labels)
            contra_prob = contra_matrix / tf.reduce_sum(contra_matrix, axis=-1, keepdims=True)
            y_pred_contra = tf.matmul(contra_prob, y_true)
        # Get unknown contrastive prediction mode.
        else:
            raise ValueError((
                "ERROR: Unknown contrastive prediction mode {} in train.conv_net."
            ).format(self.params.contra_pred_mode))
        # Forward classification block to do classification.
        # Note: The layer before the final fc layers is considered as feature vectors.
        # y_pred_bce - (batch_size, n_labels)
        y_pred_bce = self.classification_block(X_f)
        # Calculate the binary cross entropy loss.
        # loss_bce - tf.float32
        loss_bce = tf.reduce_mean(self._loss_bce(y_pred_bce, y_true))
        # Calculate the final `y_pred` & `loss`.
        # y_pred - (batch_size, n_labels); loss - tf.float32
        y_pred = (1 - self.params.contra_pred_ratio) * y_pred_bce + self.params.contra_pred_ratio * y_pred_contra
        loss = (1 - self.params.contra_loss_ratio) * loss_bce +\
            self.params.contra_loss_ratio * self.params.contra_loss_scale * loss_contra
        # Return the final `outputs` & `loss`.
        return y_pred, loss

    # def _loss_bce func
    @utils.model.tf_scope
    def _loss_bce(self, value, target):
        """
        Calculates binary cross entropy between tensors value and target.
        Get mean over last dimension to keep losses of different batches separate.
        :param value: (batch_size,) - Value of the object.
        :param target: (batch_size,) - Target of the object.
        :return loss: (batch_size,) - Loss between value and target.
        """
        # Note: `tf.nn.softmax_cross_entropy_with_logits` needs unscaled log probabilities,
        # we must not add `tf.nn.Softmax` layer at the last of the model.
        # loss - (batch_size,)
        loss = tf.nn.softmax_cross_entropy_with_logits(labels=target,logits=value) if type(value) is not list else\
            [tf.nn.softmax_cross_entropy_with_logits(labels=target[i],logits=value[i]) for i in range(len(value))]
        # Return the final `loss`.
        return loss

    """
    tool funcs
    """
    # def summary func
    @utils.model.tf_scope
    def summary(self, print_fn=None):
        """
        Summary built model.
        :param print_fn: callable - Print function to use. Defaults to `print`. It will be called on each
            line of the summary. You can set it to a custom function in order to capture the string summary.
        """
        super(conv_net, self).summary(
            # Modified summary parameters.
            print_fn=print_fn,
            # Default summary parameters.
            line_length=None, positions=None, expand_nested=True, show_trainable=True, layer_range=None
        )

# def conv_ensemble_net class
class conv_ensemble_net(K.Model):
    """
    `conv_ensemble_net` model, which considers time information.
    """

    def __init__(self, params, **kwargs):
        """
        Initialize `conv_ensemble_net` object.

        Args:
            params: Model parameters initialized by conv_net_params, updated by params.iteration.
            kwargs: The arguments related to initialize `tf.keras.Model`-style object.

        Returns:
            None
        """
        # First call super class init function to set up `K.Model`
        # style model and inherit it's functionality.
        super(conv_ensemble_net, self).__init__(**kwargs)

        # Copy hyperparameters (e.g. network sizes) from parameter dotdict,
        # usually generated from conv_net_params() in params/conv_net_params.py.
        self.params = cp.deepcopy(params)

        # Create trainable vars.
        self._init_trainable()

    """
    init funcs
    """
    # def _init_trainable func
    def _init_trainable(self):
        """
        Initialize trainable variables.

        Args:
            None

        Returns:
            None
        """
        self.models = [conv_net(self.params) for _ in range(self.params.n_models)]

    """
    network funcs
    """
    # def call func
    @utils.model.tf_scope
    def call(self, inputs, training=None, mask=None):
        """
        Forward `conv_ensemble_net` to get the final predictions.

        Args:
            inputs: tuple - The input data.
            training: Boolean or boolean scalar tensor, indicating whether to run
                the `Network` in training mode or inference mode.
            mask: A mask or list of masks. A mask can be either a tensor or None (no mask).

        Returns:
            y_pred: (batch_size, n_labels) - The output labels.
            loss: tf.float32 - The corresponding loss.
        """
        # Forward models defined in ensemble.
        # y_preds - (n_models, batch_size, n_labels)
        y_preds = []; losses = []
        for model_i in self.models:
            y_pred_i, loss_i = model_i(inputs, training=training)
            y_preds.append(y_pred_i); losses.append(loss_i)
        y_preds = tf.stack(y_preds, axis=0); losses = tf.stack(losses, axis=0)
        # Ensemble the predicted labels corresponding to ensemble type.
        # If use `average` ensemble type, directly average all prediction.
        # TODO: Average prediction before softmax to avoid miss leading.
        if self.params.ensemble_type == "average":
            # y_pred - (batch_size, n_labels)
            y_pred = tf.reduce_mean(y_preds, axis=0)
        # If use `min_loss` ensemble type, directly return the prediction with minimum loss according to target.
        # Note: This ensemble type may lead to information leak, we should not use this as the final prediction.
        elif self.params.ensemble_type == "min_loss":
            # y_pred - (batch_size, n_labels)
            y_pred = y_preds[tf.argmin(losses),:,:]
        else:
            raise ValueError("ERROR: Get unknown ensemble type ({}).".format(self.params.ensemble_type))
        # Calculate the final `loss`. Every model has the same chance to approximate target.
        # loss - tf.float32
        loss = tf.reduce_mean(losses)
        # Return the final `y_pred` & `loss`.
        return y_pred, loss

    """
    tool funcs
    """
    # def summary func
    @utils.model.tf_scope
    def summary(self, print_fn=None):
        """
        Summary built model.
        :param print_fn: callable - Print function to use. Defaults to `print`. It will be called on each
            line of the summary. You can set it to a custom function in order to capture the string summary.
        """
        self.models[0].summary(print_fn=print_fn)

if __name__ == "__main__":
    import numpy as np
    # local dep
    from params.conv_net_params import conv_net_params

    # macro
    dataset = "meg_anonymous"; batch_size = 16; seq_len = 160; n_features = 128; n_subjects = 64

    # Initialize training process.
    utils.model.set_seeds(42)

    ## Instantiate params.
    conv_net_params_inst = conv_net_params(dataset=dataset)
    n_channels = conv_net_params_inst.model.n_channels; n_labels = conv_net_params_inst.model.n_labels
    ## Evaluate conv_net.
    # Instantiate conv_net.
    conv_net_inst = conv_net(conv_net_params_inst.model)
    # Initialize input X & label y_true & representation Y_f.
    X = tf.random.normal((batch_size, seq_len, n_channels), dtype=tf.float32)
    y_true = tf.cast(tf.one_hot(tf.cast(tf.range(batch_size), dtype=tf.int64), n_labels), dtype=tf.float32)
    Y_f = tf.random.normal((batch_size, n_features), dtype=tf.float32)
    subj_id = tf.cast(np.eye(n_subjects)[np.random.randint(0, n_subjects, size=(batch_size,))], dtype=tf.float32)
    # Forward layers in `conv_net_inst`.
    y_pred, loss = conv_net_inst((X, y_true, Y_f, subj_id)); conv_net_inst.summary()
    ## Evaluate conv_ensemble_net.
    # Instantiate conv_ensemble_net.
    conv_ensemble_net_inst = conv_ensemble_net(conv_net_params_inst.model)
    # Forward layers in `conv_ensemble_net_inst`.
    y_pred, loss = conv_ensemble_net_inst((X, y_true, Y_f, subj_id)); conv_ensemble_net_inst.summary()

