#!/usr/bin/env python3
"""
Created on 14:38, Jun. 28th, 2023

@author: Anonymous
"""
import copy as cp
import tensorflow as tf
import tensorflow.keras as K
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.path.join(os.pardir, os.pardir))
    from layers import *
else:
    from .layers import *
import utils.model

__all__ = [
    "naive_transformer",
]

class naive_transformer(K.Model):
    """
    `naive_transformer` model, with considering time information.
    """

    def __init__(self, params, **kwargs):
        """
        Initialize `naive_transformer` object.

        Args:
            params: Model parameters initialized by naive_transformer_params, updated by params.iteration.
            kwargs: The arguments related to initialize `tf.keras.Model`-style object.

        Returns:
            None
        """
        # First call super class init function to set up `K.Model`
        # style model and inherit it's functionality.
        super(naive_transformer, self).__init__(**kwargs)

        # Copy hyperparameters (e.g. network sizes) from parameter dotdict,
        # usually generated from naive_transformer_params() in params/naive_transformer_params.py.
        self.params = cp.deepcopy(params)

        # Create trainable vars.
        self._init_trainable()

    """
    init funcs
    """
    # def _init_trainable func
    def _init_trainable(self):
        """
        Initialize trainable variables.

        Args:
            None

        Returns:
            None
        """
        # Initialize the embedding layer for input.
        # emb_input - (batch_size, seq_len, d_input) -> (batch_size, seq_len, d_model)
        self.emb_input = K.layers.Dense(
            # Modified `Dense` layer parameters.
            units=self.params.d_model,
            kernel_initializer=K.initializers.random_normal(mean=0., stddev=0.01),
            bias_initializer=K.initializers.constant(value=0.01),
            # Default `Dense` layer parameters.
            activation=None, use_bias=True, kernel_regularizer=None, bias_regularizer=None,
            activity_regularizer=None, kernel_constraint=None, bias_constraint=None
        )
        # Initializez the position embedding layer.
        # emb_pos - (batch_size, seq_len, d_model) -> (batch_size, seq_len, d_model)
        self.emb_pos = PositionEmbedding(self.params.max_len)
        # Initialize encoder block.
        # encoder - (batch_size, seq_len, d_model) -> (batch_size, seq_len, d_model)
        self.encoder = K.models.Sequential(layers=[TransformerBlock(
            n_heads=self.params.n_heads, d_head=self.params.d_head, mha_dropout_prob=self.params.mha_dropout_prob,
            d_ff=self.params.d_ff, ff_dropout_prob=self.params.ff_dropout_prob) for _ in range(self.params.encoder_depth)
        ], name="encoder")
        # Initialize fc block.
        self.fc = K.models.Sequential([
            K.layers.Flatten(),
            K.layers.Dense(
                # Modified `Dense` layer parameters.
                units=self.params.d_fc, activation="relu",
                kernel_initializer=K.initializers.random_normal(mean=0., stddev=0.01),
                bias_initializer=K.initializers.constant(value=0.01),
                # Default `Dense` layer parameters.
                use_bias=True, kernel_regularizer=None, bias_regularizer=None,
                activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            ),
            K.layers.Dropout(rate=self.params.fc_dropout_prob),
            K.layers.Dense(
                # Modified `Dense` layer parameters.
                units=self.params.n_labels,
                kernel_initializer=K.initializers.random_normal(mean=0., stddev=0.01),
                bias_initializer=K.initializers.constant(value=0.01),
                # Default `Dense` layer parameters.
                activation=None, use_bias=True, kernel_regularizer=None, bias_regularizer=None,
                activity_regularizer=None, kernel_constraint=None, bias_constraint=None
            ), K.layers.Softmax(axis=-1),
        ], name="fc")

    """
    network funcs
    """
    # def call func
    def call(self, inputs, training=None, mask=None):
        """
        Forward `naive_transformer` to get the final predictions.

        Args:
            inputs: tuple - The input data.
            training: Boolean or boolean scalar tensor, indicating whether to run
                the `Network` in training mode or inference mode.
            mask: A mask or list of masks. A mask can be either a tensor or None (no mask).

        Returns:
            outputs: (batch_size, n_labels) - The output labels.
            loss: tf.float32 - The corresponding loss.
        """
        # Initialize components of inputs.
        # X - (batch_size, seq_len, n_channels)
        # y - (batch_size,)
        X = inputs[0]; y_true = inputs[1]
        # Get the embedding of the original signal `X`, and add `emb_pos`.
        # X_emb - (batch_size, seq_len, d_model)
        X_emb = self.emb_pos(self.emb_input(X))
        # Forward `encoder` to get the encoder-transformed embedding.
        X_emb = self.encoder(X_emb)
        # Forward `fc` to get the final prediction.
        # y_pred - (batch_size, n_labels)
        y_pred = self.fc(X_emb)
        # Calculate the binary cross entropy loss.
        # loss - tf.float32
        loss = tf.reduce_mean(self._loss_bce(y_pred, y_true))
        # Return the final `outputs` & `loss`.
        return y_pred, loss

    # def _loss_bce func
    @utils.model.tf_scope
    def _loss_bce(self, value, target):
        """
        Calculates binary cross entropy between tensors value and target.
        Get mean over last dimension to keep losses of different batches separate.
        :param value: (batch_size,) - Value of the object.
        :param target: (batch_size,) - Target of the object.
        :return loss: (batch_size,) - Loss between value and target.
        """
        # Note: `tf.nn.softmax_cross_entropy_with_logits` needs unscaled log probabilities,
        # we must not add `tf.nn.Softmax` layer at the last of the model.
        # loss - (batch_size,)
        loss = tf.nn.softmax_cross_entropy_with_logits(labels=target,logits=value) if type(value) is not list else\
            [tf.nn.softmax_cross_entropy_with_logits(labels=target[i],logits=value[i]) for i in range(len(value))]
        # Return the final `loss`.
        return loss

    """
    tool funcs
    """
    # def summary func
    @utils.model.tf_scope
    def summary(self, print_fn=None):
        """
        Summary built model.
        :param print_fn: callable - Print function to use. Defaults to `print`. It will be called on each
            line of the summary. You can set it to a custom function in order to capture the string summary.
        """
        super(naive_transformer, self).summary(print_fn=print_fn)

if __name__ == "__main__":
    import numpy as np
    # local dep
    from params.naive_transformer_params import naive_transformer_params

    # macro
    dataset = "meg_anonymous"; batch_size = 64; seq_len = 80; n_channels = 204

    # Initialize training process.
    utils.model.set_seeds(42)

    # Instantiate params.
    naive_transformer_params_inst = naive_transformer_params(dataset=dataset)
    n_labels = naive_transformer_params_inst.model.n_labels
    # Instantiate naive_transformer.
    naive_transformer_inst = naive_transformer(naive_transformer_params_inst.model)
    # Initialize input X & label y_true.
    X = tf.random.normal((batch_size, seq_len, n_channels), dtype=tf.float32)
    y_true = tf.cast(tf.one_hot(tf.cast(tf.range(batch_size), dtype=tf.int64), n_labels), dtype=tf.float32)
    # Forward layers in `naive_transformer_inst`.
    outputs, loss = naive_transformer_inst((X, y_true)); naive_transformer_inst.summary()

