#!/usr/bin/env python3
"""
Created on 14:50, Jul. 24th, 2023

@author: Anonymous
"""
import numpy as np
import tensorflow as tf
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.pardir)
from utils import DotDict

__all__ = [
    "domain_adversarial_conv_params",
    "domain_adversarial_transformer_params",
    "subdomain_adversarial_conv_params",
    "subdomain_adversarial_transformer_params",
    "domain_contrastive_conv_params",
    "subdomain_contrastive_conv_params",
]

# def domain_adversarial_conv_params class
class domain_adversarial_conv_params(DotDict):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `DomainAdversarialConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `domain_adversarial_conv_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(domain_adversarial_conv_params, self).__init__()

        ## Generate all parameters hierarchically.
        # -- Model parameters
        self.model = domain_adversarial_conv_params._gen_model_params(dataset)
        # -- Training parameters
        self.train = domain_adversarial_conv_params._gen_train_params(dataset)

        ## Do init iteration.
        domain_adversarial_conv_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Model parameters
        # Calculate current learning rate.
        self.model.lr_i = self.model.lr

    """
    generate funcs
    """
    ## def _gen_model_* funcs
    # def _gen_model_params func
    @staticmethod
    def _gen_model_params(dataset):
        """
        Generate model parameters.
        """
        # Initialize `model_params`.
        model_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        model_params.dataset = dataset
        # Precision parameter.
        model_params.precision = getattr(tf, domain_adversarial_conv_params._precision)\
            if hasattr(tf, domain_adversarial_conv_params._precision) else tf.float32
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The size of input channels.
            model_params.n_channels = 55
            # The size of output classes.
            model_params.n_labels = 15
            # The size of output domains.
            model_params.n_domains = 2
            # The weight factor used to scale domain loss.
            model_params.w_loss_domain = 10.
            # The learning rate of optimizer.
            model_params.lr = 3e-4
        # Normal parameters related to other dataset.
        else:
            # The size of input channels.
            model_params.n_channels = 55
            # The size of output classes.
            model_params.n_labels = 15
            # The size of output domains.
            model_params.n_domains = 2
            # The weight factor used to scale domain loss.
            model_params.w_loss_domain = 1.
            # The learning rate of optimizer.
            model_params.lr = 3e-4
        ## -- CNN parameters
        model_params.cnn = domain_adversarial_conv_params._gen_model_cnn_params(model_params)
        ## -- Category classifier parameters
        model_params.cls_class = domain_adversarial_conv_params._gen_model_cls_class_params(model_params)
        ## -- Domain classifier parameters
        model_params.cls_domain = domain_adversarial_conv_params._gen_model_cls_domain_params(model_params)

        # Return the final `model_params`.
        return model_params

    # def _gen_model_cnn_params func
    @staticmethod
    def _gen_model_cnn_params(model_params):
        """
        Generate model.cnn parameters
        """
        # Initialize `model_cnn_params`.
        model_cnn_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimension of input vector.
            model_cnn_params.d_input = model_params.n_channels
            # The number of filters of each CNN layer.
            model_cnn_params.n_filters = [256, 128]
            # The size of kernel of each CNN layer.
            model_cnn_params.d_kernel = [9, 11]
            # The length of stride of each CNN layer.
            model_cnn_params.strides = [1, 1]
            # The length of padding of each CNN layer.
            model_cnn_params.padding = ["same", "same"]
            # The dilation rate of each CNN layer.
            model_cnn_params.dilation_rate = [1, 2]
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel of each CNN layer.
            model_cnn_params.d_pooling_kernel = 2
            ## -- Normal parameters (related to Dropout)
            # The dropout rate of dropout layer.
            model_cnn_params.dropout = 0.5
        # Normal parameters related to other dataset.
        else:
            # The dimension of input vector.
            model_cnn_params.d_input = model_params.n_channels
            # The number of filters of each CNN layer.
            model_cnn_params.n_filters = [256, 128]
            # The size of kernel of each CNN layer.
            model_cnn_params.d_kernel = [9, 11]
            # The length of stride of each CNN layer.
            model_cnn_params.strides = [1, 1]
            # The length of padding of each CNN layer.
            model_cnn_params.padding = ["same", "same"]
            # The dilation rate of each CNN layer.
            model_cnn_params.dilation_rate = [1, 2]
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel of each CNN layer.
            model_cnn_params.d_pooling_kernel = 2
            ## -- Normal parameters (related to Dropout)
            # The dropout rate of dropout layer.
            model_cnn_params.dropout = 0.5

        # Return the final `model_cnn_params`.
        return model_cnn_params

    # def _gen_model_cls_class_params func
    @staticmethod
    def _gen_model_cls_class_params(model_params):
        """
        Generate model.cls_class parameters.
        """
        # Initialize `model_cls_class_params`.
        model_cls_class_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of hidden layers.
            model_cls_class_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_class_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of hidden layers.
            model_cls_class_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_class_params.dropout = 0.
        # The dimension of output vector.
        model_cls_class_params.d_output = model_params.n_labels

        # Return the final `model_cls_class_params`.
        return model_cls_class_params

    # def _gen_model_cls_domain_params func
    @staticmethod
    def _gen_model_cls_domain_params(model_params):
        """
        Generate model.cls_domain parameters.
        """
        # Initialize `model_cls_domain_params`.
        model_cls_domain_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimension of hidden layers.
            model_cls_domain_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_domain_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimension of hidden layers.
            model_cls_domain_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_domain_params.dropout = 0.
        # The dimension of output vector.
        model_cls_domain_params.d_output = model_params.n_domains

        # Return the final `model_cls_domain_params`.
        return model_cls_domain_params

    ## def _gen_train_* funcs
    # def _gen_train_params func
    @staticmethod
    def _gen_train_params(dataset):
        """
        Generate train parameters.
        """
        # Initialize `train_params`.
        train_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        train_params.dataset = dataset
        # Precision parameter.
        train_params.precision = getattr(tf, domain_adversarial_conv_params._precision)\
            if hasattr(tf, domain_adversarial_conv_params._precision) else tf.float32
        # Whether use graph mode or eager mode.
        train_params.use_graph_mode = True
        # The ratio of train dataset. The rest is test dataset.
        train_params.train_ratio = 0.8
        # Size of buffer used in shuffle.
        train_params.buffer_size = int(1e4)
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if train_params.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 256
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 128

        # Return the final `train_params`.
        return train_params

# def domain_adversarial_transformer_params class
class domain_adversarial_transformer_params(DotDict):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `DomainAdversarialConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `domain_adversarial_transformer_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(domain_adversarial_transformer_params, self).__init__()

        ## Generate all parameters hierarchically.
        # -- Model parameters
        self.model = domain_adversarial_transformer_params._gen_model_params(dataset)
        # -- Training parameters
        self.train = domain_adversarial_transformer_params._gen_train_params(dataset)

        ## Do init iteration.
        domain_adversarial_transformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Model parameters
        # Calculate current learning rate of class optimizer.
        lr_class_i = self.model.lr_class_min + (self.model.lr_class_max - self.model.lr_class_min) *\
            (iteration / self.model.warmup_epochs_class) if iteration < self.model.warmup_epochs_class else\
            self.model.lr_class_min + (self.model.lr_class_max - self.model.lr_class_min) * 0.5 *\
            (1. + np.cos(np.pi * (iteration - self.model.warmup_epochs_class) /\
                (self.train.n_epochs - self.model.warmup_epochs_class)))
        self.model.lr_class_i = tf.constant(lr_class_i, dtype=tf.float32)
        # Calculate current learning rate of domain optimizer.
        lr_domain_i = self.model.lr_domain_min + (self.model.lr_domain_max - self.model.lr_domain_min) *\
            (iteration / self.model.warmup_epochs_domain) if iteration < self.model.warmup_epochs_domain else\
            self.model.lr_domain_min + (self.model.lr_domain_max - self.model.lr_domain_min) * 0.5 *\
            (1. + np.cos(np.pi * (iteration - self.model.warmup_epochs_domain) /\
                (self.train.n_epochs - self.model.warmup_epochs_domain)))
        self.model.lr_domain_i = tf.constant(lr_domain_i, dtype=tf.float32)

    """
    generate funcs
    """
    ## def _gen_model_* funcs
    # def _gen_model_params func
    @staticmethod
    def _gen_model_params(dataset):
        """
        Generate model parameters.
        """
        # Initialize `model_params`.
        model_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        model_params.dataset = dataset
        # Precision parameter.
        model_params.precision = getattr(tf, domain_adversarial_transformer_params._precision)\
            if hasattr(tf, domain_adversarial_transformer_params._precision) else tf.float32
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The size of input channels.
            model_params.n_channels = 55
            # The size of output classes.
            model_params.n_labels = 15
            # The size of output domains.
            model_params.n_domains = 2
            # The weight factor used to scale domain loss.
            model_params.w_loss_domain = 0.5
            # The maximum learning rate of class optimizer.
            model_params.lr_class_max = 1e-4
            # The minimum learning rate of class optimizer.
            model_params.lr_class_min = 0.
            # The number of warm-up epochs for class optimizer.
            model_params.warmup_epochs_class = 50
            # The maximum learning rate of domain optimizer.
            model_params.lr_domain_max = 3e-4
            # The minimum learning rate of domain optimizer.
            model_params.lr_domain_min = 3e-4
            # The number of warm-up epochs for domain optimizer.
            model_params.warmup_epochs_domain = 50
        # Normal parameters related to other dataset.
        else:
            # The size of input channels.
            model_params.n_channels = 55
            # The size of output classes.
            model_params.n_labels = 15
            # The size of output domains.
            model_params.n_domains = 2
            # The weight factor used to scale domain loss.
            model_params.w_loss_domain = 1.
            # The learning rate of optimizer.
            model_params.lr = 3e-4
        ## -- CNN parameters
        model_params.encoder = domain_adversarial_transformer_params._gen_model_encoder_params(model_params)
        ## -- Category classifier parameters
        model_params.cls_class = domain_adversarial_transformer_params._gen_model_cls_class_params(model_params)
        ## -- Domain classifier parameters
        model_params.cls_domain = domain_adversarial_transformer_params._gen_model_cls_domain_params(model_params)

        # Return the final `model_params`.
        return model_params

    # def _gen_model_encoder_params func
    @staticmethod
    def _gen_model_encoder_params(model_params):
        """
        Generate model.encoder parameters.
        """
        # Initialize `model_encoder_params`.
        model_encoder_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the embedding.
            model_encoder_params.d_model = 64
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The depth of encoder.
            model_encoder_params.n_blocks = 2
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 32
            # The dropout probability of attention weights.
            model_encoder_params.mha_dropout_prob = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 64
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout_prob = [0., 0.]
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the embedding.
            model_encoder_params.d_model = 64
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The depth of encoder.
            model_encoder_params.n_blocks = 8
            # The number of attention heads.
            model_encoder_params.n_heads = 6
            # The dimensions of attention head.
            model_encoder_params.d_head = 32
            # The dropout probability of attention weights.
            model_encoder_params.mha_dropout_prob = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 64
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout_prob = [0., 0.5]
            # The dimensions of the hidden layer in fc block.
            model_encoder_params.d_fc = 128
            # The dropout probability of the hidden layer in fc block.
            model_encoder_params.fc_dropout_prob = 0.

        # Return the final `model_encoder_params`.
        return model_encoder_params

    # def _gen_model_cls_class_params func
    @staticmethod
    def _gen_model_cls_class_params(model_params):
        """
        Generate model.cls_class parameters.
        """
        # Initialize `model_cls_class_params`.
        model_cls_class_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of hidden layers.
            model_cls_class_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_class_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of hidden layers.
            model_cls_class_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_class_params.dropout = 0.
        # The dimension of output vector.
        model_cls_class_params.d_output = model_params.n_labels

        # Return the final `model_cls_class_params`.
        return model_cls_class_params

    # def _gen_model_cls_domain_params func
    @staticmethod
    def _gen_model_cls_domain_params(model_params):
        """
        Generate model.cls_domain parameters.
        """
        # Initialize `model_cls_domain_params`.
        model_cls_domain_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimension of hidden layers.
            model_cls_domain_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_domain_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimension of hidden layers.
            model_cls_domain_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_cls_domain_params.dropout = 0.
        # The dimension of output vector.
        model_cls_domain_params.d_output = model_params.n_domains

        # Return the final `model_cls_domain_params`.
        return model_cls_domain_params

    ## def _gen_train_* funcs
    # def _gen_train_params func
    @staticmethod
    def _gen_train_params(dataset):
        """
        Generate train parameters.
        """
        # Initialize `train_params`.
        train_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        train_params.dataset = dataset
        # Precision parameter.
        train_params.precision = getattr(tf, domain_adversarial_transformer_params._precision)\
            if hasattr(tf, domain_adversarial_transformer_params._precision) else tf.float32
        # Whether use graph mode or eager mode.
        train_params.use_graph_mode = True
        # The ratio of train dataset. The rest is test dataset.
        train_params.train_ratio = 0.8
        # Size of buffer used in shuffle.
        train_params.buffer_size = int(1e4)
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if train_params.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 256
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 128

        # Return the final `train_params`.
        return train_params

# def subdomain_adversarial_conv_params class
class subdomain_adversarial_conv_params(domain_adversarial_conv_params):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `SubdomainAdversarialConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `subdomain_adversarial_conv_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(subdomain_adversarial_conv_params, self).__init__(dataset)

        ## Update all parameters hierarchically.
        # -- Model parameters
        self._update_model_params()
        # -- Train parameters
        self._update_train_params()

        ## Do init iteration.
        subdomain_adversarial_conv_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Model parameters.
        # Calculate current learning rate.
        self.model.lr_class_i = self.model.lr_class
        # Calculate current learning rate.
        self.model.lr_domain_i = self.model.lr_domain
        # The weight factor used to scale domain loss.
        self.model.w_loss_domain_i = tf.constant(min(self.model.w_loss_domain_max,
            (iteration / self.model.i_w_loss_domain) * (self.model.w_loss_domain_max - self.model.w_loss_domain_min) +\
            self.model.w_loss_domain_min), dtype=self.model.precision)

    # def _update_model_params func
    def _update_model_params(self):
        """
        Update model parameters.
        """
        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if self.model.dataset == "eeg_anonymous":
            # The learning rate of class optimizer.
            self.model.lr_class = 3e-4
            # The learning rate of domain optimizer.
            self.model.lr_domain = 1e-3
            # The maximum weight factor used to scale domain loss.
            self.model.w_loss_domain_max = 10.
            # The minimum weight factor used to scale domain loss.
            self.model.w_loss_domain_min = 10.
            # The iteration to update weight factor used to scale domain loss.
            self.model.i_w_loss_domain = 200
            # The dimension of input vector.
            self.model.cnn.d_input = self.model.n_channels
            # The number of filters of each CNN layer.
            self.model.cnn.n_filters = [256, 128]
            # The size of kernel of each CNN layer.
            self.model.cnn.d_kernel = [9, 11]
            # The length of stride of each CNN layer.
            self.model.cnn.strides = [1, 1]
            # The length of padding of each CNN layer.
            self.model.cnn.padding = ["same", "same"]
            # The dilation rate of each CNN layer.
            self.model.cnn.dilation_rate = [1, 2]
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel of each CNN layer.
            self.model.cnn.d_pooling_kernel = 2
            ## -- Normal parameters (related to Dropout)
            # The dropout rate of dropout layer.
            self.model.cnn.dropout = 0.5
            # The dimensions of hidden layers.
            self.model.cls_class.d_hidden = []
            # The dropout rate of dropout layer.
            self.model.cls_class.dropout = 0.
            # The dimensions of hidden layers.
            self.model.cls_domain.d_hidden = [256, 256]
            # The dropout rate of dropout layer.
            self.model.cls_domain.dropout = 0.2
            # The size of output domains.
            self.model.n_domains = self.model.cls_domain.d_output = 2
        # Normal parameters related to other dataset.
        else:
            # The weight factor used to scale domain loss.
            self.model.w_loss_domain = 0.5

    # def _update_train_params func
    def _update_train_params(self):
        """
        Update train parameters.
        """
        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if self.train.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            self.train.n_epochs = 200
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            self.train.n_epochs = 200

# def subdomain_adversarial_transformer_params class
class subdomain_adversarial_transformer_params(domain_adversarial_transformer_params):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `SubdomainAdversarialTransformer` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `subdomain_adversarial_transformer_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(subdomain_adversarial_transformer_params, self).__init__(dataset)

        ## Update all parameters hierarchically.
        # -- Model parameters
        self._update_model_params()
        # -- Train parameters
        self._update_train_params()

        ## Do init iteration.
        subdomain_adversarial_transformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## Iterate super parameters.
        super(subdomain_adversarial_transformer_params, self).iteration(iteration)
        ## -- Model parameters.
        # The weight factor used to scale domain loss.
        self.model.w_loss_domain_i = tf.constant(min(self.model.w_loss_domain_max,
            (iteration / self.model.i_w_loss_domain) * (self.model.w_loss_domain_max - self.model.w_loss_domain_min) +\
            self.model.w_loss_domain_min), dtype=self.model.precision)

    # def _update_model_params func
    def _update_model_params(self):
        """
        Update model parameters.
        """
        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if self.model.dataset == "eeg_anonymous":
            # The maximum learning rate of class optimizer.
            self.model.lr_class_max = 1e-4
            # The minimum learning rate of class optimizer.
            self.model.lr_class_min = 0.
            # The number of warm-up epochs for class optimizer.
            self.model.warmup_epochs_class = 50
            # The maximum learning rate of domain optimizer.
            self.model.lr_domain_max = 3e-4
            # The minimum learning rate of domain optimizer.
            self.model.lr_domain_min = 3e-4
            # The number of warm-up epochs for domain optimizer.
            self.model.warmup_epochs_domain = 50
            # The maximum weight factor used to scale domain loss.
            self.model.w_loss_domain_max = .1
            # The minimum weight factor used to scale domain loss.
            self.model.w_loss_domain_min = .1
            # The iteration to update weight factor used to scale domain loss.
            self.model.i_w_loss_domain = 200
        # Normal parameters related to other dataset.
        else:
            # The weight factor used to scale domain loss.
            self.model.w_loss_domain = 0.5

    # def _update_train_params func
    def _update_train_params(self):
        """
        Update train parameters.
        """
        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if self.train.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            self.train.n_epochs = 200
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            self.train.n_epochs = 200

# def domain_contrastive_conv_params class
class domain_contrastive_conv_params(DotDict):
    """
    This contains one single object that generates a dictionary of parameters,
    which is provided to `DomainContrastiveConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `domain_contrastive_conv_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(domain_contrastive_conv_params, self).__init__()

        ## Generate all parameters hierarchically.
        # -- Model parameters
        self.model = domain_contrastive_conv_params._gen_model_params(dataset)
        # -- Training parameters
        self.train = domain_contrastive_conv_params._gen_train_params(self.model)

        ## Do init iteration.
        domain_contrastive_conv_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Train parameters
        # Calculate current learning rate.
        self.train.lr_i = self.train.lr

    """
    generate funcs
    """
    ## def _gen_model_* funcs
    # def _gen_model_params func
    @staticmethod
    def _gen_model_params(dataset):
        """
        Generate model parameters.
        """
        # Initialize `model_params`.
        model_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        model_params.dataset = dataset
        # Precision parameter.
        model_params.precision = getattr(tf, domain_contrastive_conv_params._precision)\
            if hasattr(tf, domain_contrastive_conv_params._precision) else tf.float32
        # The number of modalities.
        model_params.n_modalities = 3
        # The flag that indicates whether use siamese network.
        model_params.use_siamese = False
        # The mode of contrastive data format.
        model_params.contra_data_mode = ["point", "sequence"][0]
        # The dimension of contrastive layer.
        model_params.d_contra = 128 if model_params.contra_data_mode == "point" else 32
        # The mode of contrastive loss calculation.
        model_params.contra_loss_mode = ["clip", "clip_orig", "unicl"][-1]
        # The mode of contrastive prediction.
        model_params.contra_pred_mode = ["max_z", "prob_z", "max_y", "prob_y"][-1]
        # The ratio of contrastive prediction.
        model_params.contra_pred_ratio = 0.
        # The ratio of contrastive loss.
        model_params.contra_loss_ratio = 0.5
        # The scale factor of contrastive loss.
        model_params.contra_loss_scale = 0.1
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The size of input channels.
            model_params.n_channels = 55
            # The size of output classes.
            model_params.n_labels = 15
        # Normal parameters related to other dataset.
        else:
            # The size of input channels.
            model_params.n_channels = 55
            # The size of output classes.
            model_params.n_labels = 15
        ## -- CNN parameters
        model_params.cnn = domain_contrastive_conv_params._gen_model_cnn_params(model_params)
        ## -- Feature parameters
        model_params.feature = domain_contrastive_conv_params._gen_model_feature_params(model_params)
        ## -- Classification parameters
        model_params.cls = domain_contrastive_conv_params._gen_model_cls_params(model_params)

        # Return the final `model_params`.
        return model_params

    # def _gen_model_cnn_params func
    @staticmethod
    def _gen_model_cnn_params(model_params):
        """
        Generate model.cnn parameters.
        """
        # Initialize `model_cnn_params`.
        model_cnn_params = DotDict()

        ## -- Normal parameters (related to Conv1d)
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimension of input vector.
            model_cnn_params.d_input = model_params.n_channels
            # The number of filters of each CNN layer.
            model_cnn_params.n_filters = [256, 128]
            # The size of kernel of each CNN layer.
            model_cnn_params.d_kernel = [9, 11]
            # The length of stride of each CNN layer.
            model_cnn_params.strides = [1, 1]
            # The length of padding of each CNN layer.
            model_cnn_params.padding = ["same", "same"]
            # The dilation rate of each CNN layer.
            model_cnn_params.dilation_rate = [1, 2]
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel of each CNN layer.
            model_cnn_params.d_pooling_kernel = 2
            ## -- Normal parameters (related to Dropout)
            # The dropout rate of dropout layer.
            model_cnn_params.dropout = 0.5
        # Normal parameters related to other dataset.
        else:
            # The dimension of input vector.
            model_cnn_params.d_input = model_params.n_channels
            # The number of filters of each CNN layer.
            model_cnn_params.n_filters = [256, 128]
            # The size of kernel of each CNN layer.
            model_cnn_params.d_kernel = [9, 11]
            # The length of stride of each CNN layer.
            model_cnn_params.strides = [1, 1]
            # The length of padding of each CNN layer.
            model_cnn_params.padding = ["same", "same"]
            # The dilation rate of each CNN layer.
            model_cnn_params.dilation_rate = [1, 2]
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel of each CNN layer.
            model_cnn_params.d_pooling_kernel = 2
            ## -- Normal parameters (related to Dropout)
            # The dropout rate of dropout layer.
            model_cnn_params.dropout = 0.5

        # Return the final `model_cnn_params`.
        return model_cnn_params

    # def _gen_model_feature_params func
    @staticmethod
    def _gen_model_feature_params(model_params):
        """
        Generate model.feature parameters.
        """
        # Initialize `model_feature_params`.
        model_feature_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layer in classification block.
            model_feature_params.d_hidden = []
            # The dropout probability of the hidden layer in classification block.
            model_feature_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layer in classification block.
            model_feature_params.d_hidden = []
            # The dropout probability of the hidden layer in classification block.
            model_feature_params.dropout = 0.
        # The dimension of classification layer in classification block.
        model_feature_params.d_output = model_params.n_labels

        # Return the final `model_feature_params`.
        return model_feature_params

    # def _gen_model_cls_params func
    @staticmethod
    def _gen_model_cls_params(model_params):
        """
        Generate model.cls parameters.
        """
        # Initialize `model_cls_params`.
        model_cls_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layer in classification block.
            model_cls_params.d_hidden = [128,]
            # The dropout probability of the hidden layer in classification block.
            model_cls_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layer in classification block.
            model_cls_params.d_hidden = [128,]
            # The dropout probability of the hidden layer in classification block.
            model_cls_params.dropout = 0.
        # The dimension of classification layer in classification block.
        model_cls_params.d_output = model_params.n_labels

        # Return the final `model_cls_params`.
        return model_cls_params

    ## def _gen_train_* funcs
    # def _gen_train_params func
    @staticmethod
    def _gen_train_params(model_params):
        """
        Generate training parameters.
        """
        # Initialize `train_params`.
        train_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        train_params.dataset = model_params.dataset
        # Precision parameter.
        train_params.precision = getattr(tf, domain_contrastive_conv_params._precision)\
            if hasattr(tf, domain_contrastive_conv_params._precision) else tf.float32
        # Whether use graph mode or eager mode.
        train_params.use_graph_mode = True
        # The ratio of train dataset. The rest is test dataset.
        train_params.train_ratio = 0.8
        # Size of buffer used in shuffle.
        train_params.buffer_size = int(1e4)
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if train_params.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 256 if model_params.use_siamese else 128
            # The learning rate of training process.
            train_params.lr = 3e-4
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 128
            # The learning rate of training process.
            train_params.lr = 3e-4

        # Return the final `train_params`.
        return train_params

# def subdomain_contrastive_conv_params class
class subdomain_contrastive_conv_params(domain_contrastive_conv_params):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `SubdomainContrastiveConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `subdomain_contrastive_conv_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(subdomain_contrastive_conv_params, self).__init__(dataset)

        ## Update all parameters hierarchically.
        # -- Model parameters
        self._update_model_params()
        # -- Train parameters
        self._update_train_params()

        ## Do init iteration.
        subdomain_contrastive_conv_params.iteration(self, 0)

    """
    update funcs
    """
    # def _update_model_params func
    def _update_model_params(self):
        """
        Update model parameters.
        """
        pass

    # def _update_train_params func
    def _update_train_params(self):
        """
        Update train parameters.
        """
        pass

# def domain_contrastive_transformer_params class
class domain_contrastive_transformer_params(DotDict):
    """
    This contains one single object that generates a dictionary of parameters,
    which is provided to `mae` on initialization.
    """
    # Initialize macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `domain_contrastive_transformer_params`.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(domain_contrastive_transformer_params, self).__init__()

        ## Generate all parameters hierarchically.
        # -- Model parameters
        self.model = domain_contrastive_transformer_params._gen_model_params(dataset)
        # -- Train parameters
        self.train = domain_contrastive_transformer_params._gen_train_params(self.model)

        ## Do init iteration.
        domain_contrastive_transformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Train parameters
        # Calculate current learning rate.
        self.train.lr_i = self.train.lr

    """
    generate funcs
    """
    ## def _gen_model_* funcs
    # def _gen_model_params func
    @staticmethod
    def _gen_model_params(dataset):
        """
        Generate model parameters.
        """
        # Initialize `model_params`.
        model_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        model_params.dataset = dataset
        # Precision parameter.
        model_params.precision = getattr(tf, domain_contrastive_transformer_params._precision)\
            if hasattr(tf, domain_contrastive_transformer_params._precision) else tf.float32
        # The number of modalities.
        model_params.n_modalities = 3
        # The flag that indicates whether use siamese network.
        model_params.use_siamese = True
        # The mode of contrastive data format.
        model_params.contra_data_mode = ["point", "sequence"][0]
        # The dimension of contrastive layer.
        model_params.d_contra = 128 if model_params.contra_data_mode == "point" else 32
        # The mode of contrastive loss calculation.
        model_params.contra_loss_mode = ["clip", "clip_orig", "unicl"][-1]
        # The mode of contrastive prediction.
        model_params.contra_pred_mode = ["max_z", "prob_z", "max_y", "prob_y"][-1]
        # The ratio of contrastive prediction.
        model_params.contra_pred_ratio = 0.
        # The ratio of contrastive loss.
        model_params.contra_loss_ratio = 0.5
        # The scale factor of contrastive loss.
        model_params.contra_loss_scale = 0.1
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The number of output classes.
            model_params.n_labels = 15
        # Normal parameters related to other dataset.
        else:
            # The number of output classes.
            model_params.n_labels = 15
        ## -- Encoder parameters
        model_params.encoder = domain_contrastive_transformer_params._gen_model_encoder_params(model_params)
        ## -- Feature parameters
        model_params.feature = domain_contrastive_transformer_params._gen_model_feature_params(model_params)
        ## -- Classification parameters
        model_params.cls = domain_contrastive_transformer_params._gen_model_cls_params(model_params)

        # Return the final `model_params`.
        return model_params

    # def _gen_model_encoder_params func
    @staticmethod
    def _gen_model_encoder_params(model_params):
        """
        Generate model.encoder parameters.
        """
        # Initialize `model_encoder_params`.
        model_encoder_params = DotDict()

        ## -- Normal parameters (related to AttentionBlock)
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The length of patch.
            model_encoder_params.patch_len = 1
            # The mode of position embedding.
            model_encoder_params.pe_mode = "sincos"
            # The dimensions of the embedding.
            model_encoder_params.d_model = 256
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The flag that indicates whether enable residual attention.
            model_encoder_params.res_attn = False
            # The number of attention blocks.
            model_encoder_params.n_blocks = 2
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 128
            # The dropout probability of attention score.
            model_encoder_params.attn_dropout = 0.
            # The dropout probability of attention projection.
            model_encoder_params.proj_dropout = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 256
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout = [0., 0.3]
            # The normalization type within transformer block.
            model_encoder_params.norm_type = "layer_norm"
            # The flag that indicates whether execute normalization first.
            model_encoder_params.norm_first = False
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel right after AttentionBlock.
            model_encoder_params.d_pooling_kernel = 1
            # The dropout rate of dropout layer.
            model_encoder_params.pool_dropout = 0.
            # The flag that indicates whether enables BatchNormalization.
            model_encoder_params.use_bn = False
        # Normal parameters related to other dataset.
        else:
            # The length of patch.
            model_encoder_params.patch_len = 8
            # The mode of position embedding.
            model_encoder_params.pe_mode = "sincos"
            # The dimensions of the embedding.
            model_encoder_params.d_model = 128
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The flag that indicates whether enable residual attention.
            model_encoder_params.res_attn = False
            # The number of attention blocks.
            model_encoder_params.n_blocks = 2
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 64
            # The dropout probability of attention score.
            model_encoder_params.attn_dropout = 0.
            # The dropout probability of attention projection.
            model_encoder_params.proj_dropout = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 128
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout = [0., 0.3]
            # The normalization type within transformer block.
            model_encoder_params.norm_type = "layer_norm"
            # The flag that indicates whether execute normalization first.
            model_encoder_params.norm_first = False
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel right after AttentionBlock.
            model_encoder_params.d_pooling_kernel = 1
            # The dropout rate of dropout layer.
            model_encoder_params.pool_dropout = 0.
            # The flag that indicates whether enables BatchNormalization.
            model_encoder_params.use_bn = True

        # Return the final `model_encoder_params`.
        return model_encoder_params

    # def _gen_model_feature_params func
    @staticmethod
    def _gen_model_feature_params(model_params):
        """
        Generate model.feature parameters.
        """
        # Initialize `model_feature_params`.
        model_feature_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layer in classification block.
            model_feature_params.d_hidden = []
            # The dropout probability of the hidden layer in classification block.
            model_feature_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layer in classification block.
            model_feature_params.d_hidden = []
            # The dropout probability of the hidden layer in classification block.
            model_feature_params.dropout = 0.
        # The dimension of classification layer in classification block.
        model_feature_params.d_output = model_params.n_labels

        # Return the final `model_feature_params`.
        return model_feature_params

    # def _gen_model_cls_params func
    @staticmethod
    def _gen_model_cls_params(model_params):
        """
        Generate model.cls parameters.
        """
        # Initialize `model_cls_params`.
        model_cls_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layer in classification block.
            model_cls_params.d_hidden = [128,]
            # The dropout probability of the hidden layer in classification block.
            model_cls_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layer in classification block.
            model_cls_params.d_hidden = [128,]
            # The dropout probability of the hidden layer in classification block.
            model_cls_params.dropout = 0.
        # The dimension of classification layer in classification block.
        model_cls_params.d_output = model_params.n_labels

        # Return the final `model_cls_params`.
        return model_cls_params

    ## def _gen_train_* funcs
    # def _gen_train_params func
    @staticmethod
    def _gen_train_params(model_params):
        """
        Generate train parameters.
        """
        # Initialize `train_params`.
        train_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        train_params.dataset = model_params.dataset
        # Precision parameter.
        train_params.precision = getattr(tf, domain_contrastive_transformer_params._precision)\
            if hasattr(tf, domain_contrastive_transformer_params._precision) else tf.float32
        # Whether use graph mode or eager mode.
        train_params.use_graph_mode = True
        # The ratio of train dataset. The rest is test dataset.
        train_params.train_ratio = 0.8
        # Size of buffer used in shuffle.
        train_params.buffer_size = int(1e4)
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if train_params.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 256 if model_params.use_siamese else 128
            # The learning rate of optimizer.
            train_params.lr = 3e-4
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            train_params.n_epochs = 20
            # Number of batch size used in training process.
            train_params.batch_size = 32
            # The learning rate of optimizer.
            train_params.lr = 3e-4

        # Return the final `train_params`.
        return train_params

# def subdomain_contrastive_transformer_params class
class subdomain_contrastive_transformer_params(domain_contrastive_transformer_params):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `SubdomainContrastiveConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `subdomain_contrastive_transformer_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(subdomain_contrastive_transformer_params, self).__init__(dataset)

        ## Update all parameters hierarchically.
        # -- Model parameters
        self._update_model_params()
        # -- Train parameters
        self._update_train_params()

        ## Do init iteration.
        subdomain_contrastive_transformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## Iterate super parameters.
        super(subdomain_contrastive_transformer_params, self).iteration(iteration)

    # def _update_model_params func
    def _update_model_params(self):
        """
        Update model parameters.
        """
        pass

    # def _update_train_params func
    def _update_train_params(self):
        """
        Update train parameters.
        """
        pass

# def domain_contrastive_roformer_params class
class domain_contrastive_roformer_params(DotDict):
    """
    This contains one single object that generates a dictionary of parameters,
    which is provided to `mae` on initialization.
    """
    # Initialize macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `domain_contrastive_roformer_params`.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(domain_contrastive_roformer_params, self).__init__()

        ## Generate all parameters hierarchically.
        # -- Model parameters
        self.model = domain_contrastive_roformer_params._gen_model_params(dataset)
        # -- Train parameters
        self.train = domain_contrastive_roformer_params._gen_train_params(self.model)

        ## Do init iteration.
        domain_contrastive_roformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Train parameters
        # Calculate current learning rate.
        self.train.lr_i = self.train.lr

    """
    generate funcs
    """
    ## def _gen_model_* funcs
    # def _gen_model_params func
    @staticmethod
    def _gen_model_params(dataset):
        """
        Generate model parameters.
        """
        # Initialize `model_params`.
        model_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        model_params.dataset = dataset
        # Precision parameter.
        model_params.precision = getattr(tf, domain_contrastive_roformer_params._precision)\
            if hasattr(tf, domain_contrastive_roformer_params._precision) else tf.float32
        # The number of modalities.
        model_params.n_modalities = 3
        # The flag that indicates whether use siamese network.
        model_params.use_siamese = True
        # The mode of contrastive data format.
        model_params.contra_data_mode = ["point", "sequence"][0]
        # The dimension of contrastive layer.
        model_params.d_contra = 128 if model_params.contra_data_mode == "point" else 32
        # The mode of contrastive loss calculation.
        model_params.contra_loss_mode = ["clip", "clip_orig", "unicl"][-1]
        # The mode of contrastive prediction.
        model_params.contra_pred_mode = ["max_z", "prob_z", "max_y", "prob_y"][-1]
        # The ratio of contrastive prediction.
        model_params.contra_pred_ratio = 0.
        # The ratio of contrastive loss.
        model_params.contra_loss_ratio = 0.5
        # The scale factor of contrastive loss.
        model_params.contra_loss_scale = 0.1
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The number of output classes.
            model_params.n_labels = 15
        # Normal parameters related to other dataset.
        else:
            # The number of output classes.
            model_params.n_labels = 15
        ## -- Encoder parameters
        model_params.encoder = domain_contrastive_roformer_params._gen_model_encoder_params(model_params)
        ## -- Feature parameters
        model_params.feature = domain_contrastive_roformer_params._gen_model_feature_params(model_params)
        ## -- Classification parameters
        model_params.cls = domain_contrastive_roformer_params._gen_model_cls_params(model_params)

        # Return the final `model_params`.
        return model_params

    # def _gen_model_encoder_params func
    @staticmethod
    def _gen_model_encoder_params(model_params):
        """
        Generate model.encoder parameters.
        """
        # Initialize `model_encoder_params`.
        model_encoder_params = DotDict()

        ## -- Normal parameters (related to AttentionBlock)
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the embedding.
            model_encoder_params.d_model = 128
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The flag that indicates whether enable residual attention.
            model_encoder_params.res_attn = False
            # The number of attention blocks.
            model_encoder_params.n_blocks = 1
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 64
            # The power base of rotation angle.
            model_encoder_params.rot_theta = 1e2
            # The dropout probability of attention score.
            model_encoder_params.attn_dropout = 0.
            # The dropout probability of attention projection.
            model_encoder_params.proj_dropout = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 128
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout = [0., 0.]
            # The normalization type within transformer block.
            model_encoder_params.norm_type = [None, "layer_norm", "batch_norm"][0]
            # The flag that indicates whether execute normalization first.
            model_encoder_params.norm_first = False
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel right after AttentionBlock.
            model_encoder_params.d_pooling_kernel = 2
            # The dropout rate of dropout layer.
            model_encoder_params.pool_dropout = 0.5
            # The flag that indicates whether enables BatchNormalization.
            model_encoder_params.use_bn = True
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the embedding.
            model_encoder_params.d_model = 128
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The flag that indicates whether enable residual attention.
            model_encoder_params.res_attn = False
            # The number of attention blocks.
            model_encoder_params.n_blocks = 2
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 64
            # The power base of rotation angle.
            model_encoder_params.rot_theta = 1e2
            # The dropout probability of attention score.
            model_encoder_params.attn_dropout = 0.
            # The dropout probability of attention projection.
            model_encoder_params.proj_dropout = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 128
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout = [0., 0.3]
            # The normalization type within transformer block.
            model_encoder_params.norm_type = [None, "layer_norm", "batch_norm"][1]
            # The flag that indicates whether execute normalization first.
            model_encoder_params.norm_first = False
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel right after AttentionBlock.
            model_encoder_params.d_pooling_kernel = 1
            # The dropout rate of dropout layer.
            model_encoder_params.pool_dropout = 0.
            # The flag that indicates whether enables BatchNormalization.
            model_encoder_params.use_bn = True

        # Return the final `model_encoder_params`.
        return model_encoder_params

    # def _gen_model_feature_params func
    @staticmethod
    def _gen_model_feature_params(model_params):
        """
        Generate model.feature parameters.
        """
        # Initialize `model_feature_params`.
        model_feature_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layer in classification block.
            model_feature_params.d_hidden = []
            # The dropout probability of the hidden layer in classification block.
            model_feature_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layer in classification block.
            model_feature_params.d_hidden = []
            # The dropout probability of the hidden layer in classification block.
            model_feature_params.dropout = 0.
        # The dimension of classification layer in classification block.
        model_feature_params.d_output = model_params.n_labels

        # Return the final `model_feature_params`.
        return model_feature_params

    # def _gen_model_cls_params func
    @staticmethod
    def _gen_model_cls_params(model_params):
        """
        Generate model.cls parameters.
        """
        # Initialize `model_cls_params`.
        model_cls_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layer in classification block.
            model_cls_params.d_hidden = [128,]
            # The dropout probability of the hidden layer in classification block.
            model_cls_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layer in classification block.
            model_cls_params.d_hidden = [128,]
            # The dropout probability of the hidden layer in classification block.
            model_cls_params.dropout = 0.
        # The dimension of classification layer in classification block.
        model_cls_params.d_output = model_params.n_labels

        # Return the final `model_cls_params`.
        return model_cls_params

    ## def _gen_train_* funcs
    # def _gen_train_params func
    @staticmethod
    def _gen_train_params(model_params):
        """
        Generate train parameters.
        """
        # Initialize `train_params`.
        train_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        train_params.dataset = model_params.dataset
        # Precision parameter.
        train_params.precision = getattr(tf, domain_contrastive_roformer_params._precision)\
            if hasattr(tf, domain_contrastive_roformer_params._precision) else tf.float32
        # Whether use graph mode or eager mode.
        train_params.use_graph_mode = True
        # The ratio of train dataset. The rest is test dataset.
        train_params.train_ratio = 0.8
        # Size of buffer used in shuffle.
        train_params.buffer_size = int(1e4)
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if train_params.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 256 if model_params.use_siamese else 128
            # The learning rate of optimizer.
            train_params.lr = 3e-4
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            train_params.n_epochs = 20
            # Number of batch size used in training process.
            train_params.batch_size = 32
            # The learning rate of optimizer.
            train_params.lr = 3e-4

        # Return the final `train_params`.
        return train_params

# def subdomain_contrastive_roformer_params class
class subdomain_contrastive_roformer_params(domain_contrastive_roformer_params):
    """
    This contains one single object that generates a dictionary of parameters,
    which is proviede to `SubdomainContrastiveConvNet` on initialization.
    """
    # Internal macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `subdomain_contrastive_roformer_params` object.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(subdomain_contrastive_roformer_params, self).__init__(dataset)

        ## Update all parameters hierarchically.
        # -- Model parameters
        self._update_model_params()
        # -- Train parameters
        self._update_train_params()

        ## Do init iteration.
        subdomain_contrastive_roformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## Iterate super parameters.
        super(subdomain_contrastive_roformer_params, self).iteration(iteration)

    # def _update_model_params func
    def _update_model_params(self):
        """
        Update model parameters.
        """
        pass

    # def _update_train_params func
    def _update_train_params(self):
        """
        Update train parameters.
        """
        pass

if __name__ == "__main__":
    # Instantiate `domain_adversarial_conv_params`.
    domain_adversarial_conv_params_inst = domain_adversarial_conv_params(dataset="eeg_anonymous")
    # Instantiate `domain_adversarial_transformer_params`.
    domain_adversarial_transformer_params_inst = domain_adversarial_transformer_params(dataset="eeg_anonymous")
    # Instantiate `subdomain_adversarial_conv_params`.
    subdomain_adversarial_conv_params_inst = subdomain_adversarial_conv_params(dataset="eeg_anonymous")
    # Instantiate `subdomain_adversarial_transformer_params`.
    subdomain_adversarial_transformer_params_inst = subdomain_adversarial_transformer_params(dataset="eeg_anonymous")
    # Instantiate `domain_contrastive_conv_params`.
    domain_contrastive_conv_params_inst = domain_contrastive_conv_params(dataset="eeg_anonymous")
    # Instantiate `subdomain_contrastive_conv_params`.
    subdomain_contrastive_conv_params_inst = subdomain_contrastive_conv_params(dataset="eeg_anonymous")
    # Instantiate `domain_contrastive_transformer_params`.
    domain_contrastive_transformer_params_inst = domain_contrastive_transformer_params(dataset="eeg_anonymous")
    # Instantiate `subdomain_contrastive_transformer_params`.
    subdomain_contrastive_transformer_params_inst = subdomain_contrastive_transformer_params(dataset="eeg_anonymous")
    # Instantiate `domain_contrastive_roformer_params`.
    domain_contrastive_roformer_params_inst = domain_contrastive_roformer_params(dataset="eeg_anonymous")
    # Instantiate `subdomain_contrastive_roformer_params`.
    subdomain_contrastive_roformer_params_inst = subdomain_contrastive_roformer_params(dataset="eeg_anonymous")

