#!/usr/bin/env python3
"""
Created on 23:15, Aug. 3rd, 2023

@author: Anonymous
"""
import tensorflow as tf
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.pardir)
from utils import DotDict

__all__ = [
    "roformer_params",
]

class roformer_params(DotDict):
    """
    This contains one single object that generates a dictionary of parameters,
    which is provided to `mae` on initialization.
    """
    # Initialize macro parameter.
    _precision = "float32"

    def __init__(self, dataset="eeg_anonymous"):
        """
        Initialize `roformer_params`.
        """
        ## First call super class init function to set up `DotDict`
        ## style object and inherit it's functionality.
        super(roformer_params, self).__init__()

        ## Generate all parameters hierarchically.
        # -- Model parameters
        self.model = roformer_params._gen_model_params(dataset)
        # -- Train parameters
        self.train = roformer_params._gen_train_params(dataset)

        ## Do init iteration.
        roformer_params.iteration(self, 0)

    """
    update funcs
    """
    # def iteration func
    def iteration(self, iteration):
        """
        Update parameters at every backpropagation iteration/gradient update.
        """
        ## -- Train parameters
        # Calculate current learning rate.
        self.train.lr_i = self.train.lr

    """
    generate funcs
    """
    ## def _gen_model_* funcs
    # def _gen_model_params func
    @staticmethod
    def _gen_model_params(dataset):
        """
        Generate model parameters.
        """
        # Initialize `model_params`.
        model_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        model_params.dataset = dataset
        # Precision parameter.
        model_params.precision = getattr(tf, roformer_params._precision)\
            if hasattr(tf, roformer_params._precision) else tf.float32
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The size of input channels.
            model_params.n_channels = 55
            # The number of output classes.
            model_params.n_labels = 15
        # Normal parameters related to other dataset.
        else:
            # The size of input channels.
            model_params.n_channels = 128
            # The number of output classes.
            model_params.n_labels = 10
        ## -- Encoder parameters
        model_params.encoder = roformer_params._gen_model_encoder_params(model_params)
        ## -- Fully connect parameters
        model_params.fc = roformer_params._gen_model_fc_params(model_params)

        # Return the final `model_params`.
        return model_params

    # def _gen_model_encoder_params func
    @staticmethod
    def _gen_model_encoder_params(model_params):
        """
        Generate model.encoder parameters.
        """
        # Initialize `model_encoder_params`.
        model_encoder_params = DotDict()

        ## -- Normal parameters (related to AttentionBlock)
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the embedding.
            model_encoder_params.d_model = 128
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The flag that indicates whether enable residual attention.
            model_encoder_params.res_attn = False
            # The number of attention blocks.
            model_encoder_params.n_blocks = 1
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 64
            # The power base of rotation angle.
            model_encoder_params.rot_theta = 1e2
            # The dropout probability of attention score.
            model_encoder_params.attn_dropout = 0.
            # The dropout probability of attention projection.
            model_encoder_params.proj_dropout = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 128
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout = [0., 0.]
            # The normalization type within transformer block.
            model_encoder_params.norm_type = [None, "layer_norm", "batch_norm"][0]
            # The flag that indicates whether execute normalization first.
            model_encoder_params.norm_first = False
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel right after AttentionBlock.
            model_encoder_params.d_pooling_kernel = 2
            # The dropout rate of dropout layer.
            model_encoder_params.pool_dropout = 0.5
            # The flag that indicates whether enables BatchNormalization.
            model_encoder_params.use_bn = True
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the embedding.
            model_encoder_params.d_model = 128
            # The maximum length of element sequence.
            model_encoder_params.max_len = 80
            # The flag that indicates whether enable residual attention.
            model_encoder_params.res_attn = False
            # The number of attention blocks.
            model_encoder_params.n_blocks = 2
            # The number of attention heads.
            model_encoder_params.n_heads = 8
            # The dimensions of attention head.
            model_encoder_params.d_head = 64
            # The power base of rotation angle.
            model_encoder_params.rot_theta = 1e2
            # The dropout probability of attention score.
            model_encoder_params.attn_dropout = 0.
            # The dropout probability of attention projection.
            model_encoder_params.proj_dropout = 0.
            # The dimensions of the hidden layer in ffn.
            model_encoder_params.d_ff = 128
            # The dropout probability of the hidden layer in ffn.
            model_encoder_params.ff_dropout = [0., 0.3]
            # The normalization type within transformer block.
            model_encoder_params.norm_type = [None, "layer_norm", "batch_norm"][1]
            # The flag that indicates whether execute normalization first.
            model_encoder_params.norm_first = False
            ## -- Normal parameters (related to MaxPool1d)
            # The size of max pooling kernel right after AttentionBlock.
            model_encoder_params.d_pooling_kernel = 2
            # The dropout rate of dropout layer.
            model_encoder_params.pool_dropout = 0.
            # The flag that indicates whether enables BatchNormalization.
            model_encoder_params.use_bn = True

        # Return the final `model_encoder_params`.
        return model_encoder_params

    # def _gen_model_fc_params func
    @staticmethod
    def _gen_model_fc_params(model_params):
        """
        Generate model.fc parameters.
        """
        # Initialize `model_fc_params`.
        model_fc_params = DotDict()

        ## -- Normal parameters
        # Normal parameters related to eeg_anonymous dataset.
        if model_params.dataset == "eeg_anonymous":
            # The dimensions of the hidden layers.
            model_fc_params.d_hidden = [128,]
            # The dropout rate of dropout layer.
            model_fc_params.dropout = 0.
        # Normal parameters related to other dataset.
        else:
            # The dimensions of the hidden layers.
            model_fc_params.d_hidden = [64,]
            # The dropout rate of dropout layer.
            model_fc_params.dropout = 0.
        # The dimension of output vector.
        model_fc_params.d_output = model_params.n_labels

        # Return the final `model_fc_params`.
        return model_fc_params

    ## def _gen_train_* funcs
    # def _gen_train_params func
    @staticmethod
    def _gen_train_params(dataset):
        """
        Generate train parameters.
        """
        # Initialize `train_params`.
        train_params = DotDict()

        ## -- Normal parameters
        # The type of dataset.
        train_params.dataset = dataset
        # Precision parameter.
        train_params.precision = getattr(tf, roformer_params._precision)\
            if hasattr(tf, roformer_params._precision) else tf.float32
        # Whether use graph mode or eager mode.
        train_params.use_graph_mode = True
        # The ratio of train dataset. The rest is test dataset.
        train_params.train_ratio = 0.8
        # Size of buffer used in shuffle.
        train_params.buffer_size = int(1e4)
        ## -- Dataset-specific parameters
        # Normal parameters related to eeg_anonymous dataset.
        if train_params.dataset == "eeg_anonymous":
            # Number of epochs used in training process.
            train_params.n_epochs = 200
            # Number of batch size used in training process.
            train_params.batch_size = 256
            # The learning rate of training process.
            train_params.lr = 3e-4
        # Normal parameters related to other dataset.
        else:
            # Number of epochs used in training process.
            train_params.n_epochs = 20
            # Number of batch size used in training process.
            train_params.batch_size = 16
            # The learning rate of training process.
            train_params.lr = 3e-4

        # Return the final `train_params`.
        return train_params

if __name__ == "__main__":
    # Instantiate `roformer_params`.
    roformer_params_inst = roformer_params(dataset="eeg_anonymous")

