#!/usr/bin/env python3
"""
Created on 18:57, Jul. 31st, 2023

@author: Anonymous
"""
import sys, time
import copy as cp
import numpy as np
import tensorflow as tf
from collections import Counter
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.path.join(os.pardir, os.pardir))
import utils; import utils.model; import utils.data.eeg
from models.clip_transformer import clip_transformer as clip_transformer_model

__all__ = [
    "train",
]

# Global variables.
params = None; paths = None
model = None; optimizer = None

"""
init funcs
"""
# def init func
def init(base_, params_):
    """
    Initialize `clip_transformer` training variables.

    Args:
        base_: str - The path of project base.
        params_: DotDict - The parameters of current training process.

    Returns:
        None
    """
    global params, paths
    # Initialize params.
    params = cp.deepcopy(params_)
    paths = utils.Paths(base=base_, params=params)
    # Initialize model.
    _init_model()
    # Initialize training process.
    _init_train()

# def _init_model func
def _init_model():
    """
    Initialize model used in the training process.

    Args:
        None

    Returns:
        None
    """
    global params
    ## Initialize tf configuration.
    # Not set random seed, should be done before instantiating `model`.
    # Set default precision.
    tf.keras.backend.set_floatx(params._precision)
    # Check whether run in graph mode or eager mode.
    tf.config.run_functions_eagerly(not params.train.use_graph_mode)

# def _init_train func
def _init_train():
    """
    Initialize the training process.

    Args:
        None

    Returns:
        None
    """
    pass

"""
data funcs
"""
# def load_data func
def load_data(load_params):
    """
    Load data from specified dataset.

    Args:
        load_params: DotDict - The load parameters of specified dataset.

    Returns:
        dataset_train: tuple - The train dataset, including (X_train, y_train).
        dataset_validation: tuple - The validation dataset, including (X_validation, y_validation).
        dataset_test: tuple - The test dataset, including (X_test, y_test).
    """
    global params
    # Load data from specified dataset.
    try:
        func = getattr(sys.modules[__name__], "_".join(["_load_data", params.train.dataset]))
        dataset_train, dataset_validation, dataset_test = func(load_params)
    except Exception:
        raise ValueError("ERROR: Unknown dataset type {} in train.clip_transformer.".format(params.train.dataset))
    # Return the final `dataset_train` & `dataset_validation` & `dataset_test`.
    return dataset_train, dataset_validation, dataset_test

# def _load_data_eeg_zhou2023cibr func
def _load_data_eeg_zhou2023cibr(load_params):
    """
    Load eeg data from the specified subject in `eeg_zhou2023cibr`.

    Args:
        load_params: DotDict - The load parameters of specified dataset.

    Returns:
        dataset_train: tuple - The train dataset, including (X_train, y_train).
        dataset_validation: tuple - The validation dataset, including (X_validation, y_validation).
        dataset_test: tuple - The test dataset, including (X_test, y_test).
    """
    global params, paths
    ## Load source-set & target-set.
    # Initialize path_run.
    path_run = os.path.join(paths.base, "data", "eeg.zhou2023cibr", "020", "20230405")\
        if not hasattr(load_params, "path_run") else load_params.path_run
    # Load data from specified subject run.
    datasets = utils.DotDict(); dataset_names = sorted(set(load_params.sourceset) | set(load_params.targetset))
    for dataset_name_i in dataset_names:
        # Load data from specified dataset name.
        session_type_i = "-".join(dataset_name_i.split("-")[:-1]); data_type_i = dataset_name_i.split("-")[-1]
        func_i = getattr(utils.data.eeg.zhou2023cibr, "_".join(["load_run", session_type_i.split("-")[0]]))
        X_i, y_i = func_i(path_run, session_type="-".join(session_type_i.split("-")[1:]))
        # Check whether current dataset has data items.
        if X_i[data_type_i] is None:
            msg = "WARNING: Skip dataset {} with no data item.".format(dataset_name_i)
            print(msg); paths.run.logger.summaries.info(msg); continue
        X_i = X_i[data_type_i].astype(np.float32); y_i = y_i[data_type_i].astype(np.int64)
        # Truncate `X_i` to let them have the same length.
        # TODO: Here, we only keep the [0.0~0.8]s-part of [audio,image] that after onset. And we should
        # note that the [0.0~0.8]s-part of image is the whole onset time of image, the [0.0~0.8]s-part
        # of audio is the sum of the whole onset time of audio and the following 0.3s padding.
        # X_i - (n_samples, seq_len, n_channels)
        # If the type of dataset is `default`.
        if load_params.type == "default":
            X_i = X_i[:,20:100,:]
        # If the type of dataset is `lvbj`.
        elif load_params.type == "lvbj":
            X_i = X_i[:,20:100,:]
        # Get unknown type of dataset.
        else:
            raise ValueError("ERROR: Unknown type {} of dataset".format(load_params.type))
        # Do cross-trial normalization.
        X_i = (X_i - np.mean(X_i)) / np.std(X_i)
        # Set the corresponding item of dataset.
        datasets[dataset_name_i] = utils.DotDict({"X":X_i,"y_class":y_i,})
    # Initialize sourceset & targetset.
    X_source = []; y_source = []; X_target = []; y_target = []
    for dataset_name_i, dataset_i in datasets.items():
        # If sourceset and targetset are not the same, construct sourceset & targetset separately.
        if dataset_name_i in load_params.sourceset and dataset_name_i not in load_params.targetset:
            X_source.append(dataset_i.X); y_source.append(dataset_i.y_class)
        # If sourceset and targetset are not the same, construct sourceset & targetset separately.
        elif dataset_name_i not in load_params.sourceset and dataset_name_i in load_params.targetset:
            X_target.append(dataset_i.X); y_target.append(dataset_i.y_class)
        # Wrong cases.
        else:
            raise ValueError("ERROR: Unknown dataset name {}.".format(dataset_name_i))
    # Check whether sourceset & targetset both have data items.
    if len(X_source) == 0 or len(X_target) == 0: return [], [], []
    # X - (n_samples, seq_len, n_channels); y_class - (n_samples,)
    X_source = np.concatenate(X_source, axis=0); y_source = np.concatenate(y_source, axis=0)
    X_target = np.concatenate(X_target, axis=0); y_target = np.concatenate(y_target, axis=0)
    # Make sure there is no overlap between X_source & X_target.
    samples_same = None; n_samples = 10; assert X_source.shape[1] == X_target.shape[1]
    for _ in range(n_samples):
        sample_idx = np.random.randint(X_source.shape[1])
        sample_same_i = np.intersect1d(X_source[:,sample_idx,0], X_target[:,sample_idx,0], return_indices=True)[-1].tolist()
        samples_same = set(sample_same_i) if samples_same is None else set(sample_same_i) & samples_same
    assert len(samples_same) == 0
    # Check whether labels are enough, then transform y to one-hot encoding.
    try:
        assert len(set(y_source)) == len(set(y_target)) == params.model.n_labels; labels = sorted(set(y_source))
    except AssertionError as e:
        msg = (
            "WARNING: Skip experiment (source:{};target:{}) due to that the classes of test cases are not enough."
        ).format(set(load_params.sourceset), set(load_params.targetset))
        print(msg); paths.run.logger.summaries.info(msg); return [], [], []
    # y - (n_samples,)
    y_source = np.array([labels.index(y_i) for y_i in y_source], dtype=np.int64)
    y_target = np.array([labels.index(y_i) for y_i in y_target], dtype=np.int64)
    ## Split target domain into train-set & validation-set & test-set.
    train_ratio = params.train.train_ratio; validation_ratio = test_ratio = (1. - train_ratio) / 2.
    train_idxs = []; validation_idxs = []; test_idxs = []; label_counter_target = Counter(y_target)
    if min(label_counter_target.values()) < 3: return [], [], []
    label_counter_target_train = utils.DotDict(); label_counter_target_test = utils.DotDict()
    label_counter_target_validation = utils.DotDict()
    for label_i in sorted(label_counter_target.keys()):
        label_counter_target_validation[label_i] = max(int(validation_ratio * label_counter_target[label_i]), 1)
        label_counter_target_test[label_i] = max(int(test_ratio * label_counter_target[label_i]), 1)
        label_counter_target_train[label_i] = label_counter_target[label_i] -\
            label_counter_target_validation[label_i] - label_counter_target_test[label_i]
    for label_i in sorted(label_counter_target.keys()):
        label_idxs_i = np.where(y_target == label_i)[0]
        validation_idxs_i = np.random.choice(label_idxs_i, size=label_counter_target_validation[label_i], replace=False)
        test_idxs_i = np.random.choice(sorted(set(label_idxs_i) - set(validation_idxs_i)),
            size=label_counter_target_test[label_i], replace=False)
        train_idxs_i = sorted(set(label_idxs_i) - set(validation_idxs_i) - set(test_idxs_i))
        train_idxs.extend(train_idxs_i); validation_idxs.extend(validation_idxs_i); test_idxs.extend(test_idxs_i)
    assert len(train_idxs) == len(set(train_idxs)); assert len(test_idxs) == len(set(test_idxs))
    assert len(validation_idxs) == len(set(validation_idxs))
    assert len(train_idxs) + len(validation_idxs) + len(test_idxs) == len(y_target)
    assert len(set(train_idxs) & set(validation_idxs)) == 0
    assert len(set(train_idxs) & set(test_idxs)) == 0
    assert len(set(validation_idxs) & set(test_idxs)) == 0
    X_target_train = X_target[train_idxs,:,:]; y_target_train = y_target[train_idxs]
    X_target_validation = X_target[validation_idxs,:,:]; y_target_validation = y_target[validation_idxs]
    X_target_test = X_target[test_idxs,:,:]; y_target_test = y_target[test_idxs]
    ## Construct train-set & validation-set & test-set according to target-set.
    # Initialize `*_train` & `*_validation` & `*_test`.
    X_train = []; X_validation = []; X_test = []; y_train = []; y_validation = []; y_test = []
    # Loop over target-train-set to construct train-set.
    for X_target_train_i, y_target_train_i in zip(X_target_train, y_target_train):
        # Get the indices of samples from source-set.
        sample_idxs = np.random.choice(np.where(y_source == y_target_train_i)[0], size=load_params.n_pairs, replace=False)
        # Construct the corresponding dataset.
        for sample_idx in sample_idxs:
            X_train.append((X_target_train_i, X_source[sample_idx])); y_train.append(y_target_train_i)
    # Loop over target-validation-set to construct validation-set.
    for X_target_validation_i, y_target_validation_i in zip(X_target_validation, y_target_validation):
        # Get the indices of samples from source-set.
        sample_idxs = np.random.choice(np.where(y_source == y_target_validation_i)[0], size=load_params.n_pairs, replace=False)
        # Construct the corresponding dataset.
        for sample_idx in sample_idxs:
            X_validation.append((X_target_validation_i, X_source[sample_idx])); y_validation.append(y_target_validation_i)
    # Loop over target-test-set to construct test-set.
    for X_target_test_i, y_target_test_i in zip(X_target_test, y_target_test):
        # Get the indices of samples from source-set.
        sample_idxs = np.random.choice(np.where(y_source == y_target_test_i)[0], size=load_params.n_pairs, replace=False)
        # Construct the corresponding dataset.
        for sample_idx in sample_idxs:
            X_test.append((X_target_test_i, X_source[sample_idx])); y_test.append(y_target_test_i)
    # Convert to `np.array`.
    # X_* - (n_samples, 2, seq_len, n_channels), y_* - (n_samples, n_labels)
    X_train = np.array(X_train, dtype=np.float32)
    y_train = np.eye(len(labels))[np.array(y_train, dtype=np.int64)]
    X_validation = np.array(X_validation, dtype=np.float32)
    y_validation = np.eye(len(labels))[np.array(y_validation, dtype=np.int64)]
    X_test = np.array(X_test, dtype=np.float32); y_test = np.eye(len(labels))[np.array(y_test, dtype=np.int64)]
    # Log information of data loading.
    msg = (
        "INFO: Data preparation complete, with train-set ({}) & validation-set ({}) & test-set ({})."
    ).format(X_target_train.shape, X_target_validation.shape, X_target_test.shape)
    print(msg); paths.run.logger.summaries.info(msg)
    # Construct dataset from data items.
    dataset_train = tf.data.Dataset.from_tensor_slices((X_train, y_train))
    dataset_validation = tf.data.Dataset.from_tensor_slices((X_validation, y_validation))
    dataset_test = tf.data.Dataset.from_tensor_slices((X_test, y_test))
    # Shuffle and then batch the dataset.
    dataset_train = dataset_train.shuffle(params.train.buffer_size).batch(
        params.train.batch_size, drop_remainder=False)
    dataset_validation = dataset_validation.shuffle(params.train.buffer_size).batch(
        params.train.batch_size, drop_remainder=False)
    dataset_test = dataset_test.shuffle(params.train.buffer_size).batch(
        params.train.batch_size, drop_remainder=False)
    # Return the final `dataset_train` & `dataset_validation` & `dataset_test`.
    return dataset_train, dataset_validation, dataset_test

"""
train funcs
"""
# def train func
def train(base_, params_):
    """
    Train the model.

    Args:
        base_: str - The base path of current project.
        params_: str - The parameters of current training process.

    Returns:
        None
    """
    global _forward, _train
    global params, paths, model, optimizer
    # Initialize parameters & variables of current training process.
    init(base_, params_)
    # Log the start of current training process.
    paths.run.logger.summaries.info("Training started with dataset {}.".format(params.train.dataset))
    # Initialize load_params. Each load_params_i corresponds to a sub-dataset.
    if params.train.dataset == "eeg_zhou2023cibr":
        # Initialize the paths of runs that we want to execute experiments.
        path_runs = [
            #os.path.join(paths.base, "data", "eeg.zhou2023cibr", "005", "20221223"),
            #os.path.join(paths.base, "data", "eeg.zhou2023cibr", "006", "20230103"),
            #os.path.join(paths.base, "data", "eeg.zhou2023cibr", "007", "20230106"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "011", "20230214"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "013", "20230308"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "018", "20230331"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "019", "20230403"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "020", "20230405"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "021", "20230407"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "023", "20230412"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "024", "20230414"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "025", "20230417"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "026", "20230419"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "027", "20230421"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "028", "20230424"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "029", "20230428"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "030", "20230504"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "031", "20230510"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-002", "20230509"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "033", "20230517"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "034", "20230519"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-003", "20230524"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-004", "20230528"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-005", "20230601"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "036", "20230526"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "037", "20230529"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "038", "20230531"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "039", "20230605"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "040", "20230607"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-006", "20230603"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-007", "20230608"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-008", "20230610"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "042", "20230614"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "043", "20230616"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "044", "20230619"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "045", "20230626"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "046", "20230628"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-009", "20230613"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-010", "20230615"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-012", "20230623"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-013", "20230627"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-014", "20230629"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-015", "20230701"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "047", "20230703"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "048", "20230705"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-016", "20230703"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-017", "20230706"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "049", "20230710"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "050", "20230712"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "051", "20230717"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "052", "20230719"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-018", "20230710"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-019", "20230712"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-020", "20230714"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-021", "20230718"),
        ]; load_type = "default"; n_pairs = 15
        # `load_params` contains all the experiments that we want to execute for every run.
        load_params = [
            # source-task-all-image-target-tmr-n23-audio
            utils.DotDict({
                "name": "source-task-all-image-target-tmr-n23-audio",
                "sourceset": [
                    "task-image-audio-pre-image", "task-audio-image-pre-image",
                    "task-image-audio-post-image", "task-audio-image-post-image",
                ],
                "targetset": ["tmr-N2/3-audio",],
                "type": load_type, "n_pairs": n_pairs,
            }),
            # source-task-all-audio-target-tmr-n23-audio
            utils.DotDict({
                "name": "source-task-all-audio-target-tmr-n23-audio",
                "sourceset": [
                    "task-image-audio-pre-audio", "task-audio-image-pre-audio",
                    "task-image-audio-post-audio", "task-audio-image-post-audio",
                ],
                "targetset": ["tmr-N2/3-audio",],
                "type": load_type, "n_pairs": n_pairs,
            }),
        ]
    else:
        raise ValueError("ERROR: Unknown dataset {} in train.naive_rnn.".format(params.train.dataset))
    # Execute experiments for each dataset run.
    for path_run_i in path_runs:
        # Log the start of current training iteration.
        msg = "Training started with sub-dataset {}.".format(path_run_i)
        print(msg); paths.run.logger.summaries.info(msg)
        for load_params_idx in range(len(load_params)):
            # Add `path_run` to `load_params_i`.
            load_params_i = cp.deepcopy(load_params[load_params_idx]); load_params_i.path_run = path_run_i
            # Load data from specified experiment.
            dataset_train, dataset_validation, dataset_test = load_data(load_params_i)
            # Check whether source-set & target-set exists.
            if len(dataset_train) == 0 or len(dataset_validation) == 0 or len(dataset_test) == 0:
                msg = (
                    "INFO: Skip experiment {} with train-set ({:d} items) &" +\
                    " validation-set ({:d} items) & test-set ({:d} items)."
                ).format(load_params_i.name, len(dataset_train), len(dataset_validation), len(dataset_test))
                print(msg); paths.run.logger.summaries.info(msg); continue
            # Start training process of current specified experiment.
            msg = "Start the training process of experiment {}.".format(load_params_i.name)
            print(msg); paths.run.logger.summaries.info(msg)

            # Train the model for each time segment.
            accuracies_train =[]; accuracies_validation = []; accuracies_test = []

            # Reset the iteration information of params.
            params.iteration(iteration=0)
            # Initialize model of current time segment.
            model = clip_transformer_model(params.model)
            # Make an ADAM optimizer for model.
            optimizer = tf.keras.optimizers.Adam(learning_rate=params.train.lr_i)
            # Re-initialize the tf.function.
            _forward = tf.function(_forward.python_function); _train = tf.function(_train.python_function)
            for epoch_idx in range(params.train.n_epochs):
                # Record the start time of preparing data.
                time_start = time.time()
                # Start training epoch.
                paths.run.logger.summaries.info("Start training epoch {:d}.".format(epoch_idx))
                # Execute train process.
                accuracy_train = []; loss_train = []; batch_size_train = []
                for batch_i in dataset_train:
                    # Get the number of current batch_i.
                    batch_size_i = len(batch_i[0]); batch_size_train.append(batch_size_i)
                    # Re-formulate `batch_i`.
                    batch_i = [tf.transpose(batch_i[0], perm=[1,0,2,3]), batch_i[1]]
                    # Train model for current batch.
                    outputs_i, loss_i = _train(batch_i); outputs_i, loss_i = outputs_i.numpy(), loss_i.numpy()
                    accuracy_train_i = np.argmax(outputs_i, axis=-1) == np.argmax(batch_i[1], axis=-1)
                    accuracy_train_i = np.sum(accuracy_train_i) / accuracy_train_i.size
                    accuracy_train.append(accuracy_train_i); loss_train.append(loss_i)
                accuracy_train = np.sum(np.array(accuracy_train) * np.array(batch_size_train)) / np.sum(batch_size_train)
                loss_train = np.sum(np.array(loss_train) * np.array(batch_size_train)) / np.sum(batch_size_train)
                accuracies_train.append(accuracy_train)
                # Execute validation process.
                accuracy_validation = []; loss_validation = []; batch_size_validation = []
                for batch_i in dataset_validation:
                    # Get the number of current batch_i.
                    batch_size_i = len(batch_i[0]); batch_size_validation.append(batch_size_i)
                    # Re-formulate `batch_i`.
                    batch_i = [tf.transpose(batch_i[0], perm=[1,0,2,3]), batch_i[1]]
                    # validation model for current batch.
                    outputs_i, loss_i = _forward(batch_i); outputs_i, loss_i = outputs_i.numpy(), loss_i.numpy()
                    accuracy_validation_i = np.argmax(outputs_i, axis=-1) == np.argmax(batch_i[1], axis=-1)
                    accuracy_validation_i = np.sum(accuracy_validation_i) / accuracy_validation_i.size
                    accuracy_validation.append(accuracy_validation_i); loss_validation.append(loss_i)
                accuracy_validation = np.sum(np.array(accuracy_validation) *\
                    np.array(batch_size_validation)) / np.sum(batch_size_validation)
                loss_validation = np.sum(np.array(loss_validation) *\
                    np.array(batch_size_validation)) / np.sum(batch_size_validation)
                accuracies_validation.append(accuracy_validation)
                # Execute test process.
                accuracy_test = []; loss_test = []; batch_size_test = []
                for batch_i in dataset_test:
                    # Get the number of current batch_i.
                    batch_size_i = len(batch_i[0]); batch_size_test.append(batch_size_i)
                    # Re-formulate `batch_i`.
                    batch_i = [tf.transpose(batch_i[0], perm=[1,0,2,3]), batch_i[1]]
                    # Test model for current batch.
                    outputs_i, loss_i = _forward(batch_i); outputs_i, loss_i = outputs_i.numpy(), loss_i.numpy()
                    accuracy_test_i = np.argmax(outputs_i, axis=-1) == np.argmax(batch_i[1], axis=-1)
                    accuracy_test_i = np.sum(accuracy_test_i) / accuracy_test_i.size
                    accuracy_test.append(accuracy_test_i); loss_test.append(loss_i)
                accuracy_test = np.sum(np.array(accuracy_test) *\
                    np.array(batch_size_test)) / np.sum(batch_size_test)
                loss_test = np.sum(np.array(loss_test) *\
                    np.array(batch_size_test)) / np.sum(batch_size_test)
                accuracies_test.append(accuracy_test)
                # Log information related to current training epoch.
                time_stop = time.time()
                msg = (
                    "Finish train epoch {:d} in {:.2f} seconds, generating {:d} concrete functions."
                ).format(epoch_idx, time_stop-time_start, len(_train.pretty_printed_concrete_signatures().split("\n\n")))
                print(msg); paths.run.logger.summaries.info(msg)
                msg = (
                    "Accuracy(train): {:.2f}%. Loss(train): {:.5f}."
                ).format(accuracy_train * 100., loss_train)
                print(msg); paths.run.logger.summaries.info(msg)
                msg = (
                    "Accuracy(validation): {:.2f}%. Loss(validation): {:.5f}."
                ).format(accuracy_validation * 100., loss_validation)
                print(msg); paths.run.logger.summaries.info(msg)
                msg = (
                    "Accuracy(test): {:.2f}%. Loss(test): {:.5f}."
                ).format(accuracy_test * 100., loss_test)
                print(msg); paths.run.logger.summaries.info(msg)
                # Summarize model information.
                if epoch_idx == 0:
                    model.summary(print_fn=print); model.summary(print_fn=paths.run.logger.summaries.info)
            # Convert `accuracies_validation` & `accuracies_test` to `np.array`.
            accuracies_validation = np.round(np.array(accuracies_validation, dtype=np.float32), decimals=4)
            accuracies_test = np.round(np.array(accuracies_test, dtype=np.float32), decimals=4)
            epoch_maxacc_idxs = np.where(accuracies_validation == np.max(accuracies_validation))[0]
            epoch_maxacc_idx = epoch_maxacc_idxs[np.argmax(accuracies_test[epoch_maxacc_idxs])]
            # Finish training process of current specified experiment.
            msg = (
                "Finish the training process of experiment {}, with test-accuracy ({:.2f}%)" +\
                " according to max validation-accuracy ({:.2f}%) at epoch {:d}, generating {:d} concrete functions."
            ).format(load_params_i.name, accuracies_test[epoch_maxacc_idx]*100.,
                accuracies_validation[epoch_maxacc_idx]*100., epoch_maxacc_idx,
                len(_train.pretty_printed_concrete_signatures().split("\n\n")))
            print(msg); paths.run.logger.summaries.info(msg)
        # Log the end of current training iteration.
        msg = "Training finished with sub-dataset {}.".format(path_run_i)
        print(msg); paths.run.logger.summaries.info(msg)
    # Log the end of current training process.
    msg = "Training finished with dataset {}.".format(params.train.dataset)
    print(msg); paths.run.logger.summaries.info(msg)

# def _forward func
@tf.function
def _forward(inputs, training=False):
    """
    Forward the model using one-step data. Everything entering this function already be a tensor.
    :param inputs: (X, locations, subject_id, y)
    :param training: Indicate whether enable training process.
    :return outputs_: (n_samples, n_labels) - The predicted labels of inputs.
    :return loss_: float - The corresponding cross-entropy loss.
    """
    global model; return model(inputs, training=training)

# def _train func
@tf.function
def _train(inputs):
    """
    Train the model using one-step data. Everything entering this function already be a tensor.
    :param inputs: (X, locations, subject_id, y)
    :return outputs_: (n_samples, n_labels) - The predicted labels of inputs.
    :return loss_: float - The corresponding cross-entropy loss.
    """
    global model, optimizer
    # Train the model using one-step data.
    with tf.GradientTape() as gt:
        outputs_, loss_ = _forward(inputs, training=True)
    # Modify weights to optimize the model.
    gradients = gt.gradient(loss_, model.trainable_variables)
    optimizer.apply_gradients(zip(gradients, model.trainable_variables))
    # Return the final `outputs_` & `loss_`.
    return outputs_, loss_

if __name__ == "__main__":
    import os
    # local dep
    from params.clip_transformer_params import clip_transformer_params

    # macro
    dataset = "eeg_zhou2023cibr"

    # Initialize random seed.
    utils.model.set_seeds(42)

    ## Instantiate naive_rnn.
    # Initialize base.
    base = os.path.join(os.getcwd(), os.pardir, os.pardir)
    # Instantiate clip_transformer_params.
    clip_transformer_params_inst = clip_transformer_params(dataset=dataset)
    # Train `clip_transformer`.
    train(base, clip_transformer_params_inst)

