#!/usr/bin/env python3
"""
Created on 21:27, Jul. 25th, 2023

@author: Anonymous
"""
import time
import copy as cp
import numpy as np
import tensorflow as tf
from collections import Counter
# local dep
if __name__ == "__main__":
    import os, sys
    sys.path.insert(0, os.path.join(os.pardir, os.pardir, os.pardir, os.pardir))
import utils; import utils.model
import utils.data.eeg
from models.domain_adaptation import SubdomainAdversarialTransformer as sdatn_model

__all__ = [
    "train",
]

# Global variables.
params = None; paths = None

"""
init funcs
"""
# def init func
def init(base_, params_):
    """
    Initialize `SubdomainAdversarialTransformer` training variables.

    Args:
        base_: str - The base path of current project.
        params_: DotDict - The parameters of current training process.

    Returns:
        None
    """
    global params, paths
    # Initialize parameters.
    params = cp.deepcopy(params_)
    paths = utils.Paths(base=base_, params=params)
    # Initialize model.
    _init_model()
    # Initialize training process.
    _init_train()

# def _init_model func
def _init_model():
    """
    Initialize model used in the training process.

    Args:
        None

    Returns:
        None
    """
    global params
    ## Initialize tf configuration.
    # Not set random seed, should be done before instantiating `model`.
    # Set default precision.
    tf.keras.backend.set_floatx(params._precision)
    # Check whether run in graph mode or eager mode.
    tf.config.run_functions_eagerly(not params.train.use_graph_mode)

# def _init_train func
def _init_train():
    """
    Initialize the training process.

    Args:
        None

    Returns:
        None
    """
    pass

"""
data funcs
"""
# def load_data func
def load_data(load_params):
    """
    Load data from specified dataset.

    Args:
        load_params: DotDict - The load parameters of specified dataset.

    Returns:
        dataset_source: tuple - The source dataset, including (X_source, y_source).
        dataset_target: tuple - The target dataset, including (X_target, y_target).
    """
    global params
    # Load data from specified dataset.
    try:
        func = getattr(sys.modules[__name__], "_".join(["_load_data", params.train.dataset]))
        dataset_source, dataset_target = func(load_params)
    except Exception:
        raise ValueError((
            "ERROR: Unknown dataset type {} in train.domain_adaptation.DomainAdversarialNet.SubdomainAdversarialTransformer."
        ).format(params.train.dataset))
    # Return the final `dataset_source` & `dataset_target`.
    return dataset_source, dataset_target

# def _load_data_eeg_zhou2023cibr func
def _load_data_eeg_zhou2023cibr(load_params):
    """
    Load eeg data from the specified subject in `eeg_zhou2023cibr`.

    Args:
        load_params: DotDict - The load parameters of specified dataset.

    Returns:
        dataset_source: tuple - The source dataset, including (X_source, y_class_source, y_domain_source).
        dataset_target: tuple - The target dataset, including (X_target, y_class_target, y_domain_target).
    """
    global params, paths
    # Initialize path_run.
    path_run = os.path.join(paths.base, "data", "eeg.zhou2023cibr", "020", "20230405")\
        if not hasattr(load_params, "path_run") else load_params.path_run
    # Load data from specified subject run.
    datasets = utils.DotDict(); dataset_names = sorted(set(load_params.sourceset) | set(load_params.targetset))
    dataset_domains = ["-".join([dataset_name_i.split("-")[0], dataset_name_i.split("-")[-1]])\
        if dataset_name_i.split("-")[0] == "task" else "-".join(dataset_name_i.split("-")[0:2])\
        for dataset_name_i in dataset_names]
    available_dataset_domains = sorted(set(dataset_domains))
    for dataset_name_i, dataset_domain_i in zip(dataset_names, dataset_domains):
        # Load data from specified dataset name.
        session_type_i = "-".join(dataset_name_i.split("-")[:-1]); data_type_i = dataset_name_i.split("-")[-1]
        func_i = getattr(utils.data.eeg.zhou2023cibr, "_".join(["load_run", session_type_i.split("-")[0]]))
        X_i, y_class_i = func_i(path_run, session_type="-".join(session_type_i.split("-")[1:]))
        # Check whether current dataset has data items.
        if X_i[data_type_i] is None:
            msg = "WARNING: Skip dataset {} with no data item.".format(dataset_name_i)
            print(msg); paths.run.logger.summaries.info(msg); continue
        X_i = X_i[data_type_i].astype(np.float32); y_class_i = y_class_i[data_type_i].astype(np.int64)
        # Get the index according to `available_dataset_domains`.
        y_domain_i = np.repeat(available_dataset_domains.index(dataset_domain_i), repeats=y_class_i.shape[0]).astype(np.int64)
        # Truncate `X_i` to let them have the same length.
        # TODO: Here, we only keep the [0.0~0.8]s-part of [audio,image] that after onset. And we should
        # note that the [0.0~0.8]s-part of image is the whole onset time of image, the [0.0~0.8]s-part
        # of audio is the sum of the whole onset time of audio and the following 0.3s padding.
        # X_i - (n_samples, seq_len, n_channels)
        # If the type of dataset is `default`.
        if load_params.type == "default":
            X_i = X_i[:,20:100,:]
        # If the type of dataset is `lvbj`.
        elif load_params.type == "lvbj":
            X_i = X_i[:,20:100,:]
        # Get unknown type of dataset.
        else:
            raise ValueError("ERROR: Unknown type {} of dataset".format(load_params.type))
        # Do cross-trial normalization.
        X_i = (X_i - np.mean(X_i)) / np.std(X_i)
        # Set the corresponding item of dataset.
        datasets[dataset_name_i] = utils.DotDict({"X":X_i,"y_class":y_class_i,"y_domain":y_domain_i,})
    # Initialize sourceset & targetset.
    X_source = []; y_class_source = []; y_domain_source = []
    X_target = []; y_class_target = []; y_domain_target = []
    for dataset_name_i, dataset_i in datasets.items():
        # If sourceset and targetset are not the same, construct sourceset & targetset separately.
        if dataset_name_i in load_params.sourceset and dataset_name_i not in load_params.targetset:
            X_source.append(dataset_i.X); y_class_source.append(dataset_i.y_class)
            y_domain_source.append(dataset_i.y_domain)
        # If sourceset and targetset are not the same, construct sourceset & targetset separately.
        elif dataset_name_i not in load_params.sourceset and dataset_name_i in load_params.targetset:
            X_target.append(dataset_i.X); y_class_target.append(dataset_i.y_class)
            y_domain_target.append(dataset_i.y_domain)
        # Wrong cases.
        else:
            raise ValueError("ERROR: Unknown dataset name {}.".format(dataset_name_i))
    # Check whether sourceset & targetset both have data items.
    if len(X_source) == 0 or len(X_target) == 0: return [], ([], [], [])
    # X - (n_samples, seq_len, n_channels); y_class - (n_samples,); y_domain - (n_samples,)
    X_source = np.concatenate(X_source, axis=0); y_class_source = np.concatenate(y_class_source, axis=0)
    y_domain_source = np.concatenate(y_domain_source, axis=0)
    X_target = np.concatenate(X_target, axis=0); y_class_target = np.concatenate(y_class_target, axis=0)
    y_domain_target = np.concatenate(y_domain_target, axis=0)
    # Make sure there is no overlap between X_source & X_target.
    samples_same = None; n_samples = 10; assert X_source.shape[1] == X_target.shape[1]
    for _ in range(n_samples):
        sample_idx = np.random.randint(X_source.shape[1])
        sample_same_i = np.intersect1d(X_source[:,sample_idx,0], X_target[:,sample_idx,0], return_indices=True)[-1].tolist()
        samples_same = set(sample_same_i) if samples_same is None else set(sample_same_i) & samples_same
    assert len(samples_same) == 0
    # Check whether labels are enough, then transform y to one-hot encoding.
    try:
        assert len(set(y_class_source)) == len(set(y_class_target)) == params.model.n_labels
        assert (len(set(y_domain_source)) < params.model.n_domains) and (len(set(y_domain_target)) < params.model.n_domains)
        labels = sorted(set(y_class_source)); domains = sorted(set(y_domain_source) | set(y_domain_target))
    except AssertionError as e:
        msg = (
            "WARNING: Skip experiment (source:{};target:{}) due to that the classes of test cases are not enough."
        ).format(set(load_params.sourceset), set(load_params.targetset))
        print(msg); paths.run.logger.summaries.info(msg); return [], ([], [], [])
    # y_class - (n_samples, n_labels)
    y_class_source = np.array([labels.index(y_i) for y_i in y_class_source], dtype=np.int64)
    y_class_source = np.eye(len(labels))[y_class_source]
    y_class_target = np.array([labels.index(y_i) for y_i in y_class_target], dtype=np.int64)
    y_class_target = np.eye(len(labels))[y_class_target]
    # y_domain - (n_samples, n_domains)
    y_domain_source = np.array([domains.index(y_i) for y_i in y_domain_source], dtype=np.int64)
    y_domain_source = np.eye(len(domains))[y_domain_source]
    y_domain_target = np.array([domains.index(y_i) for y_i in y_domain_target], dtype=np.int64)
    y_domain_target = np.eye(len(domains))[y_domain_target]
    # Split target domain into train-set & validation-set & test-set.
    train_ratio = params.train.train_ratio; validation_ratio = test_ratio = (1. - train_ratio) / 2.
    train_idxs = sorted(np.random.choice(np.arange(y_class_target.shape[0]),
        size=int(y_class_target.shape[0] * train_ratio), replace=False))
    validation_idxs = np.random.choice(sorted(set(np.arange(y_class_target.shape[0])) - set(train_idxs)),
        size=int(y_class_target.shape[0] * validation_ratio), replace=False)
    test_idxs = sorted(set(np.arange(y_class_target.shape[0])) - set(train_idxs) - set(validation_idxs))
    assert len(set(train_idxs) & set(validation_idxs)) == 0
    assert len(set(train_idxs) & set(test_idxs)) == 0
    assert len(set(validation_idxs) & set(test_idxs)) == 0
    X_target_train = X_target[train_idxs,:,:]; y_class_target_train = y_class_target[train_idxs,:]
    y_domain_target_train = y_domain_target[train_idxs,:]
    X_target_validation = X_target[validation_idxs,:,:]; y_class_target_validation = y_class_target[validation_idxs,:]
    y_domain_target_validation = y_domain_target[validation_idxs,:]
    X_target_test = X_target[test_idxs,:,:]; y_class_target_test = y_class_target[test_idxs,:]
    y_domain_target_test = y_domain_target[test_idxs,:]
    # Randomly select `load_params.n_samples.targetset_train` samples to format target-train-set.
    if load_params.n_samples.targetset_train is not None:
        train_idxs = np.random.choice(np.arange(len(X_target_train)),
            size=load_params.n_samples.targetset_train, replace=False)
        X_target_train = X_target_train[train_idxs,:,:]
        y_class_target_train = y_class_target_train[train_idxs,:]
        y_domain_target_train = y_domain_target_train[train_idxs,:]
    # Log information of data loading.
    msg = (
        "INFO: Data preparation complete, with source-set ({}) & target-train-set ({})" +\
        " & target-validation-set ({}) & target-test-set ({})."
    ).format(X_source.shape, X_target_train.shape, X_target_validation.shape, X_target_test.shape)
    print(msg); paths.run.logger.summaries.info(msg)
    # Construct dataset from data items.
    dataset_source = tf.data.Dataset.from_tensor_slices((X_source, y_class_source, y_domain_source))
    dataset_target_train = tf.data.Dataset.from_tensor_slices((X_target_train, y_class_target_train, y_domain_target_train))
    dataset_target_validation = tf.data.Dataset.from_tensor_slices(
        (X_target_validation, y_class_target_validation, y_domain_target_validation)
    )
    dataset_target_test = tf.data.Dataset.from_tensor_slices((X_target_test, y_class_target_test, y_domain_target_test))
    # Shuffle and then batch the dataset.
    dataset_source = dataset_source.shuffle(params.train.buffer_size).batch(params.train.batch_size, drop_remainder=False)
    dataset_target_train = dataset_target_train.shuffle(params.train.buffer_size).batch(
        params.train.batch_size, drop_remainder=False)
    dataset_target_validation = dataset_target_validation.shuffle(params.train.buffer_size).batch(
        params.train.batch_size, drop_remainder=False)
    dataset_target_test = dataset_target_test.shuffle(params.train.buffer_size).batch(
        params.train.batch_size, drop_remainder=False)
    # Return the final `dataset_source` & `dataset_target`.
    return dataset_source, (dataset_target_train, dataset_target_validation, dataset_target_test)

"""
train funcs
"""
# def train func
def train(base_, params_):
    """
    Train the model.

    Args:
        base_: str - The base path of current project.
        params_: str - The parameters of current training process.

    Returns:
        None
    """
    global params, paths
    # Initialize parameters & variables of current training process.
    init(base_, params_)
    # Log the start of current training process.
    paths.run.logger.summaries.info("Training started with dataset {}.".format(params.train.dataset))
    # Initialize load_params. Each load_params_i corresponds to a sub-dataset.
    if params.train.dataset == "eeg_zhou2023cibr":
        # Initialize the paths of runs that we want to execute experiments.
        path_runs = [
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "005", "20221223"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "006", "20230103"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "007", "20230106"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "011", "20230214"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "013", "20230308"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "018", "20230331"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "019", "20230403"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "020", "20230405"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "021", "20230407"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "023", "20230412"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "024", "20230414"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "025", "20230417"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "026", "20230419"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "027", "20230421"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "028", "20230424"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "029", "20230428"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "030", "20230504"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "031", "20230510"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-002", "20230509"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "033", "20230517"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "034", "20230519"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-003", "20230524"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-004", "20230528"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-005", "20230601"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "036", "20230526"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "037", "20230529"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "038", "20230531"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "039", "20230605"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "040", "20230607"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-006", "20230603"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-007", "20230608"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-008", "20230610"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "042", "20230614"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "043", "20230616"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "044", "20230619"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "045", "20230626"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "046", "20230628"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-009", "20230613"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-010", "20230615"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-012", "20230623"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-013", "20230627"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-014", "20230629"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-015", "20230701"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "047", "20230703"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "048", "20230705"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-016", "20230703"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-017", "20230706"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "049", "20230710"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "050", "20230712"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "051", "20230717"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "052", "20230719"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-018", "20230710"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-019", "20230712"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-020", "20230714"),
            os.path.join(paths.base, "data", "eeg.zhou2023cibr", "sz-021", "20230718"),
        ]; load_type = "default"; n_samples = utils.DotDict({"targetset_train":100,})
        # `load_params` contains all the experiments that we want to execute for every run.
        load_params = [
            # source-task-all-image-target-task-all-audio
            utils.DotDict({
                "name": "source-task-all-image-target-task-all-audio",
                "sourceset": [
                    "task-image-audio-pre-image", "task-audio-image-pre-image",
                    "task-image-audio-post-image", "task-audio-image-post-image",
                ],
                "targetset": [
                    "task-image-audio-pre-audio", "task-audio-image-pre-audio",
                    "task-image-audio-post-audio", "task-audio-image-post-audio",
                ],
                "type": load_type, "n_samples": n_samples, "n_domains": 2,
            }),
            # source-task-all-all-target-tmr-n23-audio
            utils.DotDict({
                "name": "source-task-all-all-target-tmr-n23-audio",
                "sourceset": [
                    "task-image-audio-pre-image", "task-audio-image-pre-image",
                    "task-image-audio-post-image", "task-audio-image-post-image",
                    "task-image-audio-pre-audio", "task-audio-image-pre-audio",
                    "task-image-audio-post-audio", "task-audio-image-post-audio",
                ],
                "targetset": ["tmr-N2/3-audio",],
                "type": load_type, "n_samples": n_samples, "n_domains": 3,
            }),
            # source-task-all-image-target-tmr-n23-audio
            utils.DotDict({
                "name": "source-task-all-image-target-tmr-n23-audio",
                "sourceset": [
                    "task-image-audio-pre-image", "task-audio-image-pre-image",
                    "task-image-audio-post-image", "task-audio-image-post-image",
                ],
                "targetset": ["tmr-N2/3-audio",],
                "type": load_type, "n_samples": n_samples, "n_domains": 2,
            }),
            # source-task-all-audio-target-tmr-n23-audio
            utils.DotDict({
                "name": "source-task-all-audio-target-tmr-n23-audio",
                "sourceset": [
                    "task-image-audio-pre-audio", "task-audio-image-pre-audio",
                    "task-image-audio-post-audio", "task-audio-image-post-audio",
                ],
                "targetset": ["tmr-N2/3-audio",],
                "type": load_type, "n_samples": n_samples, "n_domains": 2,
            }),
        ]
    else:
        raise ValueError("ERROR: Unknown dataset {} in train.naive_rnn.".format(params.train.dataset))
    # Execute experiments for each dataset run.
    for path_run_i in path_runs:
        # Log the start of current training iteration.
        msg = "Training started with sub-dataset {}.".format(path_run_i)
        print(msg); paths.run.logger.summaries.info(msg)
        for load_params_idx in range(len(load_params)):
            # Add `path_run` to `load_params_i`.
            load_params_i = cp.deepcopy(load_params[load_params_idx]); load_params_i.path_run = path_run_i
            # Update `params` according to `load_params_i`.
            params.model.n_domains = params.model.cls_domain.d_output = load_params_i.n_domains
            # Load data from specified experiment.
            dataset_source, dataset_target = load_data(load_params_i)
            dataset_target_train, dataset_target_validation, dataset_target_test = dataset_target
            # Check whether source-set & target-set exists.
            if len(dataset_source) == 0 or len(dataset_target) == 0:
                msg = (
                    "INFO: Skip experiment {} with source-set ({:d} items) & target-set ({:d} items)."
                ).format(load_params_i.name, len(dataset_source), len(dataset_target))
                print(msg); paths.run.logger.summaries.info(msg); continue
            # Start training process of current specified experiment.
            msg = "Start the training process of experiment {}.".format(load_params_i.name)
            print(msg); paths.run.logger.summaries.info(msg)

            # Train the model for each time segment.
            accuracies_source = []; accuracies_target_validation = []; accuracies_target_test = []

            # Initialize model of current time segment.
            model = sdatn_model(params.model)
            for epoch_idx in range(params.train.n_epochs):
                # Update iteration parameters.
                params.iteration(iteration=epoch_idx)
                # Record the start time of preparing data.
                time_start = time.time()
                # Prepare for model train process.
                accuracy_source = []; batch_size_source = []; loss_source = utils.DotDict({"class": [], "domain": [],})
                # Train model for current epoch.
                iter_source = iter(dataset_source); n_batches = len(dataset_source)
                iter_target = iter(dataset_target_train)
                for batch_idx in range(n_batches):
                    # Initialize `inputs_*_i` from `iter_*`.
                    inputs_source_i = iter_source.next(); inputs_target_i = iter_target.next()
                    inputs_i = utils.DotDict({"source": inputs_source_i, "target": inputs_target_i,})
                    # Update `batch_size_source`.
                    batch_size_source_i = len(inputs_source_i[0]); batch_size_source.append(batch_size_source_i)
                    # If `iter_target` is used up, re-instantiate `iter_target`.
                    if (batch_idx + 1) % len(dataset_target_train) == 0: iter_target = iter(dataset_target_train)
                    # Train model for current batch.
                    y_pred_i, loss_i = model.train(inputs_i, params.model)
                    # Calculate `accuracy_source_i`.
                    y_pred_class_source_i = y_pred_i["source"]["class"].numpy()
                    y_true_class_source_i = inputs_source_i[1].numpy()
                    accuracy_source_i = np.argmax(y_pred_class_source_i, axis=-1) == np.argmax(y_true_class_source_i, axis=-1)
                    accuracy_source_i = np.sum(accuracy_source_i) / accuracy_source_i.size
                    accuracy_source.append(accuracy_source_i)
                    # Update `loss_source`.
                    loss_source["class"].append(loss_i["source"]["class"].numpy().mean())
                    loss_source["domain"].append(loss_i["source"]["domain"].numpy().mean())
                # Calculate the averaged `accuracy` & `loss` corresponding to source domain.
                accuracy_source = np.sum(np.array(accuracy_source) * np.array(batch_size_source)) / np.sum(batch_size_source)
                loss_source["class"] = np.sum(np.array(loss_source["class"]) *\
                    np.array(batch_size_source)) / np.sum(batch_size_source)
                loss_source["domain"] = np.sum(np.array(loss_source["domain"]) *\
                    np.array(batch_size_source)) / np.sum(batch_size_source)
                # Prepare for model validation process.
                accuracy_target_validation = []; batch_size_target_validation = []
                loss_target_validation = utils.DotDict({"class": [], "domain": [],})
                # Test model for current epoch.
                iter_target = iter(dataset_target_validation); n_batches = len(dataset_target_validation)
                for batch_idx in range(n_batches):
                    # Initialize `inputs_target_i` from `iter_target`.
                    inputs_target_i = iter_target.next()
                    # Update `batch_size_target`.
                    batch_size_target_i = len(inputs_target_i[0]); batch_size_target_validation.append(batch_size_target_i)
                    # Test model for current batch.
                    y_pred_i, loss_i = model(inputs_target_i, params.model)
                    # Calculate `accuracy_target_i`.
                    y_pred_class_target_i = y_pred_i["class"].numpy()
                    y_true_class_target_i = inputs_target_i[1].numpy()
                    accuracy_target_i = np.argmax(y_pred_class_target_i, axis=-1) == np.argmax(y_true_class_target_i, axis=-1)
                    accuracy_target_i = np.sum(accuracy_target_i) / accuracy_target_i.size
                    accuracy_target_validation.append(accuracy_target_i)
                    # Update `loss_target`.
                    loss_target_validation["class"].append(loss_i["class"].numpy().mean())
                    loss_target_validation["domain"].append(loss_i["domain"].numpy().mean())
                # Calculate the averaged `accuracy` & `loss` corresponding to target domain.
                accuracy_target_validation = np.sum(np.array(accuracy_target_validation) *\
                    np.array(batch_size_target_validation)) / np.sum(batch_size_target_validation)
                loss_target_validation["class"] = np.sum(np.array(loss_target_validation["class"]) *\
                    np.array(batch_size_target_validation)) / np.sum(batch_size_target_validation)
                loss_target_validation["domain"] = np.sum(np.array(loss_target_validation["domain"]) *\
                    np.array(batch_size_target_validation)) / np.sum(batch_size_target_validation)
                # Prepare for model test process.
                accuracy_target_test = []; batch_size_target_test = []
                loss_target_test = utils.DotDict({"class": [], "domain": [],})
                # Test model for current epoch.
                iter_target = iter(dataset_target_test); n_batches = len(dataset_target_test)
                for batch_idx in range(n_batches):
                    # Initialize `inputs_target_i` from `iter_target`.
                    inputs_target_i = iter_target.next()
                    # Update `batch_size_target`.
                    batch_size_target_i = len(inputs_target_i[0]); batch_size_target_test.append(batch_size_target_i)
                    # Test model for current batch.
                    y_pred_i, loss_i = model(inputs_target_i, params.model)
                    # Calculate `accuracy_target_i`.
                    y_pred_class_target_i = y_pred_i["class"].numpy()
                    y_true_class_target_i = inputs_target_i[1].numpy()
                    accuracy_target_i = np.argmax(y_pred_class_target_i, axis=-1) == np.argmax(y_true_class_target_i, axis=-1)
                    accuracy_target_i = np.sum(accuracy_target_i) / accuracy_target_i.size
                    accuracy_target_test.append(accuracy_target_i)
                    # Update `loss_target`.
                    loss_target_test["class"].append(loss_i["class"].numpy().mean())
                    loss_target_test["domain"].append(loss_i["domain"].numpy().mean())
                # Calculate the averaged `accuracy` & `loss` corresponding to target domain.
                accuracy_target_test = np.sum(np.array(accuracy_target_test) *\
                    np.array(batch_size_target_test)) / np.sum(batch_size_target_test)
                loss_target_test["class"] = np.sum(np.array(loss_target_test["class"]) *\
                    np.array(batch_size_target_test)) / np.sum(batch_size_target_test)
                loss_target_test["domain"] = np.sum(np.array(loss_target_test["domain"]) *\
                    np.array(batch_size_target_test)) / np.sum(batch_size_target_test)
                # Log information related to current training epoch.
                time_stop = time.time(); accuracies_source.append(accuracy_source)
                accuracies_target_validation.append(accuracy_target_validation)
                accuracies_target_test.append(accuracy_target_test)
                msg = (
                    "Finish train epoch {:d} in {:.2f} seconds, generating {:d} concrete functions."
                ).format(epoch_idx, time_stop-time_start, len(model.train.pretty_printed_concrete_signatures().split("\n\n")))
                print(msg); paths.run.logger.summaries.info(msg)
                msg = (
                    "Accuracy(source): {:.2f}%. Loss(source): {:.5f} (class) {:.5f} (domain)."
                ).format(accuracy_source * 100., loss_source["class"], loss_source["domain"])
                print(msg); paths.run.logger.summaries.info(msg)
                msg = (
                    "Accuracy(target-validation): {:.2f}%. Loss(target): {:.5f} (class) {:.5f} (domain)."
                ).format(accuracy_target_validation * 100., loss_target_validation["class"], loss_target_validation["domain"])
                print(msg); paths.run.logger.summaries.info(msg)
                msg = (
                    "Accuracy(target-test): {:.2f}%. Loss(target): {:.5f} (class) {:.5f} (domain)."
                ).format(accuracy_target_test * 100., loss_target_test["class"], loss_target_test["domain"])
                print(msg); paths.run.logger.summaries.info(msg)
                # Summarize model information.
                if epoch_idx == 0:
                    model.summary(print_fn=print); model.summary(print_fn=paths.run.logger.summaries.info)
            # Convert `accuracies_source` & `accuracies_target` to `np.array`.
            accuracies_target_validation = np.round(np.array(accuracies_target_validation, dtype=np.float32), decimals=4)
            accuracies_target_test = np.round(np.array(accuracies_target_test, dtype=np.float32), decimals=4)
            epoch_maxacc_idxs = np.where(accuracies_target_validation == np.max(accuracies_target_validation))[0]
            epoch_maxacc_idx = epoch_maxacc_idxs[np.argmax(accuracies_target_test[epoch_maxacc_idxs])]
            # Finish training process of current specified experiment.
            msg = (
                "Finish the training process of experiment {}, with test-accuracy ({:.2f}%)" +\
                " according to max validation-accuracy ({:.2f}%) at epoch {:d}, generating {:d} concrete functions."
            ).format(load_params_i.name, accuracies_target_test[epoch_maxacc_idx]*100.,
                accuracies_target_validation[epoch_maxacc_idx]*100., epoch_maxacc_idx,
                len(model.train.pretty_printed_concrete_signatures().split("\n\n")))
            print(msg); paths.run.logger.summaries.info(msg)
        # Log the end of current training iteration.
        msg = "Training finished with sub-dataset {}.".format(path_run_i)
        print(msg); paths.run.logger.summaries.info(msg)
    # Log the end of current training process.
    msg = "Training finished with dataset {}.".format(params.train.dataset)
    print(msg); paths.run.logger.summaries.info(msg)

if __name__ == "__main__":
    import os
    # local dep
    from params.domain_adaptation_params import subdomain_adversarial_transformer_params

    # macro
    dataset = "eeg_zhou2023cibr"

    # Initialize random seed.
    utils.model.set_seeds(42)

    ## Instantiate naive_rnn.
    # Initialize base.
    base = os.path.join(os.getcwd(), os.pardir, os.pardir, os.pardir, os.pardir)
    # Instantiate subdomain_adversarial_transformer_params.
    subdomain_adversarial_transformer_params_inst = subdomain_adversarial_transformer_params(dataset=dataset)
    # Train `SubdomainAdversarialTransformer`.
    train(base, subdomain_adversarial_transformer_params_inst)

