from dataclasses import dataclass
from typing import Optional, Dict, Any
from enum import Enum


class ActionType(Enum):
    CONNECT = "connect"
    STOP = "stop"


@dataclass
class AssemblyAction:
    action_type: ActionType
    source_motif: Optional[str] = None
    source_site: Optional[int] = None
    target_motif: Optional[str] = None
    target_site: Optional[int] = None
    bond_type: Optional[str] = None
    confidence: float = 1.0
    reasoning: str = ""

    def is_stop_action(self) -> bool:
        return self.action_type == ActionType.STOP

    def is_connect_action(self) -> bool:
        return self.action_type == ActionType.CONNECT

    def to_dict(self) -> Dict[str, Any]:
        return {
            'action_type': self.action_type.value,
            'source_motif': self.source_motif,
            'source_site': self.source_site,
            'target_motif': self.target_motif,
            'target_site': self.target_site,
            'bond_type': self.bond_type,
            'confidence': self.confidence,
            'reasoning': self.reasoning
        }

    def to_text_representation(self) -> str:
        if self.is_stop_action():
            return f"ACTION: STOP (confidence: {self.confidence:.3f})"

        return (
            f"ACTION: CONNECT {self.source_motif}[site_{self.source_site}] "
            f"--{self.bond_type}--> {self.target_motif}[site_{self.target_site}] "
            f"(confidence: {self.confidence:.3f})"
        )

    @classmethod
    def create_stop_action(cls, confidence: float = 1.0, reasoning: str = "") -> 'AssemblyAction':
        return cls(
            action_type=ActionType.STOP,
            confidence=confidence,
            reasoning=reasoning
        )

    @classmethod
    def create_connect_action(cls, source_motif: str, source_site: int,
                            target_motif: str, target_site: int,
                            bond_type: str, confidence: float = 1.0,
                            reasoning: str = "") -> 'AssemblyAction':
        return cls(
            action_type=ActionType.CONNECT,
            source_motif=source_motif,
            source_site=source_site,
            target_motif=target_motif,
            target_site=target_site,
            bond_type=bond_type,
            confidence=confidence,
            reasoning=reasoning
        )

    def is_valid_connect_action(self) -> bool:
        if not self.is_connect_action():
            return False

        return all([
            self.source_motif is not None,
            self.source_site is not None,
            self.target_motif is not None,
            self.target_site is not None,
            self.bond_type is not None,
            self.source_motif != self.target_motif
        ])

    def __eq__(self, other: 'AssemblyAction') -> bool:
        if not isinstance(other, AssemblyAction):
            return False

        if self.action_type != other.action_type:
            return False

        if self.is_stop_action():
            return True

        return (
            self.source_motif == other.source_motif and
            self.source_site == other.source_site and
            self.target_motif == other.target_motif and
            self.target_site == other.target_site and
            self.bond_type == other.bond_type
        )

    def __hash__(self) -> int:
        if self.is_stop_action():
            return hash(("STOP",))

        return hash((
            self.source_motif,
            self.source_site,
            self.target_motif,
            self.target_site,
            self.bond_type
        ))