import argparse
import numpy as np
import os
import sys
sys.path.append('../share')
sys.path.append('../model')

import torch
import torch.backends.cudnn as cudnn
from models_mage import MAGECityPolyGen
from models_pospred import MAGECityPosition
from function import pad_poly, in_poly_idx

import json
import cv2
from shapely.geometry import Polygon



def infgen(model, modelpos, device, discard_prob = 0.4,  discre = 50, max_build = 60, use_sample=False, finetune = False):

    remain_flag = torch.ones(discre**2).to(device)
        
    endflag = 0
    gen_iter = []

    num_poly = 0
    
    for npoly in range(max_build):
        if npoly == 0:     
            x, y = torch.rand([1])*discre, torch.rand([1])*discre
            pred_pos = torch.cat([x, y],dim=0).unsqueeze(0).unsqueeze(0)
            
            predpoly = model.genfirst(pred_pos)[0]
            
            poly_add, pos_add = pad_poly(predpoly)
            poly_add = poly_add.to(device)
            pos_add = pos_add.to(device)
            samples_iter = poly_add.detach()
            pos_iter = pos_add.detach()
            num_poly+=1

            gen_iter.append(predpoly.squeeze(0).detach())
            remain_flag = remain_flag*in_poly_idx(predpoly.squeeze(0).detach().cpu()).to(device)
        else:
            pred = modelpos(samples_iter, pos_iter, None, generate = True)
            prob_pred = torch.sigmoid(pred[0, :])
            prob_pred = torch.where(prob_pred < discard_prob, torch.tensor(0.0).to(device), prob_pred) 
            while True:
                prob_pred = prob_pred*remain_flag
                if torch.max(prob_pred) < discard_prob:
                    endflag = 1
                    break      
                if use_sample == False:
                    idx_iter = torch.argmax(prob_pred)
                else:            
                    idx_iter = torch.multinomial(prob_pred, 1).squeeze(0)
                                    
                remain_flag[idx_iter] = 0

                pred_pos = torch.cat([idx_iter.unsqueeze(0)//discre, idx_iter.unsqueeze(0)%discre],dim=0).unsqueeze(0).unsqueeze(0)
                predpoly = model.infgen(samples_iter, pos_iter, pred_pos)[0]

                polygon = Polygon(np.array(predpoly[0].clone().detach().cpu()))
                intersect_flag = 0
                for pe in range(samples_iter.shape[1]):
                    polyexist = [] 
                    for k in range(samples_iter.shape[2]):
                        if samples_iter[0, pe, k, 0] != 0:
                            point = samples_iter[0, pe, k, :].clone().detach().cpu().numpy()
                            polyexist.append(point)
                    polyexist = Polygon(polyexist)
                    if polygon.intersects(polyexist):
                        intersect_flag = 1
                        break

                if intersect_flag == 0:
                    break

            if endflag == 1:
                break
            
            poly_add, pos_add = pad_poly(predpoly)
            poly_add = poly_add.to(device)
            pos_add = pos_add.to(device)
            samples_iter = torch.cat([samples_iter, poly_add], dim = 1).detach()
            pos_iter = torch.cat([pos_iter, pos_add], dim = 1).detach()
            num_poly+=1

            gen_iter.append(predpoly.squeeze(0).detach())
            remain_flag = remain_flag*in_poly_idx(predpoly.squeeze(0).detach().cpu()).to(device)

    if finetune:
        gen_iter = model.genfirst(pos_iter)
        gen_iter = [genpoly.squeeze(0).detach() for genpoly in gen_iter]
 
    print("gen:", len(gen_iter))
    return gen_iter
  

def get_args_parser():
    parser = argparse.ArgumentParser('MAE pre-training', add_help=False)
    parser.add_argument('--batch_size', default=1, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=1, type=int)

    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)

    parser.add_argument('--num_workers', default=20, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    
    parser.add_argument('--stage1_model_path', type = str, default='../results/model/stage1_2d.pth')
    parser.add_argument('--stage2_model_path', type = str, default='../results/model/stage2_2d.pth')
    parser.add_argument('--save_path', type = str, default="../results/test/states_poly_gen_2d/")
    
    parser.set_defaults(pin_mem=True)
    return parser

def main(args):
    device = torch.device(args.device)

    seed = args.seed
    torch.manual_seed(seed)
    np.random.seed(seed)

    cudnn.benchmark = True

    modelpos = MAGECityPosition(embed_dim=256, depth=6, num_heads=8, 
                                 decoder_embed_dim=16, decoder_depth=3,   
                                 decoder_num_heads=8, discre = 50, patch_size = 5, patch_num = 10, 
                                 device = device, ablation = False, patchify = True)
    
    pretrained_modelpos = torch.load(args.stage1_model_path)
    modelpos.load_state_dict(pretrained_modelpos, strict = False)
    modelpos.to(device)
    modelpos.eval()
    
    model = MAGECityPolyGen(num_heads=8, device=device,
                        depth=12, embed_dim=512, decoder_embed_dim=512,
                        decoder_depth=8, decoder_num_heads=8,
                        max_build=60, max_poly=20, discre=50)
    
    pretrained_model = torch.load(args.stage2_model_path)
    model.load_state_dict(pretrained_model)
    model.to(device)
    model.eval()

    image_num = 0
    
    assert args.batch_size == 1
    use_s = False

    list_count=[]
    list_ratio=[]
    list_area=[]
    list_edge=[]

    list_poly=[]
    unvalid = 0
    total_count = 0

    size = 500

    for t in range(1000):
        gen_poly = infgen(model = model, modelpos = modelpos, device = device,use_sample=use_s)

        total_area=0
        list_count.append(len(gen_poly))
        list_poly.append(np.array([np.array(poly.cpu()) for poly in gen_poly], dtype=object))

        img = np.ones((size,size,3),np.uint8)*255

        for num, poly in enumerate(gen_poly):
            total_count+=1
            
            pts = np.array(poly.cpu(), np.int32)
            pts = pts.reshape((-1,1,2)).astype(int)
            list_edge.append(len(poly))
            if Polygon(poly.cpu()).is_valid:
                areatem = Polygon(poly.cpu()).area
                list_area.append(areatem)
                total_area+= areatem
            else:
                unvalid+=1

            cv2.fillPoly(img, [pts], color=(255, 255, 0))
        list_ratio.append(total_area/size**2)

        for num, poly in enumerate(gen_poly):
            pts = np.array(poly.cpu(), np.int32)
            pts = pts.reshape((-1,1,2)).astype(int)

            cv2.polylines(img,[pts],True,(0,0,0),1)
    
        dir_path = args.save_path+'img/'
        if not os.path.exists(dir_path):
            os.makedirs(dir_path)
        image_num += 1
        print(image_num)

                
        cv2.imwrite(f'{dir_path}'+str(t) +'.jpg',img)
    
    dict_100 = {'count': list_count,
                'area': list_area,
                'edge': list_edge,
                "ratio":list_ratio}

    with open(f'{args.save_path}'+'wd.json', 'w') as json_file:
        json.dump(dict_100, json_file)

    np.save(f'{args.save_path}'+'poly.npy', np.array(list_poly, dtype=object))
    print(unvalid/total_count)


if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    main(args)

