import argparse
import numpy as np
import sys
import os
import torch
import torch.backends.cudnn as cudnn
sys.path.append('../share')
sys.path.append('../model')

from models_pospred import MAGECityPosition
from dataloader import PolyDataset2D
from random_mask import random_masking_test

from dataloader import PolyDataset2D

import cv2

def get_args_parser():
    parser = argparse.ArgumentParser('MAE pre-training', add_help=False)
    parser.add_argument('--batch_size', default=1, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=1, type=int)

    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)

    parser.add_argument('--num_workers', default=20, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    
    parser.add_argument('--stage1_model_path', type = str, default='../results/model/stage1_2d.pth')
    parser.add_argument('--save_path', type = str, default="../results/test/states_positionmap/")
    parser.add_argument('--data_path', type=str, default='../datasets/statespoly')

    parser.set_defaults(pin_mem=True)

    return parser

def main(args):
    device = torch.device(args.device)

    seed = args.seed
    torch.manual_seed(seed)
    np.random.seed(seed)

    cudnn.benchmark = True
    if args.save_path is not None:
        os.makedirs(args.save_path, exist_ok=True)
        
    dataset_valid = PolyDataset2D(args.data_path, train=False,split_ratio = 0.8)

    data_loader_valid = torch.utils.data.DataLoader(
        dataset_valid, 
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )

    model = MAGECityPosition(embed_dim=256, depth=6, num_heads=8, 
                                 decoder_embed_dim=16, decoder_depth=3,   
                                 decoder_num_heads=8, discre = 50, patch_size = 5, patch_num = 10, 
                                 device = device, ablation = False, patchify = True)
    
    pretrained_model = torch.load(args.stage1_model_path)
    model.load_state_dict(pretrained_model, strict = False)
    model.to(device)
    model.eval()


    remain_num = 6

    for valid_step, (samples, pos, info) in enumerate(data_loader_valid):
        if valid_step >=50:
            break

        poly_reserve, pos_reserve, _, pos_tar, _, ids_keep = random_masking_test(samples, pos, info, remain_num, max_build = 60)

        poly_reserve = poly_reserve.to(device)
        pos_reserve = pos_reserve.to(device)    

        img = np.ones((500,500,3),np.uint8)*255

        for i in range(int(info[0, 0])):   
            if i not in ids_keep:
                pts = np.array(samples[0, i][:int(info[0][i+1]), :].cpu(), np.int32)
                pts = pts.reshape((-1,1,2)).astype(int)
                cv2.fillPoly(img, [pts], color=(255, 255, 0))
                cv2.polylines(img,[pts],True,(0,0,0),1)

        for id in ids_keep:
            pts = np.array(samples[0, id][:int(info[0][id+1]), :].cpu(), np.int32)
            pts = pts.reshape((-1,1,2)).astype(int)
            cv2.fillPoly(img, [pts], color=(238, 159, 153))
            cv2.polylines(img,[pts],True,(0,0,0),1)


        with torch.no_grad():
            loss, pred = model(poly_reserve, pos_reserve, pos_tar)

        pred = torch.sigmoid(pred).view(50,50,1).permute(1,0,2).repeat(1,1,3).detach().cpu()
        posimg_pred = np.array(pred*255, dtype=np.uint8)

        cv2.imwrite(f'{args.save_path}'+str(valid_step)+ 'poly' +'.jpg',img)
        cv2.imwrite(f'{args.save_path}'+str(valid_step) +'.jpg',posimg_pred)

        print(valid_step, loss.item())      
                  

if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    main(args)

    