import argparse
import numpy as np
import os
import sys
sys.path.append('../share')
sys.path.append('../model')

import torch
import torch.backends.cudnn as cudnn
from models_mage import MAGECityPolyGenRoad
from models_pospred import MAGECityPositionRoad
from dataloader import PolyDatasetRoad
from function import pad_poly, in_poly_idx
from random_mask import random_masking

import cv2

from shapely.geometry import LineString, Polygon


def infgen(model, modelpos, road, samples_iter, pos_iter, road_poly, device, discard_prob = 0.5, discre = 100, max_build = 250, use_sample=False, finetune = False):

    remain_flag = torch.ones(discre**2).to(device)
        
    endflag = 0
    gen_iter = []

    num_poly = 0

    for npoly in range(max_build):
        pred = modelpos(samples_iter, pos_iter, None, road, generate = True)
        prob_pred = torch.sigmoid(pred[0, :])
        prob_pred = torch.where(prob_pred < discard_prob, torch.tensor(0.0).to(device), prob_pred) 
        while True:
            prob_pred = prob_pred*remain_flag
            if torch.max(prob_pred) < discard_prob:
                endflag = 1
                break      
            if use_sample == False:
                idx_iter = torch.argmax(prob_pred)
            else:            
                idx_iter = torch.multinomial(prob_pred, 1).squeeze(0)
                                
            remain_flag[idx_iter] = 0

            pred_pos = torch.cat([idx_iter.unsqueeze(0)//discre, idx_iter.unsqueeze(0)%discre],dim=0).unsqueeze(0).unsqueeze(0)
            predpoly = model.infgen(samples_iter, pos_iter, pred_pos, road)[0]

            polygon = Polygon(np.array(predpoly[0].clone().detach().cpu()))
            intersect_flag = 0
            for pe in range(samples_iter.shape[1]):
                polyexist = [] 
                for k in range(samples_iter.shape[2]):
                    if samples_iter[0, pe, k, 0] != 0:
                        point = samples_iter[0, pe, k, :].clone().detach().cpu().numpy()
                        polyexist.append(point)
                polyexist = Polygon(polyexist)
                if polygon.intersects(polyexist):
                    intersect_flag = 1
                    break

            for polyexist in road_poly:
                if polygon.intersects(polyexist):
                    intersect_flag = 1
                    break

            if intersect_flag == 0:
                break

        if endflag == 1:
            break
        
        poly_add, pos_add = pad_poly(predpoly)
        poly_add = poly_add.to(device)
        pos_add = pos_add.to(device)
        samples_iter = torch.cat([samples_iter, poly_add], dim = 1).detach()
        pos_iter = torch.cat([pos_iter, pos_add], dim = 1).detach()
        num_poly+=1

        gen_iter.append(predpoly.squeeze(0).detach())
        remain_flag = remain_flag*in_poly_idx(predpoly.squeeze(0).detach().cpu(), discre=discre).to(device)

    if finetune:
        gen_iter = model.genfirst(pos_iter)
        gen_iter = [genpoly.squeeze(0).detach() for genpoly in gen_iter]
 
    return gen_iter


def road_to_polygon(road, offset_distance = 20):
    poly = []
    for line in road[0, :int(np.where(road[0, :, 0, 0]+road[0, :, 0, 1] == 0)[0][0])]:
        if np.where(line[:, 0]+line[:, 1] == 0)[0].size > 0:
            line = LineString(np.array(line[:int(np.where(line[:, 0]+line[:, 1] == 0 )[0][0])], np.int32))
        else:
            line = LineString(np.array(line, np.int32))
        offset_line = line.parallel_offset(offset_distance, 'right')
        offset_line_ = line.parallel_offset(offset_distance, 'left')
        poly.append(Polygon(np.concatenate([np.array(offset_line_.coords), np.flip(np.array(offset_line.coords), axis = 0)], axis = 0)))
    return poly

def get_args_parser():
    parser = argparse.ArgumentParser('MAE pre-training', add_help=False)
    parser.add_argument('--batch_size', default=1, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=1, type=int)

    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)

    parser.add_argument('--num_workers', default=20, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    
    parser.add_argument('--stage1_model_path', type = str, default='../results/model/stage1_2d_road.pth')
    parser.add_argument('--stage2_model_path', type = str, default='../results/model/stage2_2d_road.pth')
    parser.add_argument('--save_path', type = str, default="../results/test/states_poly_road/")
    parser.add_argument('--data_path', type=str, default='../datasets/dataroad')
    
    parser.set_defaults(pin_mem=True)

    return parser


def main(args):
    device = torch.device(args.device)

    seed = args.seed
    torch.manual_seed(seed)
    np.random.seed(seed)

    cudnn.benchmark = True

    dataset_test = PolyDatasetRoad(args.data_path, train=False,split_ratio = 0.8)

    data_loader_test = torch.utils.data.DataLoader(
        dataset_test, 
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )
    
    modelpos = MAGECityPositionRoad(embed_dim=512, depth=6, num_heads=8, 
                                 decoder_embed_dim=16, decoder_depth=3,   
                                 decoder_num_heads=8, discre = 100, patch_size = 10, patch_num = 10, 
                                 device = device, append_road = True)
    
    pretrained_modelpos = torch.load(args.stage1_model_path)
    modelpos.load_state_dict(pretrained_modelpos, strict = False)
    modelpos.to(device)
    modelpos.eval()
    
    model = MAGECityPolyGenRoad(num_heads=8, device=device,
                        depth=12, embed_dim=512, decoder_embed_dim=512,
                        decoder_depth=8, decoder_num_heads=8,
                        max_build=250, max_poly=20, discre=100, append_road = True)
    
    pretrained_model = torch.load(args.stage2_model_path)
    model.load_state_dict(pretrained_model)
    model.to(device)
    model.eval()

    image_num = 0
    
    assert args.batch_size == 1
    use_s = False
    size = 1000

    for valid_step, (samples, pos, info, road) in enumerate(data_loader_test):
        if valid_step >=500:
            break
        
        road_poly = road_to_polygon(road)
        road = road.to(device).float()
        poly_reserve, pos_reserve, _, _, _ = random_masking(samples, pos, info, 0, max_build=250)

        poly_reserve = poly_reserve.to(device)
        pos_reserve = pos_reserve.to(device)
        
        gen_poly = infgen(model = model, modelpos = modelpos, road = road, samples_iter = poly_reserve, pos_iter = pos_reserve, road_poly = road_poly, device = device,use_sample=use_s)

        img = np.ones((size,size,3),np.uint8)*255

        road = road.detach().cpu()
        r = road[0, :int(np.where(road[0, :, 0, 0]+road[0, :, 0, 1] == 0)[0][0])]
        for ro in r:
            if np.where(ro[:, 0]+ro[:, 1] == 0)[0].size > 0:
                string = np.array(ro[:int(np.where(ro[:, 0]+ro[:, 1] == 0 )[0][0])], np.int32)
            else:
                string = np.array(ro, np.int32)
            cv2.polylines(img,[string],False,(128, 0, 0),thickness=20)
        
        for _, poly in enumerate(gen_poly):          
            pts = np.array(poly.cpu(), np.int32)
            pts = pts.reshape((-1,1,2)).astype(int)
            cv2.fillPoly(img, [pts], color=(255, 255, 0))

        for _, poly in enumerate(gen_poly):
            pts = np.array(poly.cpu(), np.int32)
            pts = pts.reshape((-1,1,2)).astype(int)

            cv2.polylines(img,[pts],True,(0,0,0),1)
    
        dir_path = args.save_path+'img/'
        if not os.path.exists(dir_path):
            os.makedirs(dir_path)
        image_num += 1
        print(image_num)
                
        cv2.imwrite(f'{dir_path}'+str(image_num) +'.jpg',img)
    
if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    main(args)

