# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.

"""
Samples a large number of images from a pre-trained DiT model using DDP.
Subsequently saves a .npz file that can be used to compute FID and other
evaluation metrics via the ADM repo: https://github.com/openai/guided-diffusion/tree/main/evaluations

For a simple single-GPU/CPU sampling script, see sample.py.
"""
import torch
import torch.distributed as dist
from models import DiT_models
from download import find_model
from diffusion import create_diffusion
from diffusers.models import AutoencoderKL
from tqdm import tqdm
import os
from PIL import Image
import numpy as np
import math
import argparse
import cv2
import matplotlib.pyplot as plt

def create_npz_from_sample_folder(sample_dir, num=50_000):
    """
    Builds a single .npz file from a folder of .png samples.
    """
    samples = []
    for i in tqdm(range(num), desc="Building .npz file from samples"):
        sample_pil = Image.open(f"{sample_dir}/{i:06d}.png")
        sample_np = np.asarray(sample_pil).astype(np.uint8)
        samples.append(sample_np)
    samples = np.stack(samples)
    assert samples.shape == (num, samples.shape[1], samples.shape[2], 3)
    npz_path = f"{sample_dir}.npz"
    np.savez(npz_path, arr_0=samples)
    print(f"Saved .npz file to {npz_path} [shape={samples.shape}].")
    return npz_path

def filter(depth):
    depth = cv2.resize(depth, (32,32), interpolation=cv2.INTER_NEAREST)
    depth = np.clip(depth/10000,0,1)
    return depth

def filter2(depth):
    depth = np.clip(depth,1000,5000)/5000
    depth = np.array(depth*256,dtype=np.uint8)
    depth = cv2.medianBlur(depth, 15)
    depth = cv2.resize(depth,(32,32),interpolation=cv2.INTER_NEAREST)/256
    return depth


def main(args):
    """
    Run sampling.
    """
    torch.backends.cuda.matmul.allow_tf32 = args.tf32  # True: fast but may lead to some small numerical differences
    assert torch.cuda.is_available(), "Sampling with DDP requires at least one GPU. sample.py supports CPU-only usage"
    torch.set_grad_enabled(False)

    # Setup DDP:
    dist.init_process_group("nccl")
    rank = dist.get_rank()
    device = rank % torch.cuda.device_count()
    seed = args.global_seed * dist.get_world_size() + rank
    torch.manual_seed(seed)
    torch.cuda.set_device(device)
    print(f"Starting rank={rank}, seed={seed}, world_size={dist.get_world_size()}.")

    if args.ckpt is None:
        assert args.model == "DiT-XL/2", "Only DiT-XL/2 models are available for auto-download."
        assert args.image_size in [256, 512]
        assert args.num_classes == 1000

    # Load model:
    latent_size = args.image_size // 8
    model = DiT_models[args.model](
        input_size=latent_size,
        num_classes=args.num_classes,
        args=args,
    ).to(device)
    # Auto-download a pre-trained model or load a custom DiT checkpoint from train.py:
    ckpt_path = args.ckpt or f"DiT-XL-2-{args.image_size}x{args.image_size}.pt"
    state_dict = find_model(ckpt_path)
    model.load_state_dict(state_dict)
    model.eval()  # important!
    diffusion = create_diffusion(str(args.num_sampling_steps))
    vae = AutoencoderKL.from_pretrained(args.vae_path).to(device)
    # assert args.cfg_scale >= 1.0, "In almost all cases, cfg_scale be >= 1.0"
    using_cfg = args.cfg_scale > 1.0

    # Create folder to save samples:
    model_string_name = args.model.replace("/", "-")
    ckpt_string_name = os.path.basename(args.ckpt).replace(".pt", "") if args.ckpt else "pretrained"
    folder_name = f"{model_string_name}-{ckpt_string_name}-size-{args.image_size}-vae-{args.vae}-" \
                  f"cfg-{args.cfg_scale}-seed-{args.global_seed}"
    sample_folder_dir = f"{args.sample_dir}/{folder_name}"
    if rank == 0:
        os.makedirs(sample_folder_dir, exist_ok=True)
        print(f"Saving .png samples at {sample_folder_dir}")
    dist.barrier()

    # Figure out how many samples we need to generate on each GPU and how many iterations we need to run:
    n = args.per_proc_batch_size
    global_batch_size = n * dist.get_world_size()
    # To make things evenly-divisible, we'll sample a bit more than we need and then discard the extra samples:
    total_samples = int(math.ceil(args.num_fid_samples / global_batch_size) * global_batch_size)
    if rank == 0:
        print(f"Total number of images that will be sampled: {total_samples}")
    assert total_samples % dist.get_world_size() == 0, "total_samples must be divisible by world_size"
    samples_needed_this_gpu = int(total_samples // dist.get_world_size())
    assert samples_needed_this_gpu % n == 0, "samples_needed_this_gpu must be divisible by the per-GPU batch size"
    iterations = int(samples_needed_this_gpu // n)
    pbar = range(iterations)
    pbar = tqdm(pbar) if rank == 0 else pbar
    total = 0
    for _ in pbar:
        # Sample inputs:
        t = args.predict_horizon
        z = torch.randn(n, model.in_channels*t, latent_size, latent_size, device=device)
        # y = torch.randint(0, args.num_classes-100, (n,), device=device)
        
        # load conditional features or labeles
        # ids = np.random.randint(24000, 25000, n)
        # ids = np.random.randint(0, 700, n)
        # ids = np.random.randint(0, 8, n)*100+np.random.randint(90, 100, n)
        ids = np.random.randint(0, 48, n)*10+9

        ids = np.array(ids,dtype=np.int32)

        assert using_cfg is False, "Text conditioning is not supported with cfg_scale > 1.0"
        assert args.text_cond == True
        text_features = []
        for id in ids:
            text_feature = np.load(os.path.join(args.feature_path, f"episode{int(id):07}/text_clip.npy",))
            text_features.append(text_feature)
        text_features = np.concatenate(text_features, axis=0)
        text_features = torch.tensor(text_features, device=device)
        y = text_features
        if args.action_steps>0:
            # TODO
            action_true = []
            import json
            with open(os.path.join(args.feature_path, "dataset_info_traj.json"), "r") as f:
                episode_info_all = json.load(f)
            for id in ids:
                traj = episode_info_all[id][str(id)]
                action_id = []
                for i in range(args.action_steps+1):
                    try:
                        action = traj[int(i*args.skip_step)]["state"][0]
                    except:
                        action = action_id[-1]
                    action_id.append(action)
                action_true.append(np.array(action_id))


        assert args.dynamics == True
        x_cond,x_pred = [],[]
        for id in ids:
            start = 0
            cond_features = np.load(os.path.join(args.feature_path, f"episode{int(id):07}/color_{args.img_prefix}_{start:04}.npy",))
            x_cond.append(cond_features)
            for i in range(args.predict_horizon):
                next = start+args.skip_step*(i+1)
                try:
                    pred_features = np.load(os.path.join(args.feature_path, f"episode{int(id):07}/color_{args.img_prefix}_{next:04}.npy",))
                    x_pred.append(pred_features)
                except:
                    x_pred.append(x_pred[-1])
        x_cond = np.concatenate(x_cond, axis=0)
        x_cond = torch.tensor(x_cond, device=device)
        x_pred = np.concatenate(x_pred, axis=0)
        x_pred = torch.tensor(x_pred, device=device)

        if args.use_depth:
            cond_depth,pred_depth = [],[]
            for id in ids:
                start = 0
                d = np.load(os.path.join(args.feature_path, f"episode{int(id):07}/depth_{args.img_prefix}_{start:04}.npy",))
                d = filter(d) if not args.depth_filter else filter2(d)
                d = d[np.newaxis]
                cond_depth.append(d)
                for i in range(args.predict_horizon):
                    next = start+args.skip_step*(i+1)
                    try:
                        d = np.load(os.path.join(args.feature_path, f"episode{int(id):07}/depth_{args.img_prefix}_{next:04}.npy",))
                        d = filter(d) if not args.depth_filter else filter2(d)
                        d = d[np.newaxis]
                        pred_depth.append(d)
                    except:
                        pred_depth.append(pred_depth[-1])
            cond_depth = np.stack(cond_depth, axis=0)
            print("cond_depth",cond_depth.shape)
            cond_depth = torch.tensor(cond_depth, device=device).float()
            pred_depth = np.stack(pred_depth, axis=0)
            print("pred_depth",pred_depth.shape)
            pred_depth = torch.tensor(pred_depth, device=device).float()

        # Setup classifier-free guidance:
        if using_cfg:
            z = torch.cat([z, z], 0)
            y_null = torch.tensor([1000] * n, device=device)
            y = torch.cat([y, y_null], 0)
            model_kwargs = dict(y=y, cfg_scale=args.cfg_scale,x_cond=x_cond)
            sample_fn = model.forward_with_cfg
        else:
            # model_kwargs = dict(y=y)
            model_kwargs = dict(y=y,x_cond=x_cond)
            sample_fn = model.forward

        if args.action_steps > 0:  
            z_a = torch.randn(n, args.action_steps, args.action_dim, device=device)
            model_kwargs["noised_action"] = z_a
        if args.use_depth:
            z_d = torch.randn(n, args.predict_horizon, args.d_hidden_size, args.d_hidden_size, device=device)
            model_kwargs["noised_depth"] = z_d
            model_kwargs["depth_cond"] = cond_depth
        
        ############################ Sample images:############################
        samples = diffusion.p_sample_loop(
            sample_fn, z.shape, z, clip_denoised=False, model_kwargs=model_kwargs, progress=False, device=device
        )
        if args.action_steps > 0 or args.use_depth:
            samples, samples_a, sample_depth = samples
        if using_cfg:
            samples, _ = samples.chunk(2, dim=0)  # Remove null class samples
        B, C, H, W = samples.shape
        t = args.predict_horizon
        samples = samples.view(B*t,int(C/t), H, W)
        samples = vae.decode(samples / 0.18215).sample
        samples = torch.clamp(127.5 * samples + 128.0, 0, 255).permute(0, 2, 3, 1).to("cpu", dtype=torch.uint8).numpy()

        reconstruct = vae.decode(x_cond / 0.18215).sample
        reconstruct = torch.clamp(127.5 * reconstruct + 128.0, 0, 255).permute(0, 2, 3, 1).to("cpu", dtype=torch.uint8).numpy()
        rec_pred = vae.decode(x_pred / 0.18215).sample
        rec_pred = torch.clamp(127.5 * rec_pred + 128.0, 0, 255).permute(0, 2, 3, 1).to("cpu", dtype=torch.uint8).numpy()

        # Save samples to disk as individual .png files
        B,H,W,C = samples.shape
        t = args.predict_horizon
        batch_size = int(B/t)
        for i in range(batch_size):
            sample = samples[i*t:(i+1)*t]
            future = rec_pred[i*t:(i+1)*t]
            index = i * dist.get_world_size() + rank + total
            if args.dynamics:
                now = reconstruct[i]
                img_up, img_bottem = now, now
                if args.action_steps>0:
                    print("idx",ids[i])
                    print("predict",samples_a[i][:,:4])
                    print("true_action",(action_true[i][1:]-action_true[i][:1])[:,:4]*10) #scale here
                for j in range(t):
                    img_up = np.concatenate((img_up,future[j]),axis=1)
                    img_bottem = np.concatenate((img_bottem,sample[j]),axis=1)
            label = ids[i]
            img = np.concatenate((img_up,img_bottem),axis=0)
            Image.fromarray(img).save(f"{sample_folder_dir}/{index:06d}_traj{ids[i]:06d}_color.png")
        if args.use_depth:
            B,_,_,_ = sample_depth.shape
            t = args.predict_horizon
            # batch_size = int(B/t)
            pred_depth = pred_depth.detach().cpu().numpy()
            cond_depth = cond_depth.detach().cpu().numpy()
            for i in range(batch_size):
                sample = sample_depth[i].detach().cpu().numpy()
                now = cond_depth[i,0]
                future = pred_depth[i*t:(i+1)*t, 0]
                index = i * dist.get_world_size() + rank + total
                img_up, img_bottem = now, now
                # print(img_up.shape,img_bottem.shape,future.shape,sample.shape)
                for j in range(t):
                    img_up = np.concatenate((img_up,future[j]),axis=1)
                    img_bottem = np.concatenate((img_bottem,sample[j]),axis=1)
                # print("index", index)
                img = np.concatenate((img_up,img_bottem),axis=0)
                plt.imsave(f"{sample_folder_dir}/{index:06d}_traj{ids[i]:06d}_depth.png", img, vmin=0, vmax=1) 
        total += global_batch_size

    # Make sure all processes have finished saving their samples before attempting to convert to .npz
    dist.barrier()
    if rank == 0:
        create_npz_from_sample_folder(sample_folder_dir, args.num_fid_samples)
        print("Done.")
    dist.barrier()
    dist.destroy_process_group()


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model", type=str, choices=list(DiT_models.keys()), default="DiT-XL/2")
    parser.add_argument("--vae",  type=str, choices=["ema", "mse"], default="ema")
    parser.add_argument("--sample-dir", type=str, default="samples")
    parser.add_argument("--per-proc-batch-size", type=int, default=32)
    parser.add_argument("--num-fid-samples", type=int, default=50_000)
    parser.add_argument("--image-size", type=int, choices=[256, 512], default=256)
    parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--cfg-scale",  type=float, default=1.5)
    parser.add_argument("--num-sampling-steps", type=int, default=250)
    parser.add_argument("--global-seed", type=int, default=0)
    parser.add_argument("--tf32", action='store_false', default=True,
                        help="By default, use TF32 matmuls. This massively accelerates sampling on Ampere GPUs.")
    parser.add_argument("--ckpt", type=str, default=None,
                        help="Optional path to a DiT checkpoint (default: auto-download a pre-trained DiT-XL/2 model).")
    parser.add_argument("--feature-path", type=str, default="features")

    parser.add_argument("--dynamics", action="store_true")
    parser.add_argument("--skip_step", type=int, default=10)
    parser.add_argument("--text_cond", action="store_true")
    parser.add_argument("--clip_path", type=str, default="/cephfs/shared/llm/clip-vit-base-patch32")
    parser.add_argument("--vae_path", type=str, default="/cephfs/shared/llm/sd-vae-ft-mse")
    parser.add_argument("--text_emb_size", type=int, default=512)

    parser.add_argument("--action_steps", type=int, default=0)
    parser.add_argument("--action_dim", type=int, default=7)
    parser.add_argument("--attn_mask", action="store_true")
    parser.add_argument("--learnable_action_pos", action="store_true")
    # predict_horizon
    parser.add_argument("--predict_horizon", type=int, default=1)
    parser.add_argument("--ckpt_wrapper", action="store_true")
    parser.add_argument("--img_prefix", type=str, default="wrist_1")
    # use_depth
    parser.add_argument("--use_depth", action="store_true")
    # d_hidden_size
    parser.add_argument("--d_hidden_size", type=int, default=32)
    parser.add_argument("--d_patch_size", type=int, default=8)
    parser.add_argument("--depth_filter", action="store_true")
    args = parser.parse_args()
    main(args)
