import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid

from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, process_images, get_model_name_from_path
from torch.utils.data import Dataset, DataLoader
from llava.model.llava_arch import LlavaMetaForCausalLM

from transformers import AutoModelForCausalLM, AutoTokenizer
from transformers.generation import GenerationConfig

from PIL import Image
import math


# import debugpy
# debugpy.listen(10010)
# print('wait debugger')
# debugpy.wait_for_client()
# print("Debugger Attached")



def split_list(lst, n):
    """Split a list into n (roughly) equal-sized chunks"""
    chunk_size = math.ceil(len(lst) / n)  # integer division
    return [lst[i:i+chunk_size] for i in range(0, len(lst), chunk_size)]


def get_chunk(lst, n, k):
    chunks = split_list(lst, n)
    return chunks[k]


# Custom dataset class
class CustomDataset(Dataset):
    def __init__(self, questions, image_folder, tokenizer, model_config):
        self.questions = questions
        self.image_folder = image_folder
        self.tokenizer = tokenizer
        # self.image_processor = image_processor
        self.model_config = model_config

    def __getitem__(self, index):
        line = self.questions[index]
        image_file = line["image"]
        qs = line["text"]
        # if self.model_config.mm_use_im_start_end:
        #     qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
        # else:
        #     qs = DEFAULT_IMAGE_TOKEN + '\n' + qs

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        image = Image.open(os.path.join(self.image_folder, image_file)).convert('RGB')
        image_tensor = image

        input_ids = tokenizer_image_token(prompt, self.tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt')

        return input_ids, image_tensor, image.size, image

    def __len__(self):
        return len(self.questions)


def collate_fn(batch):
    input_ids, image_tensors, image_sizes, images = zip(*batch)
    input_ids = torch.stack(input_ids, dim=0)
    image_tensors = None
    return input_ids, image_tensors, image_sizes, images


# DataLoader
def create_data_loader(questions, image_folder, tokenizer, model_config, batch_size=1, num_workers=4):
    assert batch_size == 1, "batch_size must be 1"
    dataset = CustomDataset(questions, image_folder, tokenizer, model_config)
    data_loader = DataLoader(dataset, batch_size=batch_size, num_workers=num_workers, shuffle=False, collate_fn=collate_fn)
    return data_loader


def eval_model(args):
    device= args.cuda_device
    if args.vision_retracing != "default" and args.vision_retracing != "append" and args.vision_retracing != "add" and args.vision_retracing != "adapt":
        print("Invalid vision retracing mode. Please choose from default, append, add, adapt.")
        return
    # Model
    disable_torch_init()
    model_path = os.path.expanduser(args.model_path)
    model_name = get_model_name_from_path(model_path)

    checkpoint = os.path.join(os.path.dirname(__file__), '..', '..', args.model_path)
    tokenizer = AutoTokenizer.from_pretrained(checkpoint, trust_remote_code=True)
    model = AutoModelForCausalLM.from_pretrained(
        checkpoint, device_map=device, trust_remote_code=True).eval()


    model.generation_config = GenerationConfig.from_pretrained(checkpoint, trust_remote_code=True)
    model.generation_config.top_p = 0.01


    model.transformer.h[0].mlp.starting_layer = args.starting_layer
    model.transformer.h[0].mlp.ending_layer = args.ending_layer
    model.transformer.h[0].mlp.vision_retracing_method = args.vision_retracing
    model.transformer.h[0].mlp.entropy_threshold = args.entropy_threshold
    for layer in range(32):
        model.transformer.h[layer].mlp.retracing_ratio = args.retracing_ratio

    questions = [json.loads(q) for q in open(os.path.expanduser(args.question_file), "r")]
    questions = get_chunk(questions, args.num_chunks, args.chunk_idx)
    answers_file = os.path.expanduser(args.answers_file)
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    ans_file = open(answers_file, "w")

    if 'plain' in model_name and 'finetune' not in model_name.lower() and 'mmtag' not in args.conv_mode:
        args.conv_mode = args.conv_mode + '_mmtag'
        print(f'It seems that this is a plain model, but it is not using a mmtag prompt, auto switching to {args.conv_mode}.')

    data_loader = create_data_loader(questions, args.image_folder, tokenizer, model.config)
    


    for (input_ids, image_tensor, image_sizes, images), line in tqdm(zip(data_loader, questions), total=len(questions)):
        idx = line["question_id"]
        image_idx = line["image"]
        cur_prompt = line["text"]
        input_ids = input_ids.to(device=device, non_blocking=True)

        images = args.image_folder + "/" + image_idx

        with torch.inference_mode():

            query = f'<img>{images}</img>\n{cur_prompt}'
            outputs, _ = model.chat(tokenizer, query=query, history=None, temperature=args.temperature, do_sample=False, max_new_tokens=args.max_new_tokens)
    
            
        
            # outputs = tokenizer.batch_decode(output_ids, skip_special_tokens=True)[0].strip()
            print("\n Outputs: ", outputs)

        ans_id = shortuuid.uuid()
        ans_file.write(json.dumps({"question_id": idx,
                                   "prompt": cur_prompt,
                                   "text": outputs,
                                   "answer_id": ans_id,
                                   "model_id": model_name,
                                   "metadata": {}}) + "\n")
        # ans_file.flush()
    ans_file.close()

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model-path", type=str, default="llava-v1.5-13b")
    parser.add_argument("--model-base", type=str, default=None)
    parser.add_argument("--image-folder", type=str, default="./playground/data/eval/pope/val2014")
    parser.add_argument("--question-file", type=str, default="./playground/data/eval/pope/llava_pope_test.jsonl")
    parser.add_argument("--answers-file", type=str, default="./playground/data/eval/pope/answers/llava-v1.5-13b.jsonl")
    parser.add_argument("--conv-mode", type=str, default="vicuna_v1")
    parser.add_argument("--num-chunks", type=int, default=1)
    parser.add_argument("--chunk-idx", type=int, default=0)
    parser.add_argument("--temperature", type=float, default=0)
    parser.add_argument("--top_p", type=float, default=None)
    parser.add_argument("--num_beams", type=int, default=1)
    parser.add_argument("--max-new-tokens", type=int, default=1)
    parser.add_argument("--target-layers", type=int, default=0)
    parser.add_argument("--cuda-device", type=str, default="cuda:0")
    parser.add_argument("--vision-retracing", type=str, default="default")
    parser.add_argument("--retracing-ratio", type=float, default=0.0)
    parser.add_argument("--entropy-threshold", type=float, default=0.75)
    parser.add_argument("--starting-layer", type=int, default=4)
    parser.add_argument("--ending-layer", type=int, default=21)
    args = parser.parse_args()

    eval_model(args)
