import numpy as np
from sklearn.cluster import KMeans
from sklearn.preprocessing import normalize


def spectral_clustering(A, K):
    gamma, v = np.linalg.eig(A)
    gamma = gamma.real
    v = v.real
    gamma = abs(gamma)
    index = gamma.argsort()
    index = list(index)
    index.reverse()
    index = index[:K]
    v = v[:, index]
    v = normalize(v)
    kmeans = KMeans(n_clusters=K).fit(v)
    return kmeans.labels_


def mean_adj(parameter):
    """
    This is the spectral clustering algorithm on the mean adjacency matrix of the multi-layer network.
    :param A: adjacency tensor
    :param K: number of communities
    :return: community assignment
    """
    A = parameter[0]
    K = parameter[1]
    tensor_type = A.shape
    n, M = tensor_type[0], tensor_type[2]
    A = A.sum(axis=2)/M
    return spectral_clustering(A, K)


def mean_adj_link_prediction(parameter):
    A = parameter[0]
    K = parameter[1]
    frac = parameter[2]
    tensor_type = A.shape
    n, M = tensor_type[0], tensor_type[2]

    B = np.random.binomial(1, frac, (n, n, M))
    for m in range(M):
        B[:, :, m] = np.triu(B[:, :, m]) + np.triu(B[:, :, m], 1).T
    B0 = np.zeros(B.shape)
    for m in range(M):
        B0[:, :, m] = np.triu(B[:, :, m])        # indicator of independent random variables in A.

    A_mean = (A * B).sum(axis=2)/M
    gamma, v = np.linalg.eig(A_mean)
    gamma = gamma.real
    v = v.real
    gamma_abs = abs(gamma)
    index = gamma_abs.argsort()
    index = list(index)
    index.reverse()
    index = index[:K]
    v = v[:, index]
    gamma = gamma[index]
    P_hat = v @ np.diag(gamma) @ v.T
    P_hat[P_hat < 0] = 0
    P_hat[P_hat > 1] = 1

    A_hat = np.zeros(A.shape)
    for m in range(M):
        A_hat[:, :, m] = np.random.binomial(1, P_hat)
    B0_tilde = 1 - B0
    for m in range(M):
        B0_tilde[:, :, m] = np.triu(B0_tilde[:, :, m])
    return (np.abs((A_hat - A) * B0_tilde)).sum() / B0_tilde.sum()

