import itertools
import numpy as np
import networkx as nx


def sort_ranking(score_matrix, expr, inter, lmbda):
    flat_array = score_matrix.flatten()
    G = nx.DiGraph()

    # Argsort on the flattened array
    sorted_flat_indices = np.argsort(-flat_array)

    # Mapping flat indices back to (i, j) format
    rows, cols = score_matrix.shape
    G.add_nodes_from(range(cols))
    sorted_indices_ij = np.unravel_index(sorted_flat_indices, (rows, cols))

    for k in range(len(sorted_indices_ij[0])):
        i, j = sorted_indices_ij[0][k], sorted_indices_ij[1][k]
        if i != j:
            score = score_matrix[i, j]
            if score > lmbda: 
                G.add_edge(i, j)
                if not nx.is_directed_acyclic_graph(G):
                    G.remove_edge(i, j)
            else:
                break
    W = np.zeros((cols, cols))
    for i, nbrdict in G.adjacency():
        for j in nbrdict.keys():
            W[i, j] = 1
    
    return W, G

def move_variable(perm, from_index, to_index):
    """Move a variable from from_index to to_index in the permutation."""
    if from_index == to_index:  # No move needed
        return perm
    new_perm = perm.copy()
    new_perm.insert(to_index, new_perm.pop(from_index))
    return new_perm

def get_neighbors(perm):
    """Generate all neighbors by moving each variable to its left or right, if possible."""
    neighbors = []
    for i in range(len(perm)):
        if i > 0:  # Can move to the left
            neighbors.append(move_variable(perm, i, i-1))
        if i < len(perm) - 1:  # Can move to the right
            neighbors.append(move_variable(perm, i, i+1))
    return neighbors

def local_search(initial_perm, score_function):
    """Perform local search to optimize the permutation."""
    current_perm = initial_perm
    current_score = score_function(current_perm)
    while True:
        neighbors = get_neighbors(current_perm)
        next_perm = None
        for neighbor in neighbors:
            neighbor_score = score_function(neighbor)
            if neighbor_score > current_score:  # Assuming we want to maximize the score
                next_perm = neighbor
                current_score = neighbor_score
                break  # Exit early if a better neighbor is found
        if next_perm is None:
            break  # No improvement found
        current_perm = next_perm
    return current_perm

def brute_force(score_function, d):
    permutations = itertools.permutations(range(d))

    # Print each permutation
    max_score = 0
    candidate = None
    for perm in permutations:
        score = score_function(perm)
        if score > max_score:
            max_score = score
            candidate = perm
    return candidate

def generate_all_possible_moves(perm):
    """Generate all possible moves of a variable to any position."""
    moves = []
    for i in range(len(perm)):
        for j in range(len(perm)):
            if i != j:
                # Generate a move by placing i-th element to j-th position
                moved_perm = move_variable(perm, i, j)
                moves.append(moved_perm)
    return moves

def local_search_extended(initial_perm, score_function):
    """Perform local search with an extended neighborhood definition."""
    current_perm = initial_perm
    current_score = score_function(current_perm)
    while True:
        all_moves = generate_all_possible_moves(current_perm)
        next_perm = None
        for move in all_moves:
            move_score = score_function(move)
            if move_score > current_score:  # Assuming we want to maximize the score
                next_perm = move
                current_score = move_score
                break  # Exit early if a better move is found
        if next_perm is None:
            break  # No improvement found
        current_perm = next_perm
    return current_perm

