import torch 
import torch.nn as nn 
from .blocks import DDSConv, Nystromer
from .subblocks import Mish, LayerNorm 
from .spline_transforms import piecewise_rational_quadratic_transform
import math 

class Log(nn.Module):
  def forward(self, x, x_mask, reverse=False, **kwargs):
    if not reverse:
      y = torch.log(torch.clamp_min(x, 1e-5)) * x_mask
      logdet = torch.sum(-y, [1, 2])
      return y, logdet
    else:
      x = torch.exp(x) * x_mask
      return x

class Flip(nn.Module):
  def forward(self, x, *args, reverse=False, **kwargs):
    x = torch.flip(x, [1])
    if not reverse:
      logdet = torch.zeros(x.size(0)).to(dtype=x.dtype, device=x.device)
      return x, logdet
    else:
      return x

class ElementwiseAffine(nn.Module):
  def __init__(self, channels):
    super().__init__()
    self.channels = channels 
    self.m = nn.Parameter(torch.zeros(channels, 1))
    self.logs = nn.Parameter(torch.zeros(channels, 1))

  def forward(self, x, x_mask, reverse=False, **kwargs):
    if not reverse:
      y = self.m + torch.exp(self.logs) * x 
      y = y * x_mask 
      logdet = torch.sum(self.logs * x_mask, [1,2])
      return y, logdet 
    else:
      x = (x-self.m) * torch.exp(-self.logs) * x_mask 
      return x 

class ConvFlow(nn.Module):
  def __init__(self, in_channels, filter_channels, kernel_size, n_layers, num_bins=10, tail_bound=5.0, attention=True):
    super().__init__()
    self.in_channels = in_channels
    self.filter_channels = filter_channels
    self.kernel_size = kernel_size
    self.n_layers = n_layers
    self.num_bins = num_bins
    self.tail_bound = tail_bound
    self.half_channels = in_channels // 2
    self.is_even = in_channels%2 == 0   
    if self.is_even:
      self.pre = nn.Conv1d(self.half_channels, filter_channels, 1) 
    else:
      self.pre = nn.Conv1d(self.half_channels+1, filter_channels, 1)
    self.convs = DDSConv(filter_channels, kernel_size, n_layers, p_dropout=0.)
    self.attention = attention
    if attention:
      self.nystrom = Nystromer(filter_channels=filter_channels, heads=2, depth=2, 
        num_landmarks=96, pinv_iterations=6, attn_dropout=0.1, ff_dropout=0.1)
      self.norm = LayerNorm(filter_channels*2)
      self.mish = Mish()
      self.proj = nn.Conv1d(filter_channels*2, self.half_channels*(num_bins * 3 - 1),1)
    else:
      self.proj = nn.Conv1d(filter_channels, self.half_channels*(num_bins * 3 - 1), 1)
    self.proj.weight.data.zero_()
    self.proj.bias.data.zero_()

  def forward(self, x, x_mask, g=None, reverse=False):
    if self.is_even:
      x0, x1 = torch.split(x, [self.half_channels]*2, 1)
    else:
      x0 , x1 = torch.split(x, [self.half_channels+1, self.half_channels],1)
    h = self.pre(x0)
    if self.attention:
      s = h + g
      s = self.nystrom(s, x_mask)
      h = self.convs(h, x_mask, g=g)
      h = torch.cat([s,h], dim=1)
      h = self.norm(h)
      h = self.mish(h)
    else:
      h = self.convs(h, x_mask, g=g)

    h = self.proj(h) * x_mask

    b, c, t = x1.shape
    h = h.reshape(b, c, -1, t).permute(0, 1, 3, 2) # [b, cx?, t] -> [b, c, t, ?]

    unnormalized_widths = h[..., :self.num_bins] / math.sqrt(self.filter_channels)
    unnormalized_heights = h[..., self.num_bins:2*self.num_bins] / math.sqrt(self.filter_channels)
    unnormalized_derivatives = h[..., 2 * self.num_bins:]

    x1, logabsdet = piecewise_rational_quadratic_transform(x1,
        unnormalized_widths,
        unnormalized_heights,
        unnormalized_derivatives,
        inverse=reverse,
        tails='linear',
        tail_bound=self.tail_bound
    )

    x = torch.cat([x0, x1], 1) * x_mask
    logdet = torch.sum(logabsdet * x_mask, [1,2])
    if not reverse:
        return x, logdet
    else:
        return x
