"""
Configuration management for HKAN experiments
"""
import os
import argparse

class Config:
    def __init__(self):
        # Data paths
        self.data_path = "heart_disease.csv"
        self.results_dir = "results"

        # Random seed
        self.seed = 42

        # Data split ratios (train:val:test = 8:1:1)
        self.test_size = 0.1
        self.val_size_from_remaining = 1/9  # 10% of total from 90% remaining

        # Training parameters
        self.epochs = 200
        self.learning_rate = 0.01
        self.weight_decay = 1e-5
        self.patience = 30
        self.grad_clip = 1.0

        # KAN parameters
        self.kan_grid = 5
        self.kan_k = 3
        self.kan_hidden_multiplier = 1
        self.fusion_hidden_layers = [2]

        # KAN regularization
        self.kan_reg_lamb_l1 = 1e-3
        self.kan_reg_lamb_entropy = 1e-3
        self.kan_reg_lamb_coef = 0.0
        self.kan_reg_lamb_coefdiff = 0.0

        # Factor regularization
        self.lambda_decorrelation = 0.01
        self.lambda_sparsity = 0.001
        self.lambda_stability = 0.01

        # Factor Quality Score weights
        self.fqs_independence_weight = 0.4
        self.fqs_stability_weight = 0.3
        self.fqs_sparsity_weight = 0.3

        # Unified fitness weights
        self.performance_weight = 0.9
        self.quality_weight = 0.1

        # GA parameters
        self.ga_pop_size = 30
        self.ga_max_generations = 20
        self.ga_min_groups = 3
        self.ga_max_groups = 6
        self.ga_min_features_per_group = 1
        self.ga_max_features_per_group = 5
        self.ga_crossover_rate = 0.7
        self.ga_mutation_rate = 0.8

def get_config():
    """Get configuration with command line argument override support"""
    parser = argparse.ArgumentParser(description='HKAN Experiments Configuration')
    parser.add_argument('--data-path', type=str, default='heart_disease.csv',
                       help='Path to the heart disease dataset')
    parser.add_argument('--results-dir', type=str, default='results',
                       help='Directory to save results')
    parser.add_argument('--seed', type=int, default=42,
                       help='Random seed for reproducibility')

    args = parser.parse_args()

    config = Config()
    config.data_path = args.data_path
    config.results_dir = args.results_dir
    config.seed = args.seed

    # Create results directory if it doesn't exist
    os.makedirs(config.results_dir, exist_ok=True)

    return config